<?php

namespace App\Http\Controllers;

use App\Models\Sticker;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class StickerController extends Controller
{
    /**
     * جلب الملصقات المعتمدة للمكتبة (للمحرر 3D)
     */
    public function approved(Request $request): JsonResponse
    {
        $query = Sticker::query()
            ->approved()
            ->active()
            ->with('media')
            ->orderByDesc('usage_count');

        if ($request->filled('category')) {
            $query->where('category', $request->category);
        }

        if ($request->filled('search')) {
            $query->where('name', 'like', '%'.$request->search.'%');
        }

        $stickers = $query->get()->map(fn (Sticker $sticker) => [
            'id' => $sticker->id,
            'name' => $sticker->name,
            'category' => $sticker->category,
            'image_url' => $sticker->image_url,
            'usage_count' => $sticker->usage_count,
            'designer' => $sticker->client?->name,
        ]);

        return response()->json([
            'success' => true,
            'stickers' => $stickers,
        ]);
    }

    /**
     * رفع ملصق جديد (للمصمم فقط)
     */
    public function store(Request $request): JsonResponse
    {
        $client = Auth::guard('client')->user();

        if (! $client || ! $client->isDesigner()) {
            return response()->json([
                'success' => false,
                'message' => 'يجب أن تكون مصمماً لرفع الملصقات',
            ], 403);
        }

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'description' => ['nullable', 'string', 'max:500'],
            'category' => ['nullable', 'string', 'max:50'],
            'image' => ['required', 'image', 'mimes:png,jpg,jpeg,webp,svg', 'max:5120'],
        ], [
            'name.required' => 'اسم الملصق مطلوب',
            'image.required' => 'صورة الملصق مطلوبة',
            'image.image' => 'يجب أن يكون الملف صورة',
            'image.max' => 'الحد الأقصى لحجم الصورة 5MB',
        ]);

        $sticker = Sticker::create([
            'client_id' => $client->id,
            'name' => $validated['name'],
            'description' => $validated['description'] ?? null,
            'category' => $validated['category'] ?? null,
            'status' => Sticker::STATUS_PENDING,
        ]);

        $sticker->addMediaFromRequest('image')
            ->toMediaCollection(Sticker::MEDIA_COLLECTION_IMAGE);

        return response()->json([
            'success' => true,
            'sticker' => [
                'id' => $sticker->id,
                'name' => $sticker->name,
                'status' => $sticker->status,
                'image_url' => $sticker->image_url,
            ],
            'message' => 'تم رفع الملصق بنجاح وسيتم مراجعته من قبل الإدارة',
        ]);
    }

    /**
     * حذف ملصق (للمصمم صاحب الملصق فقط)
     */
    public function destroy(Sticker $sticker): JsonResponse
    {
        $client = Auth::guard('client')->user();

        if (! $client || $sticker->client_id !== $client->id) {
            return response()->json([
                'success' => false,
                'message' => 'غير مصرح لك بحذف هذا الملصق',
            ], 403);
        }

        $sticker->delete();

        return response()->json([
            'success' => true,
            'message' => 'تم حذف الملصق بنجاح',
        ]);
    }

    /**
     * ملصقات المصمم (لوحة تحكم المصمم)
     */
    public function myStickers(): JsonResponse
    {
        $client = Auth::guard('client')->user();

        if (! $client || ! $client->isDesigner()) {
            return response()->json([
                'success' => false,
                'message' => 'يجب أن تكون مصمماً',
            ], 403);
        }

        $stickers = $client->stickers()
            ->with('media')
            ->latest()
            ->get()
            ->map(fn (Sticker $sticker) => [
                'id' => $sticker->id,
                'name' => $sticker->name,
                'category' => $sticker->category,
                'status' => $sticker->status,
                'status_label' => match ($sticker->status) {
                    'pending' => 'قيد المراجعة',
                    'approved' => 'معتمد',
                    'rejected' => 'مرفوض',
                    default => $sticker->status,
                },
                'rejection_reason' => $sticker->rejection_reason,
                'image_url' => $sticker->image_url,
                'usage_count' => $sticker->usage_count,
            ]);

        return response()->json([
            'success' => true,
            'stickers' => $stickers,
        ]);
    }

    /**
     * تسجيل استخدام ملصق
     */
    public function trackUsage(Sticker $sticker): JsonResponse
    {
        if ($sticker->isApproved()) {
            $sticker->incrementUsage();
        }

        return response()->json(['success' => true]);
    }
}
