<?php

namespace App\Http\Controllers;

use App\Models\Brand;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class ShopController extends Controller
{
    /**
     * Display the shop/categories page.
     */
    public function index(Request $request): View
    {
        $categories = Category::query()
            ->withCount(['products' => fn ($q) => $q->active()])
            ->orderBy('name')
            ->get();

        $brands = Brand::query()
            ->withCount(['products' => fn ($q) => $q->active()])
            ->orderBy('name')
            ->get();

        return view('store.categories', compact('categories', 'brands'));
    }

    /**
     * Fetch products as JSON for AJAX/JavaScript.
     */
    public function products(Request $request): JsonResponse
    {
        $query = Product::query()
            ->active()
            ->with(['category', 'brand', 'defaultVariant', 'variants.attributeValues.attribute'])
            ->withCount('variants');

        // Filter by category
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        // Filter by brand
        if ($request->filled('brand')) {
            $query->where('brand_id', $request->brand);
        }

        // Filter by search term
        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // Sorting
        $sort = $request->get('sort', 'newest');
        switch ($sort) {
            case 'price_asc':
                $query->orderByRaw('(SELECT MIN(retail_price) FROM product_variants WHERE product_id = products.id) ASC');
                break;
            case 'price_desc':
                $query->orderByRaw('(SELECT MIN(retail_price) FROM product_variants WHERE product_id = products.id) DESC');
                break;
            case 'name_asc':
                $query->orderBy('name', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;
            case 'newest':
            default:
                $query->latest('id');
                break;
        }

        $perPage = min((int) $request->get('per_page', 24), 100);
        $products = $query->paginate($perPage);

        // Transform products to JSON format
        $transformedProducts = $products->getCollection()->map(function ($product) {
            // Get default variant or first variant
            $baseVariant = $product->defaultVariant ?? $product->variants->first();

            // Get price based on client type
            $client = auth('client')->user();
            $priceType = ($client && $client->client_type === 'company') ? 'wholesale' : 'retail';
            $retailPrice = $baseVariant?->getPriceForType($priceType) ?? 0;
            $originalPrice = null; // يمكن إضافة منطق للسعر الأصلي لاحقاً

            // Get variant options for variable products
            $options = [];
            if ($product->is_variable && $product->variants->count() > 0) {
                $attributeGroups = [];
                foreach ($product->variants as $variant) {
                    foreach ($variant->attributeValues as $attrValue) {
                        $attrName = $attrValue->attribute->name ?? 'خيار';
                        if (! isset($attributeGroups[$attrName])) {
                            $attributeGroups[$attrName] = [];
                        }
                        $value = $attrValue->display_value ?? $attrValue->value;
                        if (! in_array($value, $attributeGroups[$attrName])) {
                            $attributeGroups[$attrName][] = $value;
                        }
                    }
                }
                $options = $attributeGroups;
            }

            return [
                'id' => $product->id,
                'variant_id' => $baseVariant?->id,
                'name' => $product->name,
                'slug' => $product->slug,
                'description' => $product->description,
                'image' => $product->thumbnail_thumb_url ?? $product->thumbnail_url,
                'images' => $product->gallery_urls,
                'price' => $retailPrice,
                'originalPrice' => $originalPrice,
                'category' => $product->category?->name,
                'category_id' => $product->category_id,
                'brand' => $product->brand?->name,
                'brand_id' => $product->brand_id,
                'is_variable' => $product->is_variable,
                'in_stock' => $product->isInStock(),
                'stock' => $product->total_stock,
                'options' => $options,
                'badge' => $originalPrice && $originalPrice > $retailPrice ? 'خصم' : null,
            ];
        });

        return response()->json([
            'success' => true,
            'products' => $transformedProducts,
            'pagination' => [
                'current_page' => $products->currentPage(),
                'last_page' => $products->lastPage(),
                'per_page' => $products->perPage(),
                'total' => $products->total(),
            ],
        ]);
    }

    /**
     * Fetch categories for filters.
     */
    public function categories(): JsonResponse
    {
        $categories = Category::query()
            ->withCount(['products' => fn ($q) => $q->active()])
            ->orderBy('name')
            ->get()
            ->map(fn ($cat) => [
                'id' => $cat->id,
                'name' => $cat->name,
                'slug' => $cat->slug ?? null,
                'count' => $cat->products_count,
            ]);

        return response()->json([
            'success' => true,
            'categories' => $categories,
        ]);
    }

    /**
     * Fetch brands for filters.
     */
    public function brands(): JsonResponse
    {
        $brands = Brand::query()
            ->withCount(['products' => fn ($q) => $q->active()])
            ->orderBy('name')
            ->get()
            ->map(fn ($brand) => [
                'id' => $brand->id,
                'name' => $brand->name,
                'count' => $brand->products_count,
            ]);

        return response()->json([
            'success' => true,
            'brands' => $brands,
        ]);
    }
}
