<?php

namespace App\Http\Controllers;

use App\Models\Product;
use Illuminate\Http\Request;

class ProductController extends Controller
{
    /**
     * Display a listing of the resource.
     */
    public function index()
    {
        //
    }

    /**
     * Show the form for creating a new resource.
     */
    public function create()
    {
        //
    }

    /**
     * Store a newly created resource in storage.
     */
    public function store(Request $request)
    {
        //
    }

    /**
     * Display the specified resource.
     */
    public function show(string $slug)
    {
        $product = Product::where('slug', $slug)
            ->with([
                'category',
                'brand',
                'variants' => fn ($q) => $q->where('is_active', true)->with(['attributeValues.attribute', 'stockBalances']),
                'media',
            ])
            ->firstOrFail();

        // Get related products from same category
        $relatedProducts = Product::where('category_id', $product->category_id)
            ->where('id', '!=', $product->id)
            ->where('is_active', true)
            ->limit(4)
            ->get();

        return view('store.product', compact('product', 'relatedProducts'));
    }

    /**
     * Fetch product data for AJAX requests.
     */
    public function fetch(string $slug)
    {
        $product = Product::where('slug', $slug)
            ->with(['category', 'variants', 'media'])
            ->firstOrFail();

        return response()->json([
            'success' => true,
            'product' => $product,
        ]);
    }

    /**
     * Quick view product data for modal.
     */
    public function quickView(int $id)
    {
        $product = Product::with([
            'category',
            'brand',
            'variants' => fn ($q) => $q->where('is_active', true)->with(['attributeValues.attribute']),
            'media',
        ])->findOrFail($id);

        // Get price for default variant
        $defaultVariant = $product->variants->firstWhere('is_default', true) ?? $product->variants->first();

        $price = 0;
        $oldPrice = 0;

        if ($defaultVariant) {
            $priceRow = \App\Models\ProductPrice::where('product_id', $product->id)
                ->where('product_variant_id', $defaultVariant->id)
                ->first();

            if ($priceRow) {
                $price = (int) ($priceRow->retail_price ?? 0);
            }

            if ($price <= 0 && $defaultVariant->cost) {
                $margin = $priceRow->retail_margin ?? 20;
                $price = (int) round($defaultVariant->cost * (1 + $margin / 100));
            }

            $oldPrice = $price > 0 ? (int) round($price * 1.15) : 0;
        }

        // Process variants with attributes
        $attributeNames = []; // Track attribute names
        $variants = $product->variants->map(function ($variant) use ($product, &$attributeNames) {
            $colorAttr = null;
            $sizeAttr = null;

            foreach ($variant->attributeValues as $av) {
                $attrName = mb_strtolower($av->attribute->name ?? '');
                if (in_array($attrName, ['color', 'لون', 'اللون'])) {
                    $colorAttr = $av;
                    $attributeNames['color'] = $av->attribute->display_name ?? $av->attribute->name ?? 'اللون';
                } elseif (in_array($attrName, ['size', 'مقاس', 'الحجم', 'حجم'])) {
                    $sizeAttr = $av;
                    $attributeNames['size'] = $av->attribute->display_name ?? $av->attribute->name ?? 'المقاس';
                }
            }

            // Get variant price
            $variantPrice = 0;
            $priceRow = \App\Models\ProductPrice::where('product_id', $product->id)
                ->where('product_variant_id', $variant->id)
                ->first();

            if ($priceRow) {
                $variantPrice = (int) ($priceRow->retail_price ?? 0);
            }

            if ($variantPrice <= 0 && $variant->cost) {
                $margin = $priceRow->retail_margin ?? 20;
                $variantPrice = (int) round($variant->cost * (1 + $margin / 100));
            }

            return [
                'id' => $variant->id,
                'name' => $variant->name,
                'sku' => $variant->sku,
                'is_default' => $variant->is_default,
                'price' => $variantPrice,
                'old_price' => $variantPrice > 0 ? (int) round($variantPrice * 1.15) : 0,
                'cost' => $variant->cost,
                'color_name' => $colorAttr?->display_value ?? $colorAttr?->value,
                'color_hex' => $colorAttr?->color_code ?? '#808080',
                'size_name' => $sizeAttr?->display_value ?? $sizeAttr?->value,
            ];
        });

        // Collect images
        $images = [];
        if ($product->thumbnail_url) {
            $images[] = $product->thumbnail_url;
        }
        if ($product->gallery_urls) {
            $images = array_merge($images, $product->gallery_urls);
        }

        return response()->json([
            'success' => true,
            'product' => [
                'id' => $product->id,
                'name' => $product->name,
                'slug' => $product->slug,
                'description' => $product->description,
                'category' => $product->category?->name,
                'brand' => $product->brand?->name,
                'price' => $price,
                'old_price' => $oldPrice,
                'variants' => $variants,
            ],
            'images' => array_values(array_filter($images)),
            'attributeNames' => $attributeNames,
        ]);
    }

    /**
     * Show the form for editing the specified resource.
     */
    public function edit(Product $product)
    {
        //
    }

    /**
     * Update the specified resource in storage.
     */
    public function update(Request $request, Product $product)
    {
        //
    }

    /**
     * Remove the specified resource from storage.
     */
    public function destroy(Product $product)
    {
        //
    }
}
