<?php

namespace App\Http\Controllers;

use App\Models\Brand;
use App\Models\Category;
use App\Models\Product;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\View\View;

class OffersController extends Controller
{
    /**
     * صفحة العروض - عرض المنتجات التي عليها خصومات
     */
    public function index(Request $request): View
    {
        // جلب المنتجات التي عليها عروض نشطة
        $productsQuery = Product::query()
            ->active()
            ->whereHas('prices', fn ($q) => $q->onSale())
            ->with(['category', 'brand', 'defaultVariant', 'variants.attributeValues.attribute', 'prices'])
            ->withCount('variants');

        // الفلترة حسب التصنيف
        if ($request->filled('category')) {
            $productsQuery->where('category_id', $request->category);
        }

        // الفلترة حسب الماركة
        if ($request->filled('brand')) {
            $productsQuery->where('brand_id', $request->brand);
        }

        // البحث
        if ($request->filled('search')) {
            $search = $request->search;
            $productsQuery->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // الترتيب
        $sort = $request->get('sort', 'newest');
        switch ($sort) {
            case 'price_asc':
                $productsQuery->orderByRaw('(SELECT MIN(retail_price) FROM product_variants WHERE product_id = products.id) ASC');
                break;
            case 'price_desc':
                $productsQuery->orderByRaw('(SELECT MIN(retail_price) FROM product_variants WHERE product_id = products.id) DESC');
                break;
            case 'discount_high':
                // ترتيب حسب نسبة الخصم (الأعلى أولاً)
                $productsQuery->leftJoin('product_prices', 'products.id', '=', 'product_prices.product_id')
                    ->where('product_prices.is_on_sale', true)
                    ->orderBy('product_prices.discount_percentage', 'desc')
                    ->select('products.*');
                break;
            case 'name_asc':
                $productsQuery->orderBy('name', 'asc');
                break;
            case 'name_desc':
                $productsQuery->orderBy('name', 'desc');
                break;
            case 'newest':
            default:
                $productsQuery->latest('id');
                break;
        }

        $perPage = min((int) $request->get('per_page', 24), 100);
        $products = $productsQuery->paginate($perPage);

        // جلب التصنيفات والماركات للفلاتر
        $categories = Category::query()
            ->withCount(['products' => fn ($q) => $q->active()->whereHas('prices', fn ($q2) => $q2->onSale())])
            ->having('products_count', '>', 0)
            ->orderBy('name')
            ->get();

        $brands = Brand::query()
            ->withCount(['products' => fn ($q) => $q->active()->whereHas('prices', fn ($q2) => $q2->onSale())])
            ->having('products_count', '>', 0)
            ->orderBy('name')
            ->get();

        return view('offers', compact('products', 'categories', 'brands'));
    }

    /**
     * API لجلب المنتجات التي عليها عروض بصيغة JSON
     */
    public function products(Request $request): JsonResponse
    {
        $query = Product::query()
            ->active()
            ->whereHas('prices', fn ($q) => $q->onSale())
            ->with(['category', 'brand', 'defaultVariant', 'variants.attributeValues.attribute', 'prices'])
            ->withCount('variants');

        // الفلاتر
        if ($request->filled('category')) {
            $query->where('category_id', $request->category);
        }

        if ($request->filled('brand')) {
            $query->where('brand_id', $request->brand);
        }

        if ($request->filled('search')) {
            $search = $request->search;
            $query->where(function ($q) use ($search) {
                $q->where('name', 'like', "%{$search}%")
                    ->orWhere('description', 'like', "%{$search}%");
            });
        }

        // الترتيب
        $sort = $request->get('sort', 'newest');
        switch ($sort) {
            case 'price_asc':
                $query->orderByRaw('(SELECT MIN(retail_price) FROM product_variants WHERE product_id = products.id) ASC');
                break;
            case 'price_desc':
                $query->orderByRaw('(SELECT MIN(retail_price) FROM product_variants WHERE product_id = products.id) DESC');
                break;
            case 'discount_high':
                $query->leftJoin('product_prices', 'products.id', '=', 'product_prices.product_id')
                    ->where('product_prices.is_on_sale', true)
                    ->orderBy('product_prices.discount_percentage', 'desc')
                    ->select('products.*');
                break;
            case 'name_asc':
                $query->orderBy('name', 'asc');
                break;
            case 'name_desc':
                $query->orderBy('name', 'desc');
                break;
            case 'newest':
            default:
                $query->latest('id');
                break;
        }

        $perPage = min((int) $request->get('per_page', 24), 100);
        $products = $query->paginate($perPage);

        // تحويل المنتجات إلى JSON
        $transformedProducts = $products->getCollection()->map(function ($product) {
            $baseVariant = $product->defaultVariant ?? $product->variants->first();
            $client = auth('client')->user();
            $priceType = ($client && $client->client_type === 'company') ? 'wholesale' : 'retail';

            // جلب معلومات السعر والعرض
            $priceModel = $product->prices()->onSale()->first();
            $retailPrice = $priceModel?->getFinalPrice($priceType) ?? $baseVariant?->getPriceForType($priceType) ?? 0;
            $originalPrice = $priceModel?->{"{$priceType}_price"} ?? null;
            $discountPercentage = $priceModel?->discount_percentage ?? 0;

            // الخيارات للمنتجات المتغيرة
            $options = [];
            if ($product->is_variable && $product->variants->count() > 0) {
                $attributeGroups = [];
                foreach ($product->variants as $variant) {
                    foreach ($variant->attributeValues as $attrValue) {
                        $attrName = $attrValue->attribute->name ?? 'خيار';
                        if (! isset($attributeGroups[$attrName])) {
                            $attributeGroups[$attrName] = [];
                        }
                        $value = $attrValue->display_value ?? $attrValue->value;
                        if (! in_array($value, $attributeGroups[$attrName])) {
                            $attributeGroups[$attrName][] = $value;
                        }
                    }
                }
                $options = $attributeGroups;
            }

            return [
                'id' => $product->id,
                'variant_id' => $baseVariant?->id,
                'name' => $product->name,
                'slug' => $product->slug,
                'description' => $product->description,
                'image' => $product->thumbnail_thumb_url ?? $product->thumbnail_url,
                'images' => $product->gallery_urls,
                'price' => $retailPrice,
                'originalPrice' => $originalPrice,
                'discountPercentage' => $discountPercentage,
                'category' => $product->category?->name,
                'category_id' => $product->category_id,
                'brand' => $product->brand?->name,
                'brand_id' => $product->brand_id,
                'is_variable' => $product->is_variable,
                'in_stock' => $product->isInStock(),
                'stock' => $product->total_stock,
                'options' => $options,
                'badge' => $discountPercentage > 0 ? "خصم {$discountPercentage}%" : null,
            ];
        });

        return response()->json([
            'success' => true,
            'products' => $transformedProducts,
            'pagination' => [
                'current_page' => $products->currentPage(),
                'last_page' => $products->lastPage(),
                'per_page' => $products->perPage(),
                'total' => $products->total(),
            ],
        ]);
    }
}
