<?php

namespace App\Http\Controllers;

use App\Models\ProductDesign;
use App\Models\ProductModel;
use Illuminate\Http\Request;

class DesignLibraryController extends Controller
{
    public function index(Request $request)
    {
        $modelsQuery = ProductModel::query()
            ->active()
            ->with(['product', 'publicDesigns' => function ($query) {
                $query->latest()->limit(3);
            }])
            ->withCount('designs');

        // Filters
        if ($request->filled('search')) {
            $modelsQuery->where('name', 'like', '%'.$request->search.'%');
        }

        if ($request->filled('product_id')) {
            $modelsQuery->where('product_id', $request->product_id);
        }

        // Sorting
        $sort = $request->get('sort', 'popular');
        match ($sort) {
            'newest' => $modelsQuery->latest(),
            'popular' => $modelsQuery->orderByDesc('designs_count'),
            'featured' => $modelsQuery->where('is_featured', true)->latest(),
            default => $modelsQuery->latest(),
        };

        $models = $modelsQuery->paginate(12);

        // Popular designs
        $popularDesigns = ProductDesign::query()
            ->public()
            ->popular()
            ->with(['productModel.product', 'user'])
            ->limit(8)
            ->get();

        // Featured designs
        $featuredDesigns = ProductDesign::query()
            ->public()
            ->featured()
            ->with(['productModel.product', 'user'])
            ->latest()
            ->limit(4)
            ->get();

        return view('design-library.index', [
            'models' => $models,
            'popularDesigns' => $popularDesigns,
            'featuredDesigns' => $featuredDesigns,
        ]);
    }

    public function designs(Request $request)
    {
        $designsQuery = ProductDesign::query()
            ->public()
            ->with(['productModel.product', 'user']);

        // Filters
        if ($request->filled('search')) {
            $designsQuery->where('name', 'like', '%'.$request->search.'%');
        }

        if ($request->filled('model_id')) {
            $designsQuery->where('product_model_id', $request->model_id);
        }

        if ($request->filled('user_id')) {
            $designsQuery->where('user_id', $request->user_id);
        }

        // Sorting
        $sort = $request->get('sort', 'popular');
        match ($sort) {
            'newest' => $designsQuery->latest(),
            'popular' => $designsQuery->popular(),
            'most_liked' => $designsQuery->mostLiked(),
            default => $designsQuery->latest(),
        };

        $designs = $designsQuery->paginate(24);

        return view('design-library.designs', [
            'designs' => $designs,
        ]);
    }

    public function show(ProductDesign $design)
    {
        $design->incrementViews();

        $design->load(['productModel.product', 'user', 'likedBy']);

        // Similar designs
        $similarDesigns = ProductDesign::query()
            ->public()
            ->where('product_model_id', $design->product_model_id)
            ->where('id', '!=', $design->id)
            ->latest()
            ->limit(6)
            ->get();

        return view('design-library.show', [
            'design' => $design,
            'similarDesigns' => $similarDesigns,
        ]);
    }
}
