<?php

namespace App\Http\Controllers;

use App\Models\ProductDesign;
use App\Models\ProductModel;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class ConfiguratorController extends Controller
{
    public function show(ProductModel $model)
    {
        $model->incrementViews();

        $model->load(['product', 'publicDesigns' => function ($query) {
            $query->latest()->limit(12);
        }]);

        // Get top 3 most popular designs for this model
        $popularDesigns = ProductDesign::query()
            ->where('product_model_id', $model->id)
            ->public()
            ->popular()
            ->with('client')
            ->limit(3)
            ->get();

        return view('configurator.show', [
            'model' => $model,
            'product' => $model->product,
            'popularDesigns' => $popularDesigns,
            'hideHeader' => true,
            'hideFooter' => true,
            'hideStoreComponents' => true,
        ]);
    }

    public function saveDesign(Request $request, ProductModel $model)
    {
        $validated = $request->validate([
            'name' => 'nullable|string|max:255',
            'configuration' => 'required|array',
            'preview_image' => 'required|string', // Base64 image
            'is_public' => 'boolean',
        ]);

        // Save preview image
        $previewPath = $this->saveBase64Image($validated['preview_image']);

        // Save with both user_id (admin) and client_id (client)
        $userId = Auth::guard('web')->id();
        $clientId = Auth::guard('client')->id();

        $design = ProductDesign::create([
            'product_model_id' => $model->id,
            'user_id' => $userId,
            'client_id' => $clientId,
            'name' => $validated['name'] ?? 'تصميمي',
            'configuration' => $validated['configuration'],
            'preview_image' => $previewPath,
            'is_public' => $validated['is_public'] ?? false,
        ]);

        $model->incrementDesigns();

        return response()->json([
            'success' => true,
            'design' => $design,
            'message' => 'تم حفظ التصميم بنجاح',
        ]);
    }

    public function loadDesign(ProductDesign $design)
    {
        $design->incrementViews();

        return response()->json([
            'success' => true,
            'design' => $design->load('productModel'),
        ]);
    }

    public function updateDesign(Request $request, ProductDesign $design)
    {
        $this->authorize('update', $design);

        $validated = $request->validate([
            'name' => 'nullable|string|max:255',
            'configuration' => 'required|array',
            'preview_image' => 'nullable|string',
            'is_public' => 'boolean',
        ]);

        if (isset($validated['preview_image'])) {
            // Delete old preview
            if ($design->preview_image) {
                Storage::disk('public')->delete($design->preview_image);
            }

            // Save new preview
            $validated['preview_image'] = $this->saveBase64Image($validated['preview_image']);
        }

        $design->update($validated);

        return response()->json([
            'success' => true,
            'design' => $design,
            'message' => 'تم تحديث التصميم بنجاح',
        ]);
    }

    public function deleteDesign(ProductDesign $design)
    {
        $this->authorize('delete', $design);

        if ($design->preview_image) {
            Storage::disk('public')->delete($design->preview_image);
        }

        $design->delete();

        return response()->json([
            'success' => true,
            'message' => 'تم حذف التصميم بنجاح',
        ]);
    }

    public function toggleLike(ProductDesign $design)
    {
        $user = Auth::user();

        if (! $user) {
            return response()->json([
                'success' => false,
                'message' => 'يجب تسجيل الدخول أولاً',
            ], 401);
        }

        $design->toggleLike($user);

        return response()->json([
            'success' => true,
            'is_liked' => $design->isLikedBy($user),
            'likes_count' => $design->likes_count,
        ]);
    }

    private function saveBase64Image(string $base64Image): string
    {
        // Remove data URI prefix (handles png, jpeg, webp, etc.)
        $image = preg_replace('/^data:image\/\w+;base64,/', '', $base64Image);
        $image = str_replace(' ', '+', $image);
        $imageData = base64_decode($image);

        // Detect format from header or default to png
        $extension = 'png';
        if (str_starts_with($base64Image, 'data:image/jpeg')) {
            $extension = 'jpg';
        } elseif (str_starts_with($base64Image, 'data:image/webp')) {
            $extension = 'webp';
        }

        $filename = 'designs/'.Str::uuid().'.'.$extension;
        Storage::disk('public')->put($filename, $imageData);

        return $filename;
    }
}
