<?php

namespace App\Http\Controllers;

use App\Models\Compare;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;

class CompareController extends Controller
{
    /**
     * Get identifier (client_id or session_id)
     */
    private function getIdentifier(): array
    {
        $clientId = Auth::guard('client')->id();

        return [
            'client_id' => $clientId,
            'session_id' => ! $clientId ? session()->getId() : null,
        ];
    }

    /**
     * Get all compare items
     */
    public function index(): JsonResponse
    {
        $identifier = $this->getIdentifier();

        $compare = Compare::when($identifier['client_id'], fn ($q) => $q->where('client_id', $identifier['client_id']))
            ->when($identifier['session_id'], fn ($q) => $q->where('session_id', $identifier['session_id']))
            ->with(['product.variants', 'product.category', 'product.brand', 'variant'])
            ->get();

        $items = $compare->map(function ($item) {
            $variant = $item->variant ?? $item->product->defaultVariant ?? $item->product->variants->first();

            return [
                'id' => $item->id,
                'product_id' => $item->product_id,
                'product_name' => $item->product->name,
                'product_slug' => $item->product->slug,
                'variant_id' => $variant?->id,
                'variant_name' => $variant?->name,
                'price' => $variant?->retail_price ?? 0,
                'image' => $item->product->thumbnail_thumb_url,
                'category' => $item->product->category?->name,
                'brand' => $item->product->brand?->name,
                'in_stock' => $item->product->isInStock(),
            ];
        });

        return response()->json([
            'success' => true,
            'items' => $items,
            'count' => $items->count(),
        ]);
    }

    /**
     * Toggle product in compare
     */
    public function toggle(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'product_id' => 'required|integer|exists:products,id',
            'variant_id' => 'nullable|integer|exists:product_variants,id',
        ]);

        $identifier = $this->getIdentifier();

        $compare = Compare::where('product_id', $validated['product_id'])
            ->when($identifier['client_id'], fn ($q) => $q->where('client_id', $identifier['client_id']))
            ->when($identifier['session_id'], fn ($q) => $q->where('session_id', $identifier['session_id']))
            ->first();

        if ($compare) {
            $compare->delete();

            return response()->json([
                'success' => true,
                'action' => 'removed',
                'message' => 'تم إزالة المنتج من المقارنة',
            ]);
        }

        // Limit compare to 4 products
        $count = Compare::when($identifier['client_id'], fn ($q) => $q->where('client_id', $identifier['client_id']))
            ->when($identifier['session_id'], fn ($q) => $q->where('session_id', $identifier['session_id']))
            ->count();

        if ($count >= 4) {
            return response()->json([
                'success' => false,
                'message' => 'لا يمكن إضافة أكثر من 4 منتجات للمقارنة',
            ], 400);
        }

        Compare::create([
            'client_id' => $identifier['client_id'],
            'session_id' => $identifier['session_id'],
            'product_id' => $validated['product_id'],
            'product_variant_id' => $validated['variant_id'] ?? null,
        ]);

        return response()->json([
            'success' => true,
            'action' => 'added',
            'message' => 'تمت إضافة المنتج للمقارنة',
        ]);
    }

    /**
     * Remove item from compare
     */
    public function remove(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'product_id' => 'required|integer',
        ]);

        $identifier = $this->getIdentifier();

        Compare::where('product_id', $validated['product_id'])
            ->when($identifier['client_id'], fn ($q) => $q->where('client_id', $identifier['client_id']))
            ->when($identifier['session_id'], fn ($q) => $q->where('session_id', $identifier['session_id']))
            ->delete();

        return response()->json([
            'success' => true,
            'message' => 'تم إزالة المنتج من المقارنة',
        ]);
    }

    /**
     * Clear compare list
     */
    public function clear(): JsonResponse
    {
        $identifier = $this->getIdentifier();

        Compare::when($identifier['client_id'], fn ($q) => $q->where('client_id', $identifier['client_id']))
            ->when($identifier['session_id'], fn ($q) => $q->where('session_id', $identifier['session_id']))
            ->delete();

        return response()->json([
            'success' => true,
            'message' => 'تم تفريغ قائمة المقارنة',
        ]);
    }
}
