<?php

namespace App\Http\Controllers;

use App\Models\City;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\Hash;
use Illuminate\Validation\Rules\Password;

class ClientDashboardController extends Controller
{
    /**
     * لوحة التحكم الرئيسية
     */
    public function index()
    {
        $client = Auth::guard('client')->user();

        // إحصائيات سريعة
        $stats = [
            'total_orders' => $client->orders()->count(),
            'pending_orders' => $client->orders()->where('status', 'pending')->count(),
            'delivered_orders' => $client->orders()->where('status', 'delivered')->count(),
            'total_spent' => $client->orders()->where('status', 'delivered')->sum('grand_total'),
        ];

        // إحصائيات المصمم
        $designerStats = [];
        if ($client->isDesigner()) {
            $designerStats = [
                'total_designs' => $client->designs()->count(),
                'public_designs' => $client->designs()->where('is_public', true)->count(),
                'total_stickers' => $client->stickers()->count(),
                'approved_stickers' => $client->stickers()->where('status', 'approved')->count(),
                'pending_stickers' => $client->stickers()->where('status', 'pending')->count(),
            ];
        }

        // آخر الطلبات
        $recentOrders = $client->orders()
            ->with(['items.product' => fn ($query) => $query->withTrashed()])
            ->latest()
            ->take(5)
            ->get();

        // آخر التصاميم (للمصمم)
        $recentDesigns = [];
        if ($client->isDesigner()) {
            $recentDesigns = $client->designs()
                ->with('productModel')
                ->latest()
                ->take(6)
                ->get();
        }

        return view('client.dashboard', compact('client', 'stats', 'designerStats', 'recentOrders', 'recentDesigns'));
    }

    /**
     * تصاميمي (للمصمم)
     */
    public function myDesigns(Request $request)
    {
        $client = Auth::guard('client')->user();

        $designs = $client->designs()
            ->with('productModel')
            ->latest()
            ->paginate(12)
            ->withQueryString();

        return view('client.designs', compact('designs'));
    }

    /**
     * مكتبة ملصقاتي (للمصمم)
     */
    public function myStickers(Request $request)
    {
        $client = Auth::guard('client')->user();

        $query = $client->stickers()->with('media')->latest();

        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $stickers = $query->paginate(12)->withQueryString();

        return view('client.stickers', compact('stickers'));
    }

    /**
     * صفحة الطلبات
     */
    public function orders(Request $request)
    {
        $client = Auth::guard('client')->user();

        $query = $client->orders()->with([
            'items.product' => fn ($query) => $query->withTrashed(),
            'shippingAddress.city',
        ]);

        // فلترة حسب الحالة
        if ($request->filled('status') && $request->status !== 'all') {
            $query->where('status', $request->status);
        }

        $orders = $query->latest()->paginate(10)->withQueryString();

        return view('client.orders', compact('orders'));
    }

    /**
     * تفاصيل الطلب
     */
    public function orderDetails($id)
    {
        $client = Auth::guard('client')->user();

        $order = $client->orders()
            ->with([
                'items.product' => fn ($query) => $query->withTrashed(),
                'shippingAddress.city',
            ])
            ->findOrFail($id);

        return view('client.order-details', compact('order'));
    }

    /**
     * صفحة الملف الشخصي
     */
    public function profile()
    {
        $client = Auth::guard('client')->user()->load('city');
        $cities = City::orderBy('name')->get();

        return view('client.profile', compact('client', 'cities'));
    }

    /**
     * تحديث الملف الشخصي
     */
    public function updateProfile(Request $request)
    {
        $client = Auth::guard('client')->user();

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'phone' => ['required', 'string', 'regex:/^07[0-9]{9}$/', 'unique:clients,phone,'.$client->id],
            'phone2' => ['nullable', 'string', 'regex:/^07[0-9]{9}$/'],
            'company' => ['nullable', 'string', 'max:255'],
            'address' => ['nullable', 'string', 'max:500'],
            'city_id' => ['nullable', 'exists:cities,id'],
        ], [
            'name.required' => 'الاسم مطلوب',
            'phone.required' => 'رقم الهاتف مطلوب',
            'phone.regex' => 'رقم الهاتف يجب أن يبدأ بـ 07 ويتكون من 11 رقم',
            'phone.unique' => 'رقم الهاتف مسجل مسبقاً',
            'phone2.regex' => 'رقم الهاتف الثاني يجب أن يبدأ بـ 07 ويتكون من 11 رقم',
        ]);

        $client->update($validated);

        return back()->with('success', 'تم تحديث الملف الشخصي بنجاح');
    }

    /**
     * تحديث كلمة المرور
     */
    public function updatePassword(Request $request)
    {
        $client = Auth::guard('client')->user();

        $validated = $request->validate([
            'current_password' => ['required', 'string'],
            'password' => ['required', 'confirmed', Password::min(8)],
        ], [
            'current_password.required' => 'كلمة المرور الحالية مطلوبة',
            'password.required' => 'كلمة المرور الجديدة مطلوبة',
            'password.min' => 'كلمة المرور يجب أن تكون 8 أحرف على الأقل',
            'password.confirmed' => 'كلمة المرور غير متطابقة',
        ]);

        // التحقق من كلمة المرور الحالية
        if (! Hash::check($validated['current_password'], $client->password)) {
            return back()->withErrors([
                'current_password' => 'كلمة المرور الحالية غير صحيحة',
            ]);
        }

        $client->update([
            'password' => Hash::make($validated['password']),
        ]);

        return back()->with('success', 'تم تحديث كلمة المرور بنجاح');
    }

    /**
     * صفحة عناوين الشحن
     */
    public function addresses()
    {
        $client = Auth::guard('client')->user();

        $addresses = $client->shippingAddresses()
            ->with('city')
            ->orderByDesc('is_default')
            ->get();

        $cities = City::orderBy('name')->get();

        return view('client.addresses', compact('addresses', 'cities'));
    }

    /**
     * إضافة عنوان شحن جديد
     */
    public function storeAddress(Request $request)
    {
        $client = Auth::guard('client')->user();

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'phone' => ['required', 'string', 'regex:/^07[0-9]{9}$/'],
            'address' => ['required', 'string', 'max:500'],
            'city_id' => ['required', 'exists:cities,id'],
            'is_default' => ['boolean'],
        ]);

        // إذا كان هذا العنوان الافتراضي، قم بإلغاء تفعيل الآخرين
        if ($validated['is_default'] ?? false) {
            $client->shippingAddresses()->update(['is_default' => false]);
        }

        $client->shippingAddresses()->create($validated);

        return back()->with('success', 'تم إضافة العنوان بنجاح');
    }

    /**
     * تحديث عنوان شحن
     */
    public function updateAddress(Request $request, $id)
    {
        $client = Auth::guard('client')->user();

        $address = $client->shippingAddresses()->findOrFail($id);

        $validated = $request->validate([
            'name' => ['required', 'string', 'max:255'],
            'phone' => ['required', 'string', 'regex:/^07[0-9]{9}$/'],
            'address' => ['required', 'string', 'max:500'],
            'city_id' => ['required', 'exists:cities,id'],
            'is_default' => ['boolean'],
        ]);

        // إذا كان هذا العنوان الافتراضي، قم بإلغاء تفعيل الآخرين
        if ($validated['is_default'] ?? false) {
            $client->shippingAddresses()
                ->where('id', '!=', $id)
                ->update(['is_default' => false]);
        }

        $address->update($validated);

        return back()->with('success', 'تم تحديث العنوان بنجاح');
    }

    /**
     * حذف عنوان شحن
     */
    public function deleteAddress($id)
    {
        $client = Auth::guard('client')->user();

        $address = $client->shippingAddresses()->findOrFail($id);

        // لا تسمح بحذف العنوان الافتراضي إذا كان هناك عناوين أخرى
        if ($address->is_default && $client->shippingAddresses()->count() > 1) {
            return back()->with('error', 'لا يمكن حذف العنوان الافتراضي. قم بتعيين عنوان آخر كافتراضي أولاً.');
        }

        $address->delete();

        return back()->with('success', 'تم حذف العنوان بنجاح');
    }

    /**
     * تعيين عنوان كافتراضي
     */
    public function setDefaultAddress($id)
    {
        $client = Auth::guard('client')->user();

        $address = $client->shippingAddresses()->findOrFail($id);

        // إلغاء تفعيل جميع العناوين الأخرى
        $client->shippingAddresses()->update(['is_default' => false]);

        // تفعيل هذا العنوان
        $address->update(['is_default' => true]);

        return back()->with('success', 'تم تعيين العنوان الافتراضي بنجاح');
    }
}
