<?php

namespace App\Http\Controllers;

use App\Enums\OrderStatus;
use App\Enums\PaymentStatus;
use App\Models\Cart;
use App\Models\City;
use App\Models\Coupon;
use App\Models\Currency;
use App\Models\Order;
use App\Models\ShippingAddress;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\RedirectResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Log;
use Illuminate\View\View;

class CheckoutController extends Controller
{
    private int $freeShippingThreshold = 100000;

    /**
     * Get cart for current user/session.
     */
    private function getCart(): Cart
    {
        $clientId = Auth::guard('client')->id();
        $sessionId = session()->getId();

        $cart = Cart::active()
            ->when($clientId, fn ($q) => $q->where('client_id', $clientId))
            ->when(! $clientId, fn ($q) => $q->where('session_id', $sessionId))
            ->first();

        if (! $cart) {
            $cart = Cart::create([
                'client_id' => $clientId,
                'session_id' => ! $clientId ? $sessionId : null,
                'currency_id' => Currency::getDefaultId(),
            ]);
        }

        return $cart;
    }

    /**
     * Display the checkout page.
     */
    public function index(): View|RedirectResponse
    {
        $cart = $this->getCart();
        $cart->load(['items.product', 'items.variant']);

        // Redirect to cart if empty
        if ($cart->items->isEmpty()) {
            return redirect()->route('cart.index')->with('error', 'السلة فارغة');
        }

        $items = $cart->items->map(function ($item) {
            return [
                'id' => $item->id,
                'product_id' => $item->product_id,
                'name' => $item->product->name,
                'slug' => $item->product->slug,
                'variant_name' => $item->variant?->name ?? $item->variant?->sku,
                'image' => $item->product->thumbnail_thumb_url ?? $item->product->thumbnail_url,
                'quantity' => $item->quantity,
                'unit_price' => $item->unit_price,
                'line_total' => $item->line_total,
            ];
        });

        $subtotal = $cart->subtotal;

        // Get cities for dropdown with shipping fees
        $cities = City::where('is_active', true)->orderBy('name')->get();

        // Get saved shipping addresses for logged in client
        $client = Auth::guard('client')->user();
        $savedAddresses = $client
            ? $client->shippingAddresses()->with('city')->orderByDesc('is_default')->get()
            : collect();

        // Get default address
        $defaultAddress = $savedAddresses->firstWhere('is_default', true) ?? $savedAddresses->first();

        // Get available coupons
        $availableCoupons = Coupon::valid()
            ->orderBy('discount_value', 'desc')
            ->take(10)
            ->get()
            ->map(function ($coupon) use ($subtotal) {
                return [
                    'code' => $coupon->code,
                    'name' => $coupon->name,
                    'description' => $coupon->type === 'percentage'
                        ? 'خصم '.$coupon->discount_value.'%'
                        : 'خصم '.number_format($coupon->discount_value).' د.ع',
                    'min_purchase' => $coupon->min_purchase_amount,
                    'can_use' => ! $coupon->min_purchase_amount || $subtotal >= $coupon->min_purchase_amount,
                ];
            });

        // Calculate initial shipping based on default address city
        $defaultCityId = $defaultAddress?->city_id ?? $client?->city_id;
        $defaultCity = $defaultCityId ? $cities->firstWhere('id', $defaultCityId) : null;
        $shipping = $defaultCity ? $defaultCity->shipping_fee : 5000;

        // Free shipping if above threshold
        if ($subtotal >= $this->freeShippingThreshold) {
            $shipping = 0;
        }

        return view('store.checkout', [
            'items' => $items,
            'cartCount' => $cart->items_count,
            'subtotal' => $subtotal,
            'shipping' => $shipping,
            'total' => $subtotal + $shipping,
            'freeShippingThreshold' => $this->freeShippingThreshold,
            'availableCoupons' => $availableCoupons,
            'cities' => $cities,
            'savedAddresses' => $savedAddresses,
            'defaultAddress' => $defaultAddress,
            'client' => $client,
        ]);
    }

    /**
     * Process the checkout and create order.
     */
    public function store(Request $request): RedirectResponse
    {
        $validated = $request->validate([
            'shipping_address_id' => 'nullable|integer|exists:shipping_addresses,id',
            'name' => 'required|string|max:255',
            'phone' => 'required|string|min:10|max:14',
            'city_id' => 'required|exists:cities,id',
            'area' => 'required|string|max:255',
            'address' => 'required|string|max:500',
            'notes' => 'nullable|string|max:1000',
            'payment_method' => 'required|in:cod,card,zaincash',
            'coupon_code' => 'nullable|string',
        ]);

        $client = Auth::guard('client')->user();
        $cart = $this->getCart();
        $cart->load(['items.product', 'items.variant']);

        if ($cart->items->isEmpty()) {
            return redirect()->route('cart.index')->with('error', 'السلة فارغة');
        }

        // Get city and shipping fee
        $city = City::find($validated['city_id']);
        if (! $city) {
            return redirect()->back()->with('error', 'المحافظة غير موجودة');
        }

        // Calculate totals
        $subtotal = $cart->subtotal;

        // Apply coupon if provided
        $coupon = null;
        $discount = 0;
        if (! empty($validated['coupon_code'])) {
            $coupon = Coupon::findByCode($validated['coupon_code']);
            if ($coupon && $coupon->is_valid && $coupon->canBeUsedBy($client)) {
                $discount = $coupon->calculateDiscount($subtotal);
            }
        }

        // Calculate shipping from city
        $subtotalAfterDiscount = $subtotal - $discount;
        $shippingFee = $subtotalAfterDiscount >= $this->freeShippingThreshold ? 0 : $city->shipping_fee;

        $grandTotal = $subtotalAfterDiscount + $shippingFee;

        try {
            DB::beginTransaction();

            // Use existing shipping address or create new one
            $shippingAddress = null;

            if (! empty($validated['shipping_address_id'])) {
                // Verify the address belongs to this client
                $shippingAddress = ShippingAddress::where('id', $validated['shipping_address_id'])
                    ->where('client_id', $client->id)
                    ->first();
            }

            if (! $shippingAddress) {
                // Create new shipping address
                $shippingAddress = ShippingAddress::create([
                    'client_id' => $client->id,
                    'city_id' => $validated['city_id'],
                    'name' => $validated['name'],
                    'phone' => $validated['phone'],
                    'address' => $validated['area'].', '.$validated['address'],
                    'is_default' => $client->shippingAddresses()->count() === 0,
                ]);
            }

            // Create the order
            $order = Order::create([
                'branch_id' => $client->branch_id ?? 1,
                'client_id' => $client->id,
                'currency_id' => Currency::getDefaultId() ?? 1,
                'exchange_rate' => 1,
                'shipping_address_id' => $shippingAddress->id,
                'shipping_method' => 'self',
                'subtotal' => $subtotal,
                'discount_total' => $discount,
                'shipping_fee' => $shippingFee,
                'cod_fee' => 0,
                'grand_total' => $grandTotal,
                'status' => OrderStatus::Pending,
                'payment_status' => PaymentStatus::UNPAID,
                'payment_method' => $validated['payment_method'],
                'coupon_id' => $coupon?->id,
                'customer_notes' => $validated['notes'],
            ]);

            // Create order items
            foreach ($cart->items as $item) {
                $order->items()->create([
                    'product_id' => $item->product_id,
                    'product_variant_id' => $item->product_variant_id,
                    'product_name' => $item->product->name,
                    'variant_name' => $item->variant?->name ?? $item->variant?->sku,
                    'sku' => $item->variant?->sku ?? $item->product->sku ?? 'N/A',
                    'quantity' => $item->quantity,
                    'unit_price' => $item->unit_price,
                    'unit_cost' => $item->variant?->cost ?? 0,
                    'discount_amount' => 0,
                    'line_total' => $item->line_total,
                ]);
            }

            // Redeem coupon if used
            if ($coupon && $discount > 0) {
                $coupon->redeem($client, $order, $discount);
            }

            // Clear the cart
            $cart->clear();

            DB::commit();

            return redirect()->route('checkout.success', ['order' => $order->order_number]);
        } catch (\Exception $e) {
            DB::rollBack();
            Log::error('Checkout error: '.$e->getMessage(), ['trace' => $e->getTraceAsString()]);

            return redirect()->back()->with('error', 'حدث خطأ أثناء إنشاء الطلب: '.$e->getMessage());
        }
    }

    /**
     * Display the order success page.
     */
    public function success(?string $orderNumber = null): View
    {
        $order = null;
        if ($orderNumber) {
            $order = Order::with(['items.product', 'shippingAddress.city'])
                ->where('order_number', $orderNumber)
                ->where('client_id', Auth::guard('client')->id())
                ->first();
        }

        return view('store.checkout-success', [
            'order' => $order,
        ]);
    }

    /**
     * Get shipping fee for a city.
     */
    public function getShippingFee(Request $request): JsonResponse
    {
        $cityId = $request->input('city_id');
        $subtotal = (int) $request->input('subtotal', 0);

        $city = City::find($cityId);

        if (! $city) {
            return response()->json(['fee' => 5000]);
        }

        $fee = $subtotal >= $this->freeShippingThreshold ? 0 : $city->shipping_fee;

        return response()->json([
            'fee' => $fee,
            'free' => $fee === 0,
        ]);
    }

    /**
     * Apply a coupon code.
     */
    public function applyCoupon(Request $request): JsonResponse
    {
        $request->validate([
            'code' => 'required|string',
            'subtotal' => 'required|numeric|min:0',
        ]);

        $code = strtoupper(trim($request->code));
        $subtotal = (int) $request->subtotal;
        $client = Auth::guard('client')->user();

        $coupon = Coupon::findByCode($code);

        if (! $coupon) {
            return response()->json([
                'success' => false,
                'message' => 'كود الخصم غير موجود',
            ]);
        }

        if (! $coupon->is_valid) {
            return response()->json([
                'success' => false,
                'message' => 'كود الخصم منتهي الصلاحية أو غير فعال',
            ]);
        }

        if (! $coupon->canBeUsedBy($client)) {
            return response()->json([
                'success' => false,
                'message' => 'لقد استخدمت هذا الكود من قبل',
            ]);
        }

        if ($coupon->min_purchase_amount && $subtotal < $coupon->min_purchase_amount) {
            return response()->json([
                'success' => false,
                'message' => 'الحد الأدنى للشراء هو '.number_format($coupon->min_purchase_amount).' د.ع',
            ]);
        }

        $discount = $coupon->calculateDiscount($subtotal);

        if ($discount <= 0) {
            return response()->json([
                'success' => false,
                'message' => 'لا يمكن تطبيق هذا الكود على طلبك الحالي',
            ]);
        }

        $description = $coupon->type === 'percentage'
            ? 'خصم '.$coupon->discount_value.'%'
            : 'خصم '.number_format($coupon->discount_value).' د.ع';

        return response()->json([
            'success' => true,
            'code' => $coupon->code,
            'discount' => $discount,
            'description' => $description,
        ]);
    }
}
