<?php

namespace App\Http\Controllers;

use App\Models\Cart;
use App\Models\Coupon;
use App\Models\Currency;
use App\Models\ProductVariant;
use Illuminate\Http\JsonResponse;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Auth;
use Illuminate\View\View;

class CartController extends Controller
{
    private int $freeShippingThreshold = 100000;

    private int $shippingFee = 5000;

    /**
     * Get or create cart for current user/session
     */
    private function getCart(): Cart
    {
        $clientId = Auth::guard('client')->id();
        $sessionId = session()->getId();

        $cart = Cart::active()
            ->when($clientId, fn ($q) => $q->where('client_id', $clientId))
            ->when(! $clientId, fn ($q) => $q->where('session_id', $sessionId))
            ->first();

        if (! $cart) {
            $cart = Cart::create([
                'client_id' => $clientId,
                'session_id' => ! $clientId ? $sessionId : null,
                'currency_id' => Currency::getDefaultId(),
            ]);
        }

        return $cart;
    }

    /**
     * Display the cart page.
     */
    public function index(): View
    {
        $cart = $this->getCart();
        $cart->load(['items.product', 'items.variant']);

        $items = $cart->items->map(function ($item) {
            return [
                'id' => $item->id,
                'product_id' => $item->product_id,
                'name' => $item->product->name,
                'slug' => $item->product->slug,
                'variant_id' => $item->product_variant_id,
                'variant_name' => $item->variant?->name ?? $item->variant?->sku,
                'image' => $item->product->thumbnail_thumb_url ?? $item->product->thumbnail_url,
                'quantity' => $item->quantity,
                'unit_price' => $item->unit_price,
                'line_total' => $item->line_total,
            ];
        });

        $subtotal = $cart->subtotal;
        $shipping = $subtotal >= $this->freeShippingThreshold ? 0 : $this->shippingFee;
        $total = $subtotal + $shipping;

        // Get active coupons
        $coupons = Coupon::valid()->orderBy('discount_value', 'desc')->limit(6)->get();

        return view('store.cart', [
            'items' => $items,
            'cartCount' => $cart->items_count,
            'subtotal' => $subtotal,
            'shipping' => $shipping,
            'total' => $total,
            'freeShippingThreshold' => $this->freeShippingThreshold,
            'coupons' => $coupons,
        ]);
    }

    /**
     * Get the cart item count.
     */
    public function count(): JsonResponse
    {
        $cart = $this->getCart();

        return response()->json([
            'success' => true,
            'count' => $cart->items_count,
        ]);
    }

    /**
     * Get the cart summary with all items.
     */
    public function summary(): JsonResponse
    {
        $cart = $this->getCart();
        $cart->load(['items.product', 'items.variant']);

        $items = $cart->items->map(function ($item) {
            return [
                'id' => $item->id,
                'product_id' => $item->product_id,
                'product_name' => $item->product->name,
                'variant_id' => $item->product_variant_id,
                'variant_name' => $item->variant?->name,
                'quantity' => $item->quantity,
                'unit_price' => $item->unit_price,
                'total_price' => $item->line_total,
                'image' => $item->product->thumbnail_thumb_url,
            ];
        });

        return response()->json([
            'success' => true,
            'items' => $items,
            'subtotal' => $cart->subtotal,
            'total' => $cart->subtotal,
            'items_count' => $cart->items_count,
        ]);
    }

    /**
     * Add item to cart.
     */
    public function add(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'variant_id' => 'required|integer|exists:product_variants,id',
            'quantity' => 'required|integer|min:1',
        ]);

        $cart = $this->getCart();
        $variant = ProductVariant::findOrFail($validated['variant_id']);

        // Get price based on client type
        $client = Auth::guard('client')->user();
        $priceType = ($client && $client->client_type === 'company') ? 'wholesale' : 'retail';
        $unitPrice = $variant->getPriceForType($priceType);

        $item = $cart->addItem($validated['variant_id'], $validated['quantity'], $unitPrice);

        return response()->json([
            'success' => true,
            'message' => 'تمت إضافة المنتج للسلة',
            'cart_count' => $cart->fresh()->items_count,
        ]);
    }

    /**
     * Update cart item quantity.
     */
    public function update(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'variant_id' => 'required|integer',
            'quantity' => 'required|integer|min:0',
        ]);

        $cart = $this->getCart();
        $cart->updateItemQuantity($validated['variant_id'], $validated['quantity']);

        return response()->json([
            'success' => true,
            'message' => 'تم تحديث الكمية',
            'cart_count' => $cart->fresh()->items_count,
        ]);
    }

    /**
     * Remove item from cart.
     */
    public function remove(Request $request): JsonResponse
    {
        $validated = $request->validate([
            'variant_id' => 'required|integer',
        ]);

        $cart = $this->getCart();
        $cart->removeItem($validated['variant_id']);

        return response()->json([
            'success' => true,
            'message' => 'تم حذف المنتج من السلة',
            'cart_count' => $cart->fresh()->items_count,
        ]);
    }

    /**
     * Clear the cart.
     */
    public function clear(): JsonResponse
    {
        $cart = $this->getCart();
        $cart->clear();

        return response()->json([
            'success' => true,
            'message' => 'تم تفريغ السلة',
        ]);
    }
}
