<?php

namespace App\Filament\Widgets;

use App\Exports\StatisticsExport;
use App\Filament\Resources\ClientReceipts\ClientReceiptResource;
use App\Filament\Resources\Finance\Expenses\ExpenseResource;
use App\Filament\Resources\Pos\PosResource;
use App\Filament\Resources\Purchases\PurchaseInvoices\PurchaseInvoiceResource;
use App\Filament\Resources\VaultTransactions\VaultTransactionResource;
use App\Models\ClientReceipt;
use App\Models\Expense;
use App\Models\Invoice;
use App\Models\PurchaseInvoice;
use App\Models\Vault;
use App\Models\VaultTransaction;
use Carbon\Carbon;
use Carbon\CarbonPeriod;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Concerns\InteractsWithForms;
use Filament\Forms\Contracts\HasForms;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Schema;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\HtmlString;
use Maatwebsite\Excel\Facades\Excel;

class Statistics extends BaseWidget implements HasForms
{
    use InteractsWithForms;

    protected static ?int $sort = 1;

    /** مهم: هذا مسار الـ Blade */
    protected string $view = 'filament.widgets.statistics';

    /** State للفلاتر */
    public ?array $data = [];

    public function mount(): void
    {
        $this->form->fill([
            'period' => 'today',
            'from_date' => null,
            'to_date' => null,
            'show_comparison' => '1',
        ]);
    }

    /* =========================
     * Helpers: read from data then (optional) filters
     * ========================= */

    protected function fv(string $key, mixed $default = null): mixed
    {
        if (is_array($this->data) && array_key_exists($key, $this->data) && $this->data[$key] !== null) {
            return $this->data[$key];
        }

        // إذا عندك Page Filters كذلك، نخليها fallback
        if (property_exists($this, 'filters') && is_array($this->filters) && array_key_exists($key, $this->filters) && $this->filters[$key] !== null) {
            return $this->filters[$key];
        }

        return $default;
    }

    protected function periodValue(): string
    {
        return (string) ($this->fv('period') ?? $this->fv('dateRange') ?? 'today');
    }

    protected function fromValue(): mixed
    {
        return $this->fv('from_date') ?? $this->fv('startDate');
    }

    protected function toValue(): mixed
    {
        return $this->fv('to_date') ?? $this->fv('endDate');
    }

    protected function showComparison(): bool
    {
        $v = $this->fv('show_comparison', '1');

        if (is_bool($v)) {
            return $v;
        }

        return (bool) ((int) $v);
    }

    /** يظهر شريط الفترة/المقارنة بعد أي تغيير */
    public function hasActiveSelection(): bool
    {
        $period = $this->periodValue();

        if ($period !== 'today') {
            return true;
        }

        if ($period === 'custom' && ($this->fromValue() || $this->toValue())) {
            return true;
        }

        if ((string) $this->fv('show_comparison', '1') !== '1') {
            return true;
        }

        return false;
    }

    /* =========================
     * Form
     * ========================= */

    public function form(Schema $form): Schema
    {
        return $form
            ->schema([
                Grid::make(4)->schema([

                    Select::make('period')
                        ->label('اختر الفترة')
                        ->options([
                            'today' => 'اليوم',
                            'yesterday' => 'أمس',
                            'this_week' => 'الأسبوع الحالي',
                            'last_week' => 'الأسبوع الماضي',
                            'last_7_days' => 'آخر 7 أيام',
                            'this_month' => 'الشهر الحالي',
                            'last_month' => 'الشهر الماضي',
                            'last_30_days' => 'آخر 30 يوم',
                            'last_90_days' => 'آخر 90 يوم',
                            'this_year' => 'هذه السنة',
                            'last_year' => 'السنة الماضية',
                            'custom' => 'مخصص',
                            'all' => 'كل الفترات',
                        ])
                        ->default('today')
                        ->native(false)
                        ->reactive()
                        ->afterStateUpdated(function ($state) {
                            if ($state !== 'custom') {
                                $this->data['from_date'] = null;
                                $this->data['to_date'] = null;
                            }

                            $this->dispatch('$refresh');
                        }),

                    DatePicker::make('from_date')
                        ->label('من تاريخ')
                        ->visible(fn ($get) => $get('period') === 'custom')
                        ->native(false)
                        ->maxDate(fn ($get) => $get('to_date') ?: now())
                        ->reactive()
                        ->afterStateUpdated(fn () => $this->dispatch('$refresh')),

                    DatePicker::make('to_date')
                        ->label('إلى تاريخ')
                        ->visible(fn ($get) => $get('period') === 'custom')
                        ->native(false)
                        ->minDate(fn ($get) => $get('from_date'))
                        ->reactive()
                        ->afterStateUpdated(fn () => $this->dispatch('$refresh')),

                    Select::make('show_comparison')
                        ->label('المقارنة مع اليوم')
                        ->options([
                            '1' => 'نعم',
                            '0' => 'لا',
                        ])
                        ->default('1')
                        ->native(false)
                        ->reactive()
                        ->afterStateUpdated(fn () => $this->dispatch('$refresh')),
                ]),
            ])
            ->statePath('data');
    }

    public function resetFilters(): void
    {
        $this->form->fill([
            'period' => 'today',
            'from_date' => null,
            'to_date' => null,
            'show_comparison' => '1',
        ]);

        $this->dispatch('$refresh');
    }

    /* =========================
     * Export
     * ========================= */

    public function exportStats()
    {
        $rows = $this->getStatsData(); // نفس البيانات اللي تعرضها لوحة المقارنة

        [$from, $to] = $this->resolveRange();

        $period = $this->periodValue();
        $periodNames = [
            'today' => 'اليوم',
            'yesterday' => 'أمس',
            'this_week' => 'الأسبوع الحالي',
            'last_week' => 'الأسبوع الماضي',
            'last_7_days' => 'آخر 7 أيام',
            'this_month' => 'الشهر الحالي',
            'last_month' => 'الشهر الماضي',
            'last_30_days' => 'آخر 30 يوم',
            'last_90_days' => 'آخر 90 يوم',
            'this_year' => 'هذه السنة',
            'last_year' => 'السنة الماضية',
            'custom' => 'مخصص',
            'all' => 'كل الفترات',
        ];

        $rangeLabel = ($from && $to)
            ? ($from->toDateString() === $to->toDateString()
                ? 'التاريخ: '.$from->toDateString()
                : 'من '.$from->toDateString().' إلى '.$to->toDateString())
            : 'كل الفترات';

        $meta = [
            'period_label' => $periodNames[$period] ?? $period,
            'range_label' => $rangeLabel,
            'comparison' => $this->showComparison(),
        ];

        $filename = 'statistics_'.$period.'_'.now()->format('Y-m-d_H-i-s').'.xlsx';

        return Excel::download(new StatisticsExport($rows, $meta), $filename);
    }

    /* =========================
     * Date Range
     * ========================= */

    protected function resolveRange(): array
    {
        $period = $this->periodValue();
        $now = now();

        return match ($period) {
            'today' => [$now->copy()->startOfDay(), $now->copy()->endOfDay()],
            'yesterday' => [$now->copy()->subDay()->startOfDay(), $now->copy()->subDay()->endOfDay()],
            'this_week' => [$now->copy()->startOfWeek(), $now->copy()->endOfWeek()],
            'last_week' => [$now->copy()->subWeek()->startOfWeek(), $now->copy()->subWeek()->endOfWeek()],
            'last_7_days' => [$now->copy()->subDays(6)->startOfDay(), $now->copy()->endOfDay()],
            'this_month' => [$now->copy()->startOfMonth(), $now->copy()->endOfMonth()],
            'last_month' => [$now->copy()->subMonth()->startOfMonth(), $now->copy()->subMonth()->endOfMonth()],
            'last_30_days' => [$now->copy()->subDays(29)->startOfDay(), $now->copy()->endOfDay()],
            'last_90_days' => [$now->copy()->subDays(89)->startOfDay(), $now->copy()->endOfDay()],
            'this_year' => [$now->copy()->startOfYear(), $now->copy()->endOfYear()],
            'last_year' => [$now->copy()->subYear()->startOfYear(), $now->copy()->subYear()->endOfYear()],
            'custom' => [
                $this->fromValue()
                    ? Carbon::parse($this->fromValue())->startOfDay()
                    : $now->copy()->startOfDay(),
                $this->toValue()
                    ? Carbon::parse($this->toValue())->endOfDay()
                    : $now->copy()->endOfDay(),
            ],
            'all' => [null, null],
            default => [$now->copy()->startOfDay(), $now->copy()->endOfDay()],
        };
    }

    /* =========================
     * Comparison helpers (compare with today)
     * ========================= */

    protected function todayRange(): array
    {
        return [Carbon::today()->startOfDay(), Carbon::today()->endOfDay()];
    }

    protected function compareMetaInt(int $current, int $today): array
    {
        if ($today === 0 && $current === 0) {
            return [
                'has' => false,
                'diff' => 0,
                'pct' => 0,
                'dir' => 'same',
                'text' => 'لا توجد بيانات للمقارنة',
            ];
        }

        if ($today === 0) {
            return [
                'has' => true,
                'diff' => $current,
                'pct' => 100,
                'dir' => $current > 0 ? 'up' : 'same',
                'text' => 'أعلى من اليوم',
            ];
        }

        $diff = $current - $today;
        $dir = $diff > 0 ? 'up' : ($diff < 0 ? 'down' : 'same');

        $pct = (int) round(abs(($diff / $today) * 100));

        $text = $dir === 'up'
            ? "أعلى من اليوم {$pct}%"
            : ($dir === 'down' ? "أقل من اليوم {$pct}%" : 'مساوي لبيانات اليوم');

        return [
            'has' => true,
            'diff' => (int) $diff,
            'pct' => (int) $pct,
            'dir' => $dir,
            'text' => $text,
        ];
    }

    /* =========================
     * Data for export + comparison panel
     * ========================= */

    public function getStatsData(): array
    {
        $branchId = function_exists('user_info') ? (int) user_info('branch_id') : null;

        [$from, $to] = $this->resolveRange();
        $showComparison = $this->showComparison();
        [$todayFrom, $todayTo] = $this->todayRange();

        $fromDate = $from?->toDateString();
        $toDate = $to?->toDateString();

        $todayFromDate = $todayFrom->toDateString();
        $todayToDate = $todayTo->toDateString();

        /* المبيعات */
        $salesQ = Invoice::query()
            ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
            ->when($from && $to, fn (Builder $q) => $q->whereBetween('invoice_date', [$fromDate, $toDate]));

        $salesTotal = (int) $salesQ->sum('grand_total');
        $salesCount = (int) $salesQ->count();

        $todaySalesTotal = 0;
        if ($showComparison) {
            $todaySalesTotal = (int) Invoice::query()
                ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
                ->whereBetween('invoice_date', [$todayFromDate, $todayToDate])
                ->sum('grand_total');
        }

        /* المشتريات */
        $purchasesQ = PurchaseInvoice::query()
            ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
            ->when($from && $to, fn (Builder $q) => $q->whereBetween('invoice_date', [$fromDate, $toDate]));

        $purchasesTotal = (int) $purchasesQ->sum('total');
        $purchasesCount = (int) $purchasesQ->count();

        $todayPurchasesTotal = 0;
        if ($showComparison) {
            $todayPurchasesTotal = (int) PurchaseInvoice::query()
                ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
                ->whereBetween('invoice_date', [$todayFromDate, $todayToDate])
                ->sum('total');
        }

        /* المصروفات */
        $expensesQ = Expense::query()
            ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
            ->when($from && $to, fn (Builder $q) => $q->whereBetween('spent_at', [$from, $to]));

        $expensesTotal = (int) $expensesQ->sum('amount');
        $expensesCount = (int) $expensesQ->count();

        $todayExpensesTotal = 0;
        if ($showComparison) {
            $todayExpensesTotal = (int) Expense::query()
                ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
                ->whereBetween('spent_at', [$todayFrom, $todayTo])
                ->sum('amount');
        }

        /* الأقساط */
        $receiptsQ = ClientReceipt::query()
            ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
            ->when($from && $to, fn (Builder $q) => $q->whereBetween('receipt_date', [$fromDate, $toDate]));

        $receiptsPaid = (int) $receiptsQ->sum('total_paid');
        $receiptsCount = (int) $receiptsQ->count();

        $todayReceiptsPaid = 0;
        if ($showComparison) {
            $todayReceiptsPaid = (int) ClientReceipt::query()
                ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
                ->whereBetween('receipt_date', [$todayFromDate, $todayToDate])
                ->sum('total_paid');
        }

        /* الخزنة (صافي) */
        $vaultIds = Vault::query()
            ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
            ->pluck('id')
            ->all();

        $netVault = 0;
        if (! empty($vaultIds) && $from && $to) {
            $dep = (int) VaultTransaction::query()
                ->whereIn('vault_id', $vaultIds)
                ->where('transaction_type', VaultTransaction::TYPE_DEPOSIT)
                ->whereBetween('created_at', [$from, $to])
                ->sum('amount');

            $wd = (int) VaultTransaction::query()
                ->whereIn('vault_id', $vaultIds)
                ->where('transaction_type', VaultTransaction::TYPE_WITHDRAW)
                ->whereBetween('created_at', [$from, $to])
                ->sum('amount');

            $netVault = (int) ($dep - $wd);
        }

        $todayNetVault = 0;
        if ($showComparison && ! empty($vaultIds)) {
            $depT = (int) VaultTransaction::query()
                ->whereIn('vault_id', $vaultIds)
                ->where('transaction_type', VaultTransaction::TYPE_DEPOSIT)
                ->whereBetween('created_at', [$todayFrom, $todayTo])
                ->sum('amount');

            $wdT = (int) VaultTransaction::query()
                ->whereIn('vault_id', $vaultIds)
                ->where('transaction_type', VaultTransaction::TYPE_WITHDRAW)
                ->whereBetween('created_at', [$todayFrom, $todayTo])
                ->sum('amount');

            $todayNetVault = (int) ($depT - $wdT);
        }

        return [
            'sales' => [
                'title' => 'المبيعات',
                'total' => $salesTotal,
                'count' => $salesCount,
                'today_total' => $todaySalesTotal,
                'cmp' => $this->compareMetaInt($salesTotal, $todaySalesTotal),
            ],
            'purchases' => [
                'title' => 'المشتريات',
                'total' => $purchasesTotal,
                'count' => $purchasesCount,
                'today_total' => $todayPurchasesTotal,
                'cmp' => $this->compareMetaInt($purchasesTotal, $todayPurchasesTotal),
            ],
            'expenses' => [
                'title' => 'المصروفات',
                'total' => $expensesTotal,
                'count' => $expensesCount,
                'today_total' => $todayExpensesTotal,
                'cmp' => $this->compareMetaInt($expensesTotal, $todayExpensesTotal),
            ],
            'receipts' => [
                'title' => 'الأقساط',
                'total' => $receiptsPaid,
                'count' => $receiptsCount,
                'today_total' => $todayReceiptsPaid,
                'cmp' => $this->compareMetaInt($receiptsPaid, $todayReceiptsPaid),
            ],
            'vault' => [
                'title' => 'حركة الخزنة (صافي)',
                'total' => $netVault,
                'count' => 0,
                'today_total' => $todayNetVault,
                'cmp' => $this->compareMetaInt($netVault, $todayNetVault),
            ],
        ];
    }

    /* =========================
     * Cards (بدون مقارنة داخل الكارت)
     * ========================= */

    protected function getStats(): array
    {
        $branchId = function_exists('user_info') ? (int) user_info('branch_id') : null;

        [$from, $to] = $this->resolveRange();
        $fromDate = $from?->toDateString();
        $toDate = $to?->toDateString();

        $cardHover = $this->hoverCardClasses();

        /* المبيعات */
        $salesQ = Invoice::query()
            ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
            ->when($from && $to, fn (Builder $q) => $q->whereBetween('invoice_date', [$fromDate, $toDate]));

        $salesTotal = (int) $salesQ->sum('grand_total');
        $salesCount = (int) $salesQ->count();
        $salesPaid = (int) $salesQ->sum('paid_amount');
        $salesDue = (int) $salesQ->sum('due_amount');

        $salesChart = $this->sparklineSumByDate(
            baseQuery: Invoice::query()
                ->when($branchId, fn ($q) => $q->where('branch_id', $branchId)),
            dateColumn: 'invoice_date',
            sumColumn: 'grand_total',
            days: 8,
            endDate: $toDate ?: today()->toDateString()
        );

        /* المشتريات */
        $purchasesQ = PurchaseInvoice::query()
            ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
            ->when($from && $to, fn (Builder $q) => $q->whereBetween('invoice_date', [$fromDate, $toDate]));

        $purchasesTotal = (int) $purchasesQ->sum('total');
        $purchasesCount = (int) $purchasesQ->count();
        $purchasesPaid = (int) $purchasesQ->sum('paid');
        $purchasesDue = (int) $purchasesQ->sum('due');

        $purchasesChart = $this->sparklineSumByDate(
            baseQuery: PurchaseInvoice::query()
                ->when($branchId, fn ($q) => $q->where('branch_id', $branchId)),
            dateColumn: 'invoice_date',
            sumColumn: 'total',
            days: 8,
            endDate: $toDate ?: today()->toDateString()
        );

        /* المصروفات */
        $expensesQ = Expense::query()
            ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
            ->when($from && $to, fn (Builder $q) => $q->whereBetween('spent_at', [$from, $to]));

        $expensesTotal = (int) $expensesQ->sum('amount');
        $expensesCount = (int) $expensesQ->count();

        $expensesChart = $this->sparklineSumByDateTime(
            baseQuery: Expense::query()
                ->when($branchId, fn ($q) => $q->where('branch_id', $branchId)),
            dateTimeColumn: 'spent_at',
            sumColumn: 'amount',
            days: 8,
            endDate: $to ?: now()
        );

        /* الأقساط */
        $receiptsQ = ClientReceipt::query()
            ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
            ->when($from && $to, fn (Builder $q) => $q->whereBetween('receipt_date', [$fromDate, $toDate]));

        $receiptsCount = (int) $receiptsQ->count();
        $receiptsPaid = (int) $receiptsQ->sum('total_paid');
        $receiptsTotal = (int) $receiptsQ->sum('total_amount');
        $receiptsRemain = (int) $receiptsQ->sum('total_remaining');

        $receiptsChart = $this->sparklineSumByDate(
            baseQuery: ClientReceipt::query()
                ->when($branchId, fn ($q) => $q->where('branch_id', $branchId)),
            dateColumn: 'receipt_date',
            sumColumn: 'total_paid',
            days: 8,
            endDate: $toDate ?: today()->toDateString()
        );

        /* الخزنة */
        $vaultIds = Vault::query()
            ->when($branchId, fn (Builder $q) => $q->where('branch_id', $branchId))
            ->pluck('id')
            ->all();

        $depositsQ = VaultTransaction::query()
            ->when(! empty($vaultIds), fn (Builder $q) => $q->whereIn('vault_id', $vaultIds))
            ->when($from && $to, fn (Builder $q) => $q->whereBetween('created_at', [$from, $to]))
            ->where('transaction_type', VaultTransaction::TYPE_DEPOSIT);

        $withdrawsQ = VaultTransaction::query()
            ->when(! empty($vaultIds), fn (Builder $q) => $q->whereIn('vault_id', $vaultIds))
            ->when($from && $to, fn (Builder $q) => $q->whereBetween('created_at', [$from, $to]))
            ->where('transaction_type', VaultTransaction::TYPE_WITHDRAW);

        $depositsCount = (int) $depositsQ->count();
        $depositsTotal = (int) $depositsQ->sum('amount');

        $withdrawsCount = (int) $withdrawsQ->count();
        $withdrawsTotal = (int) $withdrawsQ->sum('amount');

        $netVault = (int) ($depositsTotal - $withdrawsTotal);

        $vaultChart = $this->sparklineVaultNetByDay(
            vaultIds: $vaultIds,
            days: 8,
            endDate: $to ?: now()
        );

        // روابط الموارد
        $posUrl = PosResource::getUrl('index');
        $purchaseUrl = $this->resourceIndexUrl(PurchaseInvoiceResource::class);
        $expenseUrl = $this->resourceIndexUrl(ExpenseResource::class);
        $receiptUrl = $this->resourceIndexUrl(ClientReceiptResource::class);
        $vaultTxUrl = $this->resourceIndexUrl(VaultTransactionResource::class);

        return [
            Stat::make('المبيعات', $this->animatedAmount($salesTotal, 'sales'))
                ->description('عدد: '.$salesCount.' | مدفوع: '.$this->formatNumber($salesPaid).' | متبقي: '.$this->formatNumber($salesDue))
                ->descriptionIcon('heroicon-o-shopping-cart')
                ->chart($salesChart)
                ->color('success')
                ->url($posUrl ?: null)
                ->extraAttributes(['class' => $cardHover.' cursor-pointer']),

            Stat::make('المشتريات', $this->animatedAmount($purchasesTotal, 'purchases'))
                ->description('عدد: '.$purchasesCount.' | مدفوع: '.$this->formatNumber($purchasesPaid).' | متبقي: '.$this->formatNumber($purchasesDue))
                ->descriptionIcon('heroicon-o-shopping-bag')
                ->chart($purchasesChart)
                ->color('warning')
                ->url($purchaseUrl ?: null)
                ->extraAttributes(['class' => $cardHover.' cursor-pointer']),

            Stat::make('المصروفات', $this->animatedAmount($expensesTotal, 'expenses'))
                ->description('عدد: '.$expensesCount)
                ->descriptionIcon('heroicon-o-banknotes')
                ->chart($expensesChart)
                ->color('danger')
                ->url($expenseUrl ?: null)
                ->extraAttributes(['class' => $cardHover.' cursor-pointer']),

            Stat::make('الأقساط', $this->animatedAmount($receiptsPaid, 'receipts'))
                ->description('عدد: '.$receiptsCount.' | إجمالي: '.$this->formatNumber($receiptsTotal).' | متبقي: '.$this->formatNumber($receiptsRemain))
                ->descriptionIcon('heroicon-o-receipt-refund')
                ->chart($receiptsChart)
                ->color('info')
                ->url($receiptUrl ?: null)
                ->extraAttributes(['class' => $cardHover.' cursor-pointer']),

            Stat::make('حركة الخزنة (صافي)', $this->animatedAmount($netVault, 'vault'))
                ->description(
                    'إيداع: '.$this->formatNumber($depositsTotal).' ('.$depositsCount.')'.
                        ' | سحب: '.$this->formatNumber($withdrawsTotal).' ('.$withdrawsCount.')'
                )
                ->descriptionIcon('heroicon-o-building-library')
                ->chart($vaultChart)
                ->color($netVault >= 0 ? 'success' : 'danger')
                ->url($vaultTxUrl ?: null)
                ->extraAttributes(['class' => $cardHover.' cursor-pointer']),
        ];
    }

    /* =========================
     * Animated numbers (مع wire:key حتى يتحدث بعد الفلترة)
     * ========================= */

    protected function animatedAmount(int $amount, string $key): HtmlString
    {
        return new HtmlString(
            $this->countUpSpan($amount, $key).
                '<span class="ms-1 text-sm font-medium text-gray-500 dark:text-gray-400">د.ع</span>'
        );
    }

    protected function animatedNumber(int $number, string $key): HtmlString
    {
        return new HtmlString($this->countUpSpan($number, $key));
    }

    protected function countUpSpan(int $to, string $key): string
    {
        // مهم: نخليها int + لا نخليها سالبة داخل الحركة (إذا تريد سالب بالخزنة نقدر نخليه)
        $value = (int) $to;
        $abs = abs($value);

        $period = $this->periodValue();
        $from = $this->fromValue() ? Carbon::parse($this->fromValue())->toDateString() : 'na';
        $toD = $this->toValue() ? Carbon::parse($this->toValue())->toDateString() : 'na';

        $wireKey = "stat-{$key}-{$period}-{$from}-{$toD}-{$value}";

        $sign = $value < 0 ? '-' : '';

        return <<<HTML
<span
    wire:key="{$wireKey}"
    class="tabular-nums"
    x-data="{
        to: {$abs},
        v: 0,
        dur: 780,
        start() {
            const start = performance.now();
            const from = 0;
            const to = this.to;

            const tick = (t) => {
                const p = Math.min(1, (t - start) / this.dur);
                const eased = 1 - Math.pow(1 - p, 3);
                this.v = Math.round(from + (to - from) * eased);
                if (p < 1) requestAnimationFrame(tick);
            };

            requestAnimationFrame(tick);
        },
        fmt(n) { return new Intl.NumberFormat('en-US').format(n); }
    }"
    x-init="start()"
    x-text="'{$sign}' + fmt(v)"
></span>
HTML;
    }

    /* =========================
     * UI helpers
     * ========================= */

    protected function hoverCardClasses(): string
    {
        return 'transition-all duration-200 ease-out hover:-translate-y-0.5 hover:shadow-xl dark:hover:shadow-2xl';
    }

    protected function formatNumber(int|float $value): string
    {
        return number_format((int) $value, 0, '.', ',');
    }

    protected function getColumns(): int
    {
        return 4;
    }

    protected function resourceIndexUrl(string $resourceClass, array $params = []): ?string
    {
        if (! class_exists($resourceClass)) {
            return null;
        }

        return method_exists($resourceClass, 'getUrl')
            ? $resourceClass::getUrl('index', $params)
            : null;
    }

    /* =========================
     * Sparklines
     * ========================= */

    protected function sparklineSumByDate(
        Builder $baseQuery,
        string $dateColumn,
        string $sumColumn,
        int $days = 8,
        ?string $endDate = null
    ): array {
        $end = $endDate ? Carbon::parse($endDate) : today();
        $start = (clone $end)->subDays($days - 1);

        $rows = (clone $baseQuery)
            ->whereBetween($dateColumn, [$start->toDateString(), $end->toDateString()])
            ->selectRaw("DATE($dateColumn) as d, COALESCE(SUM($sumColumn), 0) as s")
            ->groupBy('d')
            ->pluck('s', 'd')
            ->all();

        $out = [];
        foreach (CarbonPeriod::create($start->toDateString(), $end->toDateString()) as $day) {
            $k = Carbon::parse($day)->toDateString();
            $out[] = (int) ($rows[$k] ?? 0);
        }

        return $out;
    }

    protected function sparklineSumByDateTime(
        Builder $baseQuery,
        string $dateTimeColumn,
        string $sumColumn,
        int $days = 8,
        ?Carbon $endDate = null
    ): array {
        $end = $endDate ? $endDate->copy() : now();
        $start = $end->copy()->subDays($days - 1)->startOfDay();

        $rows = (clone $baseQuery)
            ->whereBetween($dateTimeColumn, [$start, $end])
            ->selectRaw("DATE($dateTimeColumn) as d, COALESCE(SUM($sumColumn), 0) as s")
            ->groupBy('d')
            ->pluck('s', 'd')
            ->all();

        $out = [];
        foreach (CarbonPeriod::create($start->toDateString(), $end->toDateString()) as $day) {
            $k = Carbon::parse($day)->toDateString();
            $out[] = (int) ($rows[$k] ?? 0);
        }

        return $out;
    }

    protected function sparklineCountByDateTime(
        Builder $baseQuery,
        string $dateTimeColumn,
        int $days = 8,
        ?Carbon $endDate = null
    ): array {
        $end = $endDate ? $endDate->copy() : now();
        $start = $end->copy()->subDays($days - 1)->startOfDay();

        $rows = (clone $baseQuery)
            ->whereBetween($dateTimeColumn, [$start, $end])
            ->selectRaw("DATE($dateTimeColumn) as d, COUNT(*) as c")
            ->groupBy('d')
            ->pluck('c', 'd')
            ->all();

        $out = [];
        foreach (CarbonPeriod::create($start->toDateString(), $end->toDateString()) as $day) {
            $k = Carbon::parse($day)->toDateString();
            $out[] = (int) ($rows[$k] ?? 0);
        }

        return $out;
    }

    protected function sparklineVaultNetByDay(array $vaultIds, int $days = 8, ?Carbon $endDate = null): array
    {
        $end = $endDate ? $endDate->copy() : now();
        $start = $end->copy()->subDays($days - 1)->startOfDay();

        if (empty($vaultIds)) {
            return array_fill(0, $days, 0);
        }

        $dep = VaultTransaction::query()
            ->whereIn('vault_id', $vaultIds)
            ->where('transaction_type', VaultTransaction::TYPE_DEPOSIT)
            ->whereBetween('created_at', [$start, $end])
            ->selectRaw('DATE(created_at) as d, COALESCE(SUM(amount), 0) as s')
            ->groupBy('d')
            ->pluck('s', 'd')
            ->all();

        $wd = VaultTransaction::query()
            ->whereIn('vault_id', $vaultIds)
            ->where('transaction_type', VaultTransaction::TYPE_WITHDRAW)
            ->whereBetween('created_at', [$start, $end])
            ->selectRaw('DATE(created_at) as d, COALESCE(SUM(amount), 0) as s')
            ->groupBy('d')
            ->pluck('s', 'd')
            ->all();

        $out = [];
        foreach (CarbonPeriod::create($start->toDateString(), $end->toDateString()) as $day) {
            $k = Carbon::parse($day)->toDateString();
            $out[] = (int) (($dep[$k] ?? 0) - ($wd[$k] ?? 0));
        }

        return $out;
    }
}
