<?php

namespace App\Filament\Resources\WorkSchedules\Tables;

use Filament\Actions\ActionGroup;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class WorkSchedulesTable
{
    public static function configure(Table $table): Table
    {
        $isAdmin = user_info()->hasRole('admin');
        $branchId = (int) (user_info('branch_id') ?? 0);

        return $table
            /**
             * حماية الاستعلام:
             * غير الأدمن يشاهد فقط جداول فرعه.
             */
            ->modifyQueryUsing(function (Builder $query) use ($isAdmin, $branchId) {
                if (! $isAdmin && $branchId > 0) {
                    $query->where('branch_id', $branchId);
                }
            })

            ->columns([
                TextColumn::make('name')
                    ->label('الوصف')
                    ->sortable()
                    ->searchable(),

                // يظهر فقط للأدمن
                TextColumn::make('branch.name')
                    ->label('الفرع')
                    ->sortable()
                    ->toggleable()
                    ->visible($isAdmin),

                TextColumn::make('employee.name')
                    ->label('الموظف')
                    ->placeholder('عام')
                    ->sortable()
                    ->toggleable()
                    ->searchable(),

                TextColumn::make('weekday')
                    ->label('اليوم')
                    ->formatStateUsing(fn($state) => match ((int) $state) {
                        0 => 'الأحد',
                        1 => 'الإثنين',
                        2 => 'الثلاثاء',
                        3 => 'الأربعاء',
                        4 => 'الخميس',
                        5 => 'الجمعة',
                        6 => 'السبت',
                        default => '-',
                    })
                    ->sortable(),

                TextColumn::make('expected_start')
                    ->label('بداية')
                    ->time('H:i')
                    ->sortable(),

                TextColumn::make('expected_end')
                    ->label('نهاية')
                    ->time('H:i')
                    ->sortable(),

                TextColumn::make('expected_minutes')
                    ->label('الدقائق')
                    ->numeric()
                    ->sortable()
                    ->alignEnd(),

                TextColumn::make('grace_period_minutes')
                    ->label('سماح')
                    ->numeric()
                    ->alignEnd()
                    ->toggleable(),

                TextColumn::make('overtime_rate_percentage')
                    ->label('نسبة إضافي %')
                    ->numeric()
                    ->alignEnd()
                    ->toggleable(),

                IconColumn::make('is_active')
                    ->label('مفعل')
                    ->boolean(),
            ])

            ->filters([
                // فلتر الفرع للأدمن فقط + إصلاح hasRole
                SelectFilter::make('branch_id')
                    ->label('الفرع')
                    ->relationship('branch', 'name')
                    ->visible($isAdmin),

                SelectFilter::make('weekday')
                    ->label('اليوم')
                    ->options([
                        0 => 'الأحد',
                        1 => 'الإثنين',
                        2 => 'الثلاثاء',
                        3 => 'الأربعاء',
                        4 => 'الخميس',
                        5 => 'الجمعة',
                        6 => 'السبت',
                    ]),

                SelectFilter::make('is_active')
                    ->label('مفعل')
                    ->options([
                        1 => 'نعم',
                        0 => 'لا',
                    ]),
            ])

            ->recordActions([
                ActionGroup::make([
                    EditAction::make(),
                    DeleteAction::make(),
                ]),
            ])

            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }
}
