<?php

namespace App\Filament\Resources\WorkSchedules\Schemas;

use Filament\Schemas\Schema;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Toggle;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Section;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\TimePicker;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;

class WorkScheduleForm
{
    public static function configure(Schema $schema): Schema
    {
        $isAdmin = user_info()->hasRole('admin');
        $userBranchId = (int) (user_info('branch_id') ?? 0);

        return $schema->components([
            Section::make('تفاصيل جدول الدوام')
                ->schema([
                    /**
                     * الفرع:
                     * - للأدمن: Select
                     * - لغير الأدمن: Hidden + تثبيت قيمة الفرع
                     */
                    Select::make('branch_id')
                        ->label('الفرع')
                        ->relationship('branch', 'name')
                        ->searchable()
                        ->preload()
                        ->required()
                        ->live()
                        ->visible($isAdmin),

                    Hidden::make('branch_id')
                        ->default($userBranchId)
                        ->dehydrated(! $isAdmin)
                        ->visible(! $isAdmin),

                    /**
                     * الموظف:
                     * يفلتر حسب الفرع (الفرع المختار للأدمن، وفرع المستخدم لغير الأدمن)
                     */
                    Select::make('employee_id')
                        ->label('الموظف (اختياري)')
                        ->relationship(
                            name: 'employee',
                            titleAttribute: 'name',
                            modifyQueryUsing: function ($query) use ($isAdmin, $userBranchId) {
                                $branchId = $isAdmin
                                    ? (int) request()->input('branch_id', 0) // احتياطي
                                    : $userBranchId;

                                // إذا لا يوجد branch واضح، لا نفلتر حتى لا تُكسر الشاشة
                                if ($branchId > 0) {
                                    $query->where('branch_id', $branchId);
                                }

                                return $query->orderBy('name');
                            }
                        )
                        ->searchable()
                        ->preload()
                        ->helperText('اتركه فارغاً لتطبيق الجدول على جميع موظفي .')
                        ->live()
                        ->disabled(fn(Get $get) => $isAdmin && blank($get('branch_id'))),

                    TextInput::make('name')
                        ->label('الاسم/الوصف')
                        ->maxLength(150)
                        ->placeholder('مثال: صباحي، مسائي...'),

                    Select::make('weekday')
                        ->label('اليوم')
                        ->required()
                        ->options([
                            0 => 'الأحد',
                            1 => 'الإثنين',
                            2 => 'الثلاثاء',
                            3 => 'الأربعاء',
                            4 => 'الخميس',
                            5 => 'الجمعة',
                            6 => 'السبت',
                        ]),

                    TimePicker::make('expected_start')
                        ->label('بداية الدوام')
                        ->seconds(false)
                        ->required()
                        ->live()
                        ->afterStateUpdated(function (Get $get, Set $set) {
                            self::syncExpectedMinutes($get, $set);
                        }),

                    TimePicker::make('expected_end')
                        ->label('نهاية الدوام')
                        ->seconds(false)
                        ->required()
                        ->live()
                        ->afterStateUpdated(function (Get $get, Set $set) {
                            self::syncExpectedMinutes($get, $set);
                        }),

                    /**
                     * الدقائق المتوقعة:
                     * محسوبة تلقائياً من start/end (عدد صحيح).
                     * يمكن جعلها readOnly لمنع التعديل اليدوي.
                     */
                    TextInput::make('expected_minutes')
                        ->label('الدقائق المتوقعة')
                        ->numeric()
                        ->required()
                        ->minValue(0)
                        ->default(0)
                        ->readOnly()
                        ->helperText('تُحسب تلقائياً من وقت البداية والنهاية (تدعم الدوام الليلي).'),

                    TextInput::make('grace_period_minutes')
                        ->label('فترة السماح (دقائق)')
                        ->numeric()
                        ->minValue(0)
                        ->default(15)
                        ->required(),

                    TextInput::make('overtime_rate_percentage')
                        ->label('نسبة الوقت الإضافي %')
                        ->numeric()
                        ->minValue(100)
                        ->default(150)
                        ->required(),

                    DatePicker::make('effective_from')
                        ->label('يبدأ من')
                        ->native(false)
                        ->live(),

                    DatePicker::make('effective_to')
                        ->label('ينتهي في')
                        ->native(false)
                        ->rule('after_or_equal:effective_from'),

                    Toggle::make('is_active')
                        ->label('مفعل')
                        ->default(true),
                ])
                ->columns(2)
                ->columnSpanFull(),
        ]);
    }

    /**
     * يحسب expected_minutes من وقت البداية والنهاية (يدعم عبور منتصف الليل)
     * الناتج عدد صحيح.
     */
    protected static function syncExpectedMinutes(Get $get, Set $set): void
    {
        $start = $get('expected_start');
        $end = $get('expected_end');

        if (blank($start) || blank($end)) {
            $set('expected_minutes', 0);
            return;
        }

        // صيغة TimePicker غالباً "HH:MM" أو "HH:MM:SS"
        $startParts = explode(':', (string) $start);
        $endParts = explode(':', (string) $end);

        $startMinutes = ((int) ($startParts[0] ?? 0) * 60) + (int) ($startParts[1] ?? 0);
        $endMinutes = ((int) ($endParts[0] ?? 0) * 60) + (int) ($endParts[1] ?? 0);

        $diff = $endMinutes - $startMinutes;

        // دعم الدوام الليلي (مثلاً 22:00 إلى 06:00)
        if ($diff < 0) {
            $diff += 24 * 60;
        }

        $set('expected_minutes', (int) max(0, $diff));
    }
}
