<?php

namespace App\Filament\Resources\VaultTransactions\Widgets;

use App\Models\Vault;
use App\Models\VaultTransaction;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Support\Facades\Auth;

class VaultSourcesToday extends BaseWidget
{
    protected ?string $pollingInterval = null;

    protected static bool $isLazy = true;

    protected function getStats(): array
    {
        $branchId = (int) (Auth::user()?->branch_id ?? 0);

        if (! $branchId) {
            return [
                Stat::make('مصادر اليوم', '—')
                    ->description('لا يوجد فرع مرتبط بالمستخدم')
                    ->color('secondary'),
            ];
        }

        $vault = Vault::forBranch($branchId);

        $from = now()->startOfDay();
        $to = now()->endOfDay();

        /**
         * نريد Top 4 حسب net
         * Invoice => بيع مباشر
         * وباقي المصادر: ClientReceipt, ...
         */
        $rows = VaultTransaction::query()
            ->selectRaw("
                CASE
                    WHEN source_type = ? THEN 'InvoicePos'
                    ELSE source_type
                END as bucket,
                SUM(CASE WHEN transaction_type = 1 THEN amount ELSE 0 END) AS deposits,
                SUM(CASE WHEN transaction_type = 2 THEN amount ELSE 0 END) AS withdrawals,
                SUM(CASE WHEN transaction_type = 1 THEN amount ELSE -amount END) AS net
            ", [
                \App\Models\Invoice::class,
            ])
            ->where('vault_id', $vault->id)
            ->whereBetween('created_at', [$from, $to])
            ->groupBy('bucket')
            ->get()
            ->sortByDesc(fn ($r) => (int) $r->net)
            ->take(4)
            ->values();

        $stats = [];

        foreach ($rows as $row) {
            $bucket = (string) $row->bucket;

            $info = $this->mapBucket($bucket);
            $net = (int) $row->net;
            $dep = (int) $row->deposits;
            $wd = (int) $row->withdrawals;

            $color = $net > 0 ? 'success' : ($net < 0 ? 'danger' : 'secondary');

            $stats[] = Stat::make($info['label'], $this->fmt($net))
                ->description('إيداع: '.$this->fmt($dep).' | سحب: '.$this->fmt($wd))
                ->descriptionIcon($net >= 0 ? 'heroicon-m-arrow-trending-up' : 'heroicon-m-arrow-trending-down')
                ->color($color);
        }

        if (empty($stats)) {
            $netToday = (int) VaultTransaction::query()
                ->where('vault_id', $vault->id)
                ->whereBetween('created_at', [$from, $to])
                ->selectRaw('COALESCE(SUM(CASE WHEN transaction_type=1 THEN amount ELSE -amount END),0) AS s')
                ->value('s');

            $stats[] = Stat::make('صافي اليوم', $this->fmt($netToday))
                ->description('لا توجد حركات اليوم')
                ->descriptionIcon($netToday >= 0 ? 'heroicon-m-arrow-trending-up' : 'heroicon-m-arrow-trending-down')
                ->color($netToday >= 0 ? 'success' : 'danger');
        }

        return $stats;
    }

    protected function fmt(int $amount): string
    {
        return number_format($amount).' د.ع';
    }

    /**
     * map للبكِت بعد التقسيم
     */
    protected function mapBucket(string $bucket): array
    {
        // Buckets الخاصة بـ Invoice
        if ($bucket === 'InvoicePos') {
            return ['label' => 'فواتير البيع', 'color' => 'success'];
        }

        // مصادر أخرى على شكل FQCN
        $type = class_basename($bucket);

        return match ($type) {
            'ClientReceipt' => ['label' => 'وصولات (سندات قبض)', 'color' => 'info'],
            'Expense' => ['label' => 'مصروفات', 'color' => 'danger'],
            'User' => ['label' => 'إدخال مباشر', 'color' => 'gray'],
            default => ['label' => ($type ?: 'غير معروف'), 'color' => 'secondary'],
        };
    }
}
