<?php

namespace App\Filament\Resources\VaultTransactions\Widgets;

use App\Models\Vault;
use App\Models\VaultTransaction;
use Filament\Widgets\StatsOverviewWidget as BaseWidget;
use Filament\Widgets\StatsOverviewWidget\Stat;
use Illuminate\Support\Facades\Auth;

class VaultOverviewStats extends BaseWidget
{
    protected ?string $pollingInterval = '10s';

    protected static bool $isLazy = false;

    protected function getStats(): array
    {
        $branchId = (int) (Auth::user()?->branch_id ?? 0);

        if (! $branchId) {
            return [
                Stat::make('رصيد الخزنة', '—')
                    ->description('لا يوجد فرع مرتبط بالمستخدم')
                    ->color('secondary'),
            ];
        }

        $vault = Vault::forBranch($branchId);
        $balance = (int) ($vault->balance ?? 0);

        $todayStart = now()->startOfDay();
        $todayEnd = now()->endOfDay();

        $todayDeposits = (int) VaultTransaction::query()
            ->where('vault_id', $vault->id)
            ->where('transaction_type', VaultTransaction::TYPE_DEPOSIT)
            ->whereBetween('created_at', [$todayStart, $todayEnd])
            ->sum('amount');

        $todayWithdrawals = (int) VaultTransaction::query()
            ->where('vault_id', $vault->id)
            ->where('transaction_type', VaultTransaction::TYPE_WITHDRAW)
            ->whereBetween('created_at', [$todayStart, $todayEnd])
            ->sum('amount');

        $spark = $this->netPerDay($vault->id, 14);

        return [
            Stat::make('رصيد الخزنة', $this->fmt($balance))
                ->description($this->trendDescription($spark))
                ->descriptionIcon($this->trendIcon($spark))
                ->chart($spark)
                ->color('success'),

            Stat::make('إيراد اليوم', $this->fmt($todayDeposits))
                ->description('مجموع الإيداعات اليوم')
                ->color('primary'),

            Stat::make('مصروف اليوم', $this->fmt($todayWithdrawals))
                ->description('مجموع السحوبات اليوم')
                ->color('danger'),
        ];
    }

    protected function netPerDay(int $vaultId, int $days = 14): array
    {
        $start = now()->copy()->startOfDay()->subDays($days - 1);

        $rows = VaultTransaction::query()
            ->selectRaw('
                DATE(created_at) as d,
                SUM(CASE WHEN transaction_type = ? THEN amount ELSE -amount END) as net
            ', [VaultTransaction::TYPE_DEPOSIT])
            ->where('vault_id', $vaultId)
            ->whereDate('created_at', '>=', $start->toDateString())
            ->groupBy('d')
            ->orderBy('d')
            ->get()
            ->keyBy('d');

        $data = [];
        for ($i = 0; $i < $days; $i++) {
            $day = $start->copy()->addDays($i)->toDateString();
            $data[] = (int) ($rows[$day]->net ?? 0);
        }

        return $data;
    }

    protected function fmt(int $amount): string
    {
        return number_format($amount).' د.ع';
    }

    protected function trendDescription(array $spark): string
    {
        if (count($spark) < 2) {
            return '—';
        }

        $last = (int) $spark[array_key_last($spark)];
        $prev = (int) ($spark[array_key_last($spark) - 1] ?? 0);
        $diff = $last - $prev;

        $abs = number_format(abs($diff));

        return $diff >= 0 ? "{$abs} زيادة" : "{$abs} انخفاض";
    }

    protected function trendIcon(array $spark): ?string
    {
        if (count($spark) < 2) {
            return null;
        }

        $last = (int) $spark[array_key_last($spark)];
        $prev = (int) ($spark[array_key_last($spark) - 1] ?? 0);

        return ($last - $prev) >= 0
            ? 'heroicon-m-arrow-trending-up'
            : 'heroicon-m-arrow-trending-down';
    }
}
