<?php

namespace App\Filament\Resources\VaultTransactions\Tables;

use App\Filament\Resources\ClientReceipts\ClientReceiptResource;
use App\Filament\Resources\Pos\PosResource;
use App\Models\ClientReceipt;
use App\Models\Invoice;
use App\Models\VaultTransaction;
// Resources (روابط التحرير)
use Filament\Actions\BulkActionGroup;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;

class VaultTransactionsTable
{
    /* ===================== Helpers ===================== */

    protected static function isDeposit($record): bool
    {
        return (int) $record->transaction_type === VaultTransaction::TYPE_DEPOSIT;
    }

    protected static function sourceTypeBase($record): string
    {
        $model = $record->source;

        return $model
            ? class_basename($model)
            : class_basename((string) ($record->source_type ?? ''));
    }

    /**
     * يحدد لابل/لون النشاط (إيداع/سحب) مع تخصيص للموظف
     */
    protected static function activityMeta($record): array
    {
        $isDeposit = static::isDeposit($record);
        $srcType = static::sourceTypeBase($record);

        if ($srcType === 'User') {
            return $isDeposit
                ? ['label' => 'إيداع من موظف', 'color' => 'success']
                : ['label' => 'سحب من موظف',   'color' => 'danger'];
        }

        return $isDeposit
            ? ['label' => 'إيداع', 'color' => 'success']
            : ['label' => 'سحب',   'color' => 'danger'];
    }

    /**
     * رقم المستند المعروض:
     * Invoice => invoice_number
     * ClientReceipt => receipt_number إن وجد وإلا id
     */
    protected static function sourceNumber($record): string
    {
        $model = $record->source;

        if ($model) {
            foreach (['invoice_number', 'contract_number', 'receipt_number', 'number', 'code', 'reference', 'no'] as $field) {
                if (isset($model->{$field}) && $model->{$field}) {
                    return (string) $model->{$field};
                }
            }
        }

        return '#'.(string) $record->getKey();
    }

    /**
     * ✅ جميع الفواتير تستخدم PosResource
     */
    protected static function invoiceResourceFor($invoice): string
    {
        return PosResource::class;
    }

    /**
     * رابط فتح المصدر (edit) إذا مدعوم
     */
    protected static function sourceUrl($record): ?string
    {
        $model = $record->source;
        $typeRaw = static::sourceTypeBase($record);
        $id = $model?->getKey() ?? $record->source_id;

        if (! $id) {
            return null;
        }

        // مصدر موظف/مباشر بدون مصدر
        if ($typeRaw === 'User' || $typeRaw === '') {
            return null;
        }

        // Invoice: يقرر حسب نوعها
        if ($typeRaw === 'Invoice' && $model) {
            $resource = static::invoiceResourceFor($model);

            return $resource::getUrl('edit', ['record' => $id]);
        }

        // ClientReceipt
        if ($typeRaw === 'ClientReceipt') {
            return ClientReceiptResource::getUrl('edit', ['record' => $id]);
        }

        // إذا عندك مصادر أخرى لاحقاً أضفها هنا
        return null;
    }

    /**
     * Badge يوضح "مصدر الحركة" بشكل ودّي
     */
    protected static function sourceInfo($record): array
    {
        $type = static::sourceTypeBase($record);

        return match ($type) {
            'Invoice' => ['label' => 'فاتورة',        'color' => 'success'],
            'ClientReceipt' => ['label' => 'سند قبض',       'color' => 'info'],
            'Expense' => ['label' => 'مصروفات',      'color' => 'danger'],
            'Order' => ['label' => 'حجز',          'color' => 'warning'],
            'User' => static::isDeposit($record)
                ? ['label' => 'إيداع من موظف', 'color' => 'success']
                : ['label' => 'سحب من موظف',   'color' => 'danger'],
            default => ['label' => ($type ?: 'غير معروف'), 'color' => 'secondary'],
        };
    }

    public static function configure(Table $table): Table
    {
        return $table
            // ✅ آخر عملية أولاً
            ->defaultSort('created_at', 'desc')

            ->columns([
                TextColumn::make('transaction_type')
                    ->label('النشاط المالي')
                    ->badge()
                    ->formatStateUsing(fn ($state) => ((int) $state === VaultTransaction::TYPE_DEPOSIT) ? 'إيداع' : 'سحب')
                    ->color(fn ($state) => ((int) $state === VaultTransaction::TYPE_DEPOSIT) ? 'success' : 'danger')
                    ->sortable(),

                // ✅ مصدر الحركة (فاتورة/عقد/سند قبض...) Badge
                TextColumn::make('source_kind')
                    ->label('المصدر')
                    ->badge()
                    ->state(fn ($record) => static::sourceInfo($record)['label'])
                    ->color(fn ($record) => static::sourceInfo($record)['color'])
                    ->toggleable(),

                TextColumn::make('activity')
                    ->label('نوع النشاط المالي')
                    ->badge()
                    ->state(fn ($record) => static::activityMeta($record)['label'])
                    ->color(fn ($record) => static::activityMeta($record)['color'])
                    ->sortable(query: function ($query, $direction) {
                        return $query->orderBy('transaction_type', $direction);
                    }),

                TextColumn::make('source_number')
                    ->label('رقم المستند')
                    ->state(fn ($record) => static::sourceNumber($record))
                    ->url(fn ($record) => static::sourceUrl($record))
                    ->openUrlInNewTab()
                    ->copyable(),

                TextColumn::make('description')
                    ->label('التفاصيل')
                    ->limit(40)
                    ->tooltip(fn ($state) => $state)
                    ->toggleable(),

                TextColumn::make('amount')
                    ->label('المبلغ')
                    ->formatStateUsing(fn ($state) => number_format((int) $state).' د.ع')
                    ->alignEnd()
                    ->weight('bold')
                    ->sortable(),

                TextColumn::make('balance_after')
                    ->label('الرصيد بعد الحركة')
                    ->formatStateUsing(fn ($state) => number_format((int) $state).' د.ع')
                    ->alignEnd()
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('user.name')
                    ->label('الموظف')
                    ->getStateUsing(fn ($record) => $record->user->name ?? (string) $record->user_id)
                    ->searchable()
                    ->sortable()
                    ->badge()
                    ->toggleable(),

                TextColumn::make('created_at')
                    ->label('التاريخ')
                    ->dateTime('Y-m-d h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('updated_at')
                    ->label('آخر تحديث')
                    ->dateTime('Y-m-d h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('transaction_type')
                    ->label('نوع الحركة')
                    ->options([
                        VaultTransaction::TYPE_DEPOSIT => 'إيداع',
                        VaultTransaction::TYPE_WITHDRAW => 'سحب',
                    ]),

                // فلترة حسب نوع المصدر
                SelectFilter::make('source_type')
                    ->label('نوع المصدر')
                    ->options([
                        Invoice::class => 'فاتورة',
                        ClientReceipt::class => 'سند قبض',
                    ]),

                Filter::make('created_between')
                    ->label('الفترة')
                    ->schema([
                        DatePicker::make('from')->label('من تاريخ')->native(false),
                        DatePicker::make('until')->label('إلى تاريخ')->native(false),
                    ])
                    ->query(function ($query, array $data) {
                        return $query
                            ->when($data['from'] ?? null, fn ($q, $date) => $q->whereDate('created_at', '>=', $date))
                            ->when($data['until'] ?? null, fn ($q, $date) => $q->whereDate('created_at', '<=', $date));
                    }),
            ])
            ->recordActions([])
            ->toolbarActions([
                BulkActionGroup::make([
                    // شاشة سجل مالي: عادة بلا حذف
                ]),
            ]);
    }
}
