<?php

namespace App\Filament\Resources\VaultTransactions\Pages;

use App\Filament\Resources\VaultTransactions\VaultTransactionResource;
use App\Filament\Resources\VaultTransactions\Widgets\VaultOverviewStats;
use App\Filament\Resources\VaultTransactions\Widgets\VaultSourcesToday;
use App\Models\BranchSetting;
use App\Models\VaultTransaction;
use Filament\Actions\Action;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\ListRecords;
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Hash;

class ListVaultTransactions extends ListRecords
{
    protected static string $resource = VaultTransactionResource::class;

    protected function getHeaderActions(): array
    {
        $amountField = function () {
            return TextInput::make('amount')
                ->label('المبلغ')
                ->required()
                ->formatStateUsing(fn ($state) => $state !== null && $state !== '' ? number_format((int) $state) : '')
                ->dehydrateStateUsing(fn ($state) => (int) preg_replace('/[^\d]/', '', (string) $state))
                ->extraInputAttributes([
                    'inputmode' => 'numeric',
                    'x-data' => '{}',
                    'x-on:input' => <<<'JS'
                        const raw = $el.value.replace(/[^\d]/g, '');
                        $el.value = raw ? new Intl.NumberFormat('en-US').format(raw) : '';
                    JS,
                ])
                ->helperText('المبلغ يُخزَّن كعدد صحيح (IQD).')
                ->columnSpanFull();
        };

        return [
            // ========= إيداع =========
            Action::make('deposit')
                ->label('إضافة مبلغ')
                ->icon('heroicon-o-banknotes')
                ->modalHeading('إضافة مبلغ إلى الخزنة')
                ->modalSubmitActionLabel('إضافة')
                ->schema([$amountField()])
                ->action(function (array $data) {
                    $user = user_info();
                    $vault = current_vault();

                    abort_unless($user && $vault, 422, 'لا يمكن تحديد خزنة للاشتراك الحالي.');

                    $tx = $vault->deposit(
                        source: $user,
                        amount: (int) $data['amount'],
                        description: 'إيداع مباشر من قبل: '.$user->name,
                        userId: $user->id
                    );

                    Notification::make()
                        ->title('تم الإيداع بنجاح')
                        ->body('الرصيد بعد العملية: '.number_format((int) $tx->balance_after).' د.ع')
                        ->success()
                        ->send();

                    $this->dispatch('refresh');
                })
                ->extraModalFooterActions([
                    Action::make('deposit_and_continue')
                        ->label('إضافة وبدء إضافة المزيد')
                        ->outlined()
                        ->action(function (array $data, Action $action) {
                            $user = user_info();
                            $vault = current_vault();

                            abort_unless($user && $vault, 422, 'لا يمكن تحديد خزنة للاشتراك الحالي.');

                            $tx = $vault->deposit(
                                source: $user,
                                amount: (int) $data['amount'],
                                description: 'إيداع مباشر من قبل: '.$user->name,
                                userId: $user->id
                            );

                            Notification::make()
                                ->title('تم الإيداع')
                                ->body('الرصيد بعد العملية: '.number_format((int) $tx->balance_after).' د.ع')
                                ->success()
                                ->send();

                            $action->halt();
                            $this->dispatch('fill-form', data: ['amount' => '']);
                            $this->dispatch('refresh');
                        }),
                ]),

            // ========= سحب =========
            Action::make('withdraw')
                ->label('سحب مبلغ')
                ->icon('heroicon-o-arrow-down-circle')
                ->color('danger')
                ->modalHeading('سحب مبلغ من الخزنة')
                ->modalSubmitActionLabel('سحب')
                ->schema([$amountField()->helperText('سيتم التحقق من كفاية الرصيد قبل السحب.')])
                ->action(function (array $data) {
                    $user = user_info();
                    $vault = current_vault();

                    abort_unless($user && $vault, 422, 'لا يمكن تحديد خزنة للاشتراك الحالي.');

                    try {
                        $tx = $vault->withdraw(
                            source: $user,
                            amount: (int) $data['amount'],
                            description: 'سحب مباشر من قبل: '.$user->name,
                            userId: $user->id
                        );

                        Notification::make()
                            ->title('تم السحب بنجاح')
                            ->body('الرصيد بعد العملية: '.number_format((int) $tx->balance_after).' د.ع')
                            ->success()
                            ->send();

                        $this->dispatch('refresh');
                    } catch (\Throwable $e) {
                        Notification::make()
                            ->title('فشل السحب')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();
                    }
                }),

            // ========= تصفير الخزنة =========
            Action::make('resetVault')
                ->label('تصفير الخزنة')
                ->color('danger')
                ->icon('heroicon-o-lock-closed')
                ->visible(fn () => user_info()?->hasAnyRole(['admin', 'branch_manager', 'security']))
                ->requiresConfirmation()
                ->modalHeading('تصفير الخزنة')
                ->modalDescription('سيتم تصفير رصيد الخزنة وتسجيل حركة (تصفير).')
                ->modalSubmitActionLabel('تصفير')
                ->schema([
                    TextInput::make('password')
                        ->label('كلمة سر تصفير الخزنة')
                        ->password()
                        ->revealable()
                        ->required(),
                ])
                ->action(function (array $data) {

                    // تحقق Role (Server-side)
                    if (! user_info()?->hasAnyRole(['admin', 'branch_manager', 'security'])) {
                        Notification::make()->title('غير مصرح لك بتنفيذ العملية')->danger()->send();

                        return;
                    }

                    $user = user_info();
                    $vault = current_vault();

                    abort_unless($user && $vault, 422, 'لا يمكن تحديد خزنة للاشتراك الحالي.');

                    $branchId = (int) user_info('branch_id');

                    // كلمة السر من إعدادات الفرع
                    $setting = BranchSetting::query()
                        ->where('branch_id', $branchId)
                        ->first();

                    if (! $setting || ! $setting->vault_reset_password) {
                        Notification::make()
                            ->title('لم يتم تعيين كلمة سر لتصفير الخزنة لهذا الفرع')
                            ->danger()
                            ->send();

                        return;
                    }

                    if (! Hash::check((string) $data['password'], (string) $setting->vault_reset_password)) {
                        Notification::make()
                            ->title('كلمة السر غير صحيحة')
                            ->danger()
                            ->send();

                        return;
                    }

                    // تنفيذ التصفير
                    DB::transaction(function () use ($vault, $user) {
                        $balance = (int) ($vault->balance ?? 0);

                        // إذا الرصيد 0 لا نسوي حركة
                        if ($balance === 0) {
                            return;
                        }

                        // تصفير الرصيد
                        $vault->update(['balance' => 0]);

                        // تسجيل حركة تصفير مع جميع الحقول المطلوبة
                        VaultTransaction::create([
                            'vault_id' => $vault->id,
                            'transaction_type' => VaultTransaction::TYPE_RESET,
                            'amount' => $balance,
                            'description' => 'تصفير الخزنة',
                            'user_id' => (int) $user->id,
                            'balance_after' => 0,
                            'source_type' => get_class($user),
                            'source_id' => (int) $user->getKey(),
                        ]);
                    });

                    Notification::make()
                        ->title('تم تصفير الخزنة بنجاح')
                        ->success()
                        ->send();

                    $this->dispatch('refresh');
                }),
        ];
    }

    protected function getHeaderWidgets(): array
    {
        return [
            VaultOverviewStats::class,
            VaultSourcesToday::class,
        ];
    }
}
