<?php

namespace App\Filament\Resources\Users\Tables;

use App\Enums\RoleEnum;
use Filament\Tables\Table;
use Filament\Actions\EditAction;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Spatie\Permission\Models\Role;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ViewColumn;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Contracts\Database\Eloquent\Builder;
use Filafly\IdentityColumn\Tables\Columns\IdentityColumn;

class UsersTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                // ViewColumn::make('name')
                //     ->label('المستخدم')
                //     ->sortable('name')
                //     ->view('filament.tables.columns.user-main'),

                IdentityColumn::make('username')
                    ->label('المستخدم')
                    ->avatar('avatar_url')
                    ->primary('username')
                    ->secondary('phone')
                    ->secondaryUrl('profile_url', openInNewTab: true),
                TextColumn::make('name')
                    ->label('الاسم الثلاثي')
                    ->searchable()
                    ->sortable(),
                TextColumn::make('email')
                    ->label('البريد الالكتروني')
                    ->searchable()
                    ->toggleable(isToggledHiddenByDefault: true),
                TextColumn::make('address')
                    ->label('العنوان')
                    ->searchable()
                    ->sortable()
                    ->limit(30),
                TextColumn::make('branch.name')
                    ->label('االفرع')
                    ->searchable()
                    ->visible(user_info()->hasRole('admin'))
                    ->sortable(),
                // الأدوار (Spatie)
                TextColumn::make('roles.name')
                    ->label('الدور')
                    ->badge()
                    ->separator(', ')
                    ->color(function ($state) {
                        $enum = RoleEnum::tryFrom($state);
                        return $enum?->getColor() ?? 'gray';
                    })
                    ->formatStateUsing(fn(?string $state) => RoleEnum::tryFromValue($state)?->getLabel())
                    ->icon(fn(?string $state) => \App\Enums\RoleEnum::tryFromValue($state)?->getIcon())
                    ->searchable()
                    ->sortable(),
                TextColumn::make('status')
                    ->label('الحالة')
                    ->badge()
                    ->formatStateUsing(fn($state) => (int) $state === 1 ? 'نشط' : 'متوقف')
                    ->colors([
                        'success' => fn($state) => (int) $state === 1,
                        'danger'  => fn($state) => (int) $state === 0,
                    ]),

                TextColumn::make('created_at')
                    ->sortable()
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d h:i A')
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('is_active')
                    ->label('الحالة')
                    ->options([1 => 'نشط', 0 => 'متوقف']),

                SelectFilter::make('role')
                    ->label('الدور')
                    ->multiple()                 // ← يسمح باختيار أكثر من دور
                    ->options(function () {
                        return Role::query()
                            ->where('name', '!=', 'admin')
                            ->pluck('name', 'name')
                            ->mapWithKeys(fn($name) => [$name => __('roles.' . $name)])
                            ->all();
                    })
                    ->searchable()
                    ->query(function (Builder $query, array $data) {
                        $values = (array) ($data['values'] ?? $data['value'] ?? []);
                        if ($values) {
                            $query->whereHas('roles', fn($q) => $q->whereIn('name', $values));
                        }
                    }),

            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make()
                        ->label('تعديل')
                        ->icon('heroicon-m-pencil')
                        ->color('gray'),

                    DeleteAction::make()
                        ->label('حذف')
                        ->icon('heroicon-m-trash')
                        ->color('gray')
                        ->requiresConfirmation(),
                ])
                    ->label('') // نخليها أيقونة فقط
                    ->icon('heroicon-m-ellipsis-vertical')
                    ->iconButton()
                    ->color('gray')
                    ->size('lg')
                    ->extraAttributes(['class' => 'fi-dd-trigger']),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }
}
