<?php

namespace App\Filament\Resources\Users\Schemas;

use App\Enums\RoleEnum;
use App\Models\User;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\ViewField;
use Filament\Schemas\Components\Group;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Illuminate\Support\Facades\Hash;

class UserForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('المعلومات الشخصية')
                    ->schema([
                        SpatieMediaLibraryFileUpload::make('avatar')
                            ->label('الصورة الشخصية')
                            ->collection('avatar')
                            ->disk('upload')
                            ->image()
                            ->avatar()
                            ->imageEditor()
                            ->openable()
                            ->downloadable()
                            ->maxFiles(1)
                            ->conversion('thumb')
                            ->columnSpanFull(),

                        TextInput::make('name')
                            ->label('الاسم الثلاثي')
                            ->required(),

                        TextInput::make('phone')
                            ->label('الهاتف')
                            ->tel()
                            ->extraAttributes(['inputmode' => 'numeric'])
                            ->minLength(11)
                            ->maxLength(11)
                            ->rule('digits:11')
                            ->dehydrateStateUsing(fn (?string $state) => $state ? preg_replace('/\D+/', '', $state) : null),

                        TextInput::make('username')
                            ->label('المعرف')
                            ->required()
                            ->unique(ignoreRecord: true),

                        TextInput::make('address')
                            ->label('عنوان السكن'),

                        // كلمة المرور: مطلوبة فقط في create، وفي edit اختيارية
                        Group::make()->schema([
                            TextInput::make('password')
                                ->label('كلمة المرور')
                                ->password()
                                ->revealable()
                                ->required(fn (string $operation) => $operation === 'create')
                                ->minLength(8)
                                ->rule('regex:/^(?=.*[a-z])(?=.*[A-Z])(?=.*\d).+$/')
                                ->helperText('8 أحرف على الأقل وتحتوي على حرف كبير وصغير ورقم.')
                                ->extraAttributes(['id' => 'user_password'])
                                // إذا كانت فارغة أثناء edit لا تغيّر كلمة المرور
                                ->dehydrateStateUsing(fn (?string $state) => filled($state) ? Hash::make($state) : null)
                                ->dehydrated(fn (?string $state) => filled($state)),

                            ViewField::make('password_meter')
                                ->view('filament.forms.password-strength-meter', [
                                    'targetId' => 'user_password',
                                ]),
                        ])->columns(1),

                        Select::make('roles')
                            ->label('الأدوار')
                            ->options(function () {
                                $opts = RoleEnum::options();

                                // غير الأدمن لا يشوف/يمنح دور admin
                                if (! user_info()?->hasRole(RoleEnum::admin->value)) {
                                    unset($opts[RoleEnum::admin->value]);
                                }

                                return $opts;
                            })
                            ->multiple()
                            ->searchable()
                            ->preload()
                            ->native(false)
                            ->afterStateHydrated(function (Select $component, $state, ?User $record) {
                                $roles = $record?->getRoleNames()->toArray() ?? [];

                                if (! user_info()?->hasRole(RoleEnum::admin->value)) {
                                    $roles = array_values(array_filter(
                                        $roles,
                                        fn ($r) => $r !== RoleEnum::admin->value
                                    ));
                                }

                                $component->state($roles);
                            }),

                        Select::make('status')
                            ->label('الحالة')
                            ->default(true)
                            ->options([1 => 'نشط', 0 => 'غير نشط'])
                            ->visibleOn('edit')
                            ->required(fn (string $operation): bool => $operation === 'edit')
                            ->afterStateHydrated(fn (Select $component, $state) => $component->state((int) $state)),
                    ])
                    ->columnSpanFull()
                    ->columns(2),
            ]);
    }
}
