<?php

namespace App\Filament\Resources\Suppliers\Pages;

use App\Filament\Resources\Suppliers\SupplierResource;
use App\Models\PurchaseInvoice;
use App\Models\PurchaseReturn;
use App\Models\Supplier;
use App\Models\SupplierLedger;
use App\Models\SupplierPayment;
use Filament\Resources\Pages\ViewRecord;

class ViewSupplier extends ViewRecord
{
    protected static string $resource = SupplierResource::class;

    /**
     * استخدام ملف Blade مخصص:
     * resources/views/filament/suppliers/view.blade.php
     */
    protected string $view = 'filament.suppliers.view';

    /**
     * تمرير بيانات إضافية للـ Blade.
     */
    protected function getViewData(): array
    {
        /** @var Supplier $supplier */
        $supplier = $this->record;

        /*=========================================
         = إحصائيات فواتير الشراء
         =========================================*/
        $invoiceStats = PurchaseInvoice::query()
            ->where('supplier_id', $supplier->id)
            ->whereNull('deleted_at')
            ->selectRaw('
                COUNT(*) AS invoices_count,
                COALESCE(SUM(total), 0)  AS total,
                COALESCE(SUM(paid), 0)   AS paid,
                COALESCE(SUM(due), 0)    AS due
            ')
            ->first();

        /*=========================================
         = إحصائيات المدفوعات
         =========================================*/
        $paymentStats = SupplierPayment::query()
            ->where('supplier_id', $supplier->id)
            ->whereNull('deleted_at')
            ->selectRaw('
                COUNT(*) AS payments_count,
                COALESCE(SUM(amount), 0) AS total_paid
            ')
            ->first();

        // آخر دفعة فعلية
        $lastPayment = SupplierPayment::query()
            ->where('supplier_id', $supplier->id)
            ->whereNull('deleted_at')
            ->latest('payment_date')
            ->latest('id')
            ->first();

        /*=========================================
         = إحصائيات المرتجعات
         =========================================*/
        $returnsStats = PurchaseReturn::query()
            ->where('supplier_id', $supplier->id)
            ->whereNull('deleted_at')
            ->selectRaw('
                COUNT(*) AS returns_count,
                COALESCE(SUM(total), 0) AS total_returns
            ')
            ->first();

        $returnsCount = (int) ($returnsStats->returns_count ?? 0);
        $returnsTotal = (int) ($returnsStats->total_returns ?? 0);
        $returnsAvg   = $returnsCount > 0 ? (int) floor($returnsTotal / $returnsCount) : 0;

        $returnsSummary = [
            'count' => $returnsCount,
            'total' => $returnsTotal,
            'avg'   => $returnsAvg,
        ];

        /*=========================================
         = دفتر المورد (مدين / دائن / رصيد)
         =========================================*/
        // هنا نعتبر:
        // - purchase            => مدين (يزيد ما ندين به للمورد)
        // - payment             => دائن (يقلل الدين)
        // - purchase_return     => دائن (مرتجعات تقلل الدين)
        // - debit_adjustment    => مدين (تعديل يرفع الدين)    [لو استخدمته لاحقاً]
        // - credit_adjustment   => دائن (تعديل يخفض الدين)   [لو استخدمته لاحقاً]
        $ledgerStats = SupplierLedger::query()
            ->where('supplier_id', $supplier->id)
            ->selectRaw("
                COALESCE(SUM(CASE
                    WHEN type IN ('purchase', 'debit_adjustment') THEN amount
                    ELSE 0
                END), 0) AS debits,
                COALESCE(SUM(CASE
                    WHEN type IN ('payment', 'purchase_return', 'credit_adjustment') THEN amount
                    ELSE 0
                END), 0) AS credits
            ")
            ->first();

        $debits  = (int) ($ledgerStats->debits ?? 0);
        $credits = (int) ($ledgerStats->credits ?? 0);

        // الرصيد = مجموع المدين - مجموع الدائن
        $balance = $debits - $credits;

        // نخزن الرصيد في الكائن حتى يقرأه الـ Blade:
        $ledgerStats->balance = $balance;

        return [
            'supplier'       => $supplier,
            'invoiceStats'   => $invoiceStats,
            'paymentStats'   => $paymentStats,
            'returnsStats'   => $returnsStats,
            'returnsSummary' => $returnsSummary,
            'ledgerStats'    => $ledgerStats,
            'lastPayment'    => $lastPayment,
        ];
    }

    /**
     * عنوان الصفحة (اختياري).
     */
    public function getTitle(): string
    {
        /** @var Supplier $supplier */
        $supplier = $this->record;

        return 'عرض ' . $supplier->name;
    }
}
