<?php

namespace App\Filament\Resources\StoreSettings\Schemas;

use App\Models\StoreSetting;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Tabs;
use Filament\Schemas\Components\Tabs\Tab;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Schema;

class StoreSettingForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            Tabs::make('storeSettingsTabs')
                ->tabs([
                    self::tabBasicInfo(),
                    self::tabShipping(),
                    self::tabPayment(),
                    self::tabInventory(),
                    self::tabOrders(),
                    self::tabPromotions(),
                    self::tabSeo(),
                ])
                ->persistTabInQueryString()
                ->columnSpanFull(),
        ]);
    }

    /* ================== Helpers ================== */

    protected static function moneyInput(string $name, string $label, bool $required = false): TextInput
    {
        $input = TextInput::make($name)
            ->label($label)
            ->numeric()
            ->default(0)
            ->suffix('د.ع')
            ->minValue(0);

        return $required ? $input->required() : $input;
    }

    /* ================== التبويبات ================== */

    protected static function tabBasicInfo(): Tab
    {
        return Tab::make('المعلومات الأساسية')
            ->icon('heroicon-o-building-storefront')
            ->schema([
                Section::make('معلومات المتجر')
                    ->description('البيانات الأساسية للمتجر الإلكتروني')
                    ->schema([
                        Grid::make(2)->schema([
                            TextInput::make('store_name')
                                ->label('اسم المتجر')
                                ->required()
                                ->maxLength(255)
                                ->placeholder('مثال: متجري الإلكتروني')
                                ->prefixIcon('heroicon-o-building-storefront'),

                            Textarea::make('description')
                                ->label('وصف المتجر')
                                ->rows(2)
                                ->placeholder('وصف مختصر للمتجر...')
                                ->columnSpanFull(),
                        ]),
                    ]),

                Section::make('العملة')
                    ->schema([
                        Select::make('default_currency_id')
                            ->label('العملة الافتراضية')
                            ->relationship('defaultCurrency', 'name')
                            ->searchable()
                            ->preload()
                            ->prefixIcon('heroicon-o-currency-dollar'),
                    ]),

                Section::make('الصور والهوية')
                    ->schema([
                        Grid::make(3)->schema([
                            SpatieMediaLibraryFileUpload::make('store_logo')
                                ->label('شعار المتجر')
                                ->collection(StoreSetting::MEDIA_LOGO)
                                ->image()
                                ->imageEditor()
                                ->directory('store')
                                ->disk('upload')
                                ->maxSize(2048),

                            SpatieMediaLibraryFileUpload::make('store_favicon')
                                ->label('أيقونة المتجر')
                                ->collection(StoreSetting::MEDIA_FAVICON)
                                ->image()
                                ->imageEditor()
                                ->imageCropAspectRatio('1:1')
                                ->directory('store')
                                ->disk('upload')
                                ->maxSize(1024),

                            SpatieMediaLibraryFileUpload::make('store_banner')
                                ->label('بانر المتجر')
                                ->collection(StoreSetting::MEDIA_BANNER)
                                ->image()
                                ->imageEditor()
                                ->directory('store')
                                ->disk('upload')
                                ->maxSize(5120),
                        ]),

                    ]),

                Toggle::make('is_active')
                    ->label('المتجر مفعّل')
                    ->default(true)
                    ->inline(false),
            ]);
    }

    protected static function tabShipping(): Tab
    {
        return Tab::make('الشحن والتوصيل')
            ->icon('heroicon-o-truck')
            ->schema([
                Section::make('إعدادات الشحن')
                    ->schema([
                        Grid::make(2)->schema([
                            Toggle::make('enable_shipping')
                                ->label('تفعيل الشحن')
                                ->default(true)
                                ->live()
                                ->inline(false),

                            self::moneyInput('default_shipping_fee', 'رسوم الشحن الافتراضية')
                                ->visible(fn (Get $get) => $get('enable_shipping')),

                            self::moneyInput('free_shipping_threshold', 'حد الشحن المجاني')
                                ->helperText('اترك فارغاً لتعطيل الشحن المجاني')
                                ->visible(fn (Get $get) => $get('enable_shipping')),
                        ]),
                    ]),
            ]);
    }

    protected static function tabPayment(): Tab
    {
        return Tab::make('الدفع')
            ->icon('heroicon-o-credit-card')
            ->schema([
                Section::make('طرق الدفع')
                    ->schema([
                        Grid::make(2)->schema([
                            Select::make('payment_methods')
                                ->label('طرق الدفع المتاحة')
                                ->multiple()
                                ->options([
                                    'cod' => 'الدفع عند الاستلام',
                                    'online' => 'الدفع الإلكتروني',
                                    'transfer' => 'التحويل البنكي',
                                    'wallet' => 'المحفظة الإلكترونية',
                                ])
                                ->default(['cod']),

                            Toggle::make('enable_cod')
                                ->label('تفعيل الدفع عند الاستلام')
                                ->default(true)
                                ->live()
                                ->inline(false),

                            self::moneyInput('cod_fee', 'رسوم الدفع عند الاستلام')
                                ->visible(fn (Get $get) => $get('enable_cod')),
                        ]),
                    ]),
            ]);
    }

    protected static function tabInventory(): Tab
    {
        return Tab::make('المخزون')
            ->icon('heroicon-o-archive-box')
            ->schema([
                Section::make('إعدادات المخزون')
                    ->schema([
                        Grid::make(2)->schema([
                            Toggle::make('track_inventory')
                                ->label('تتبع المخزون')
                                ->default(true)
                                ->helperText('تفعيل تتبع الكميات المتوفرة')
                                ->inline(false),

                            Toggle::make('allow_backorders')
                                ->label('السماح بالطلب عند نفاد المخزون')
                                ->default(false)
                                ->inline(false),

                            TextInput::make('low_stock_threshold')
                                ->label('حد التنبيه للمخزون المنخفض')
                                ->numeric()
                                ->default(5)
                                ->minValue(0)
                                ->suffix('قطعة'),
                        ]),
                    ]),
            ]);
    }

    protected static function tabOrders(): Tab
    {
        return Tab::make('الطلبات')
            ->icon('heroicon-o-shopping-cart')
            ->schema([
                Section::make('إعدادات الطلبات')
                    ->schema([
                        Grid::make(2)->schema([
                            self::moneyInput('min_order_amount', 'الحد الأدنى للطلب')
                                ->helperText('اترك 0 لعدم وجود حد أدنى'),

                            TextInput::make('max_order_items')
                                ->label('الحد الأقصى لعناصر الطلب')
                                ->numeric()
                                ->minValue(1)
                                ->placeholder('غير محدود')
                                ->helperText('اترك فارغاً لعدم وجود حد'),

                            Toggle::make('require_phone')
                                ->label('رقم الهاتف مطلوب')
                                ->default(true)
                                ->inline(false),

                            Toggle::make('require_address')
                                ->label('العنوان مطلوب')
                                ->default(true)
                                ->inline(false),
                        ]),
                    ]),
            ]);
    }

    protected static function tabPromotions(): Tab
    {
        return Tab::make('العروض والخصومات')
            ->icon('heroicon-o-gift')
            ->schema([
                Section::make('إعدادات العروض')
                    ->schema([
                        Grid::make(2)->schema([
                            Toggle::make('enable_promotions')
                                ->label('تفعيل العروض')
                                ->default(true)
                                ->inline(false),

                            Toggle::make('enable_coupons')
                                ->label('تفعيل الكوبونات')
                                ->default(true)
                                ->inline(false),

                            Toggle::make('allow_multiple_coupons')
                                ->label('السماح بأكثر من كوبون')
                                ->default(false)
                                ->helperText('السماح للعميل باستخدام أكثر من كوبون في نفس الطلب')
                                ->inline(false),
                        ]),
                    ]),
            ]);
    }

    protected static function tabSeo(): Tab
    {
        return Tab::make('SEO')
            ->icon('heroicon-o-magnifying-glass')
            ->schema([
                Section::make('تحسين محركات البحث')
                    ->schema([
                        TextInput::make('meta_title')
                            ->label('عنوان الصفحة (Meta Title)')
                            ->maxLength(70)
                            ->helperText('يُفضل أن يكون بين 50-60 حرفاً'),

                        Textarea::make('meta_description')
                            ->label('وصف الصفحة (Meta Description)')
                            ->rows(2)
                            ->maxLength(160)
                            ->helperText('يُفضل أن يكون بين 120-160 حرفاً'),

                        TextInput::make('meta_keywords_input')
                            ->label('الكلمات المفتاحية')
                            ->placeholder('كلمة1, كلمة2, كلمة3')
                            ->helperText('افصل بين الكلمات بفاصلة')
                            ->afterStateHydrated(function ($component, $record) {
                                if ($record && $record->meta_keywords) {
                                    $component->state(implode(', ', $record->meta_keywords));
                                }
                            })
                            ->dehydrateStateUsing(function ($state) {
                                if (! $state) {
                                    return null;
                                }

                                return array_map('trim', explode(',', $state));
                            }),
                    ]),
            ]);
    }
}
