<?php

namespace App\Filament\Resources\Stickers\Tables;

use App\Models\Sticker;
use Filament\Actions\ActionGroup;
use Filament\Actions\BulkAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ForceDeleteAction;
use Filament\Actions\RestoreAction;
use Filament\Support\Icons\Heroicon;
use Filament\Tables\Columns\SpatieMediaLibraryImageColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Collection;

class StickersTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                SpatieMediaLibraryImageColumn::make('sticker_image')
                    ->label('الصورة')
                    ->collection(Sticker::MEDIA_COLLECTION_IMAGE)
                    ->circular()
                    ->defaultImageUrl(url('/images/placeholder.png'))
                    ->imageSize(50),

                TextColumn::make('name')
                    ->label('اسم الملصق')
                    ->searchable()
                    ->sortable()
                    ->weight('bold'),

                TextColumn::make('client.name')
                    ->label('المصمم')
                    ->searchable()
                    ->sortable(),

                TextColumn::make('category')
                    ->label('التصنيف')
                    ->sortable()
                    ->badge()
                    ->color('gray'),

                TextColumn::make('status')
                    ->label('الحالة')
                    ->sortable()
                    ->badge()
                    ->formatStateUsing(fn (string $state): string => match ($state) {
                        'pending' => 'قيد المراجعة',
                        'approved' => 'معتمد',
                        'rejected' => 'مرفوض',
                        default => $state,
                    })
                    ->color(fn (string $state): string => match ($state) {
                        'pending' => 'warning',
                        'approved' => 'success',
                        'rejected' => 'danger',
                        default => 'gray',
                    }),

                TextColumn::make('usage_count')
                    ->label('الاستخدامات')
                    ->sortable()
                    ->color('gray')
                    ->icon(Heroicon::OutlinedCursorArrowRays),

                TextColumn::make('created_at')
                    ->label('تاريخ الرفع')
                    ->dateTime('Y-m-d h:i A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('status')
                    ->label('الحالة')
                    ->options([
                        Sticker::STATUS_PENDING => 'قيد المراجعة',
                        Sticker::STATUS_APPROVED => 'معتمد',
                        Sticker::STATUS_REJECTED => 'مرفوض',
                    ]),

                SelectFilter::make('category')
                    ->label('التصنيف')
                    ->options([
                        'زخارف' => 'زخارف',
                        'نصوص' => 'نصوص',
                        'شعارات' => 'شعارات',
                        'أنماط' => 'أنماط',
                        'رموز' => 'رموز',
                        'أخرى' => 'أخرى',
                    ]),

                TrashedFilter::make(),
            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make(),
                    \Filament\Actions\Action::make('approve')
                        ->label('اعتماد')
                        ->icon(Heroicon::OutlinedCheck)
                        ->color('success')
                        ->requiresConfirmation()
                        ->modalHeading('اعتماد الملصق')
                        ->modalDescription('هل أنت متأكد من اعتماد هذا الملصق؟')
                        ->action(fn (Sticker $record) => $record->approve())
                        ->visible(fn (Sticker $record): bool => $record->isPending()),
                    \Filament\Actions\Action::make('reject')
                        ->label('رفض')
                        ->icon(Heroicon::OutlinedXMark)
                        ->color('danger')
                        ->requiresConfirmation()
                        ->modalHeading('رفض الملصق')
                        ->modalDescription('يرجى ذكر سبب الرفض')
                        ->form([
                            \Filament\Forms\Components\Textarea::make('rejection_reason')
                                ->label('سبب الرفض')
                                ->required()
                                ->maxLength(500),
                        ])
                        ->action(fn (Sticker $record, array $data) => $record->reject($data['rejection_reason']))
                        ->visible(fn (Sticker $record): bool => $record->isPending()),
                    DeleteAction::make(),
                    RestoreAction::make(),
                    ForceDeleteAction::make(),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    BulkAction::make('approveSelected')
                        ->label('اعتماد المحددة')
                        ->icon(Heroicon::OutlinedCheck)
                        ->color('success')
                        ->requiresConfirmation()
                        ->action(fn (Collection $records) => $records->each->approve()),
                    BulkAction::make('rejectSelected')
                        ->label('رفض المحددة')
                        ->icon(Heroicon::OutlinedXMark)
                        ->color('danger')
                        ->requiresConfirmation()
                        ->action(fn (Collection $records) => $records->each(fn ($r) => $r->reject('رفض جماعي'))),
                    DeleteBulkAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف الملصقات المحددة'),
                ]),
            ])
            ->defaultSort('created_at', 'desc');
    }
}
