<?php

namespace App\Filament\Resources\Stickers\Schemas;

use App\Models\Sticker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;
use Filament\Support\Icons\Heroicon;

class StickerForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('معلومات الملصق')
                    ->description('المعلومات الأساسية للملصق')
                    ->icon(Heroicon::OutlinedSparkles)
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                TextInput::make('name')
                                    ->label('اسم الملصق')
                                    ->required()
                                    ->maxLength(255)
                                    ->placeholder('مثال: زخرفة إسلامية')
                                    ->columnSpan(1),

                                Select::make('client_id')
                                    ->label('المصمم')
                                    ->relationship('client', 'name', fn ($query) => $query->where('client_type', 'designer'))
                                    ->searchable()
                                    ->preload()
                                    ->required()
                                    ->columnSpan(1),

                                Select::make('category')
                                    ->label('التصنيف')
                                    ->options([
                                        'زخارف' => 'زخارف',
                                        'نصوص' => 'نصوص',
                                        'شعارات' => 'شعارات',
                                        'أنماط' => 'أنماط',
                                        'رموز' => 'رموز',
                                        'أخرى' => 'أخرى',
                                    ])
                                    ->searchable()
                                    ->columnSpan(1),

                                Select::make('status')
                                    ->label('الحالة')
                                    ->options([
                                        Sticker::STATUS_PENDING => 'قيد المراجعة',
                                        Sticker::STATUS_APPROVED => 'معتمد',
                                        Sticker::STATUS_REJECTED => 'مرفوض',
                                    ])
                                    ->default(Sticker::STATUS_PENDING)
                                    ->required()
                                    ->columnSpan(1),

                                Textarea::make('description')
                                    ->label('الوصف')
                                    ->rows(3)
                                    ->maxLength(500)
                                    ->placeholder('وصف مختصر للملصق...')
                                    ->columnSpan(2),

                                Textarea::make('rejection_reason')
                                    ->label('سبب الرفض')
                                    ->rows(2)
                                    ->maxLength(500)
                                    ->placeholder('سبب رفض الملصق...')
                                    ->visible(fn ($get) => $get('status') === Sticker::STATUS_REJECTED)
                                    ->columnSpan(2),
                            ]),
                    ]),

                Section::make('صورة الملصق')
                    ->description('رفع صورة الملصق (PNG شفاف)')
                    ->icon(Heroicon::OutlinedPhoto)
                    ->schema([
                        SpatieMediaLibraryFileUpload::make('sticker_image')
                            ->label('صورة الملصق')
                            ->collection(Sticker::MEDIA_COLLECTION_IMAGE)
                            ->image()
                            ->imageEditor()
                            ->maxSize(5120)
                            ->required()
                            ->helperText('يفضل صورة PNG بخلفية شفافة (الحد الأقصى: 5MB)')
                            ->downloadable()
                            ->openable()
                            ->deletable(),
                    ]),

                Section::make('الإعدادات')
                    ->description('إعدادات الملصق')
                    ->icon(Heroicon::OutlinedCog)
                    ->schema([
                        Grid::make(2)
                            ->schema([
                                Toggle::make('is_active')
                                    ->label('نشط')
                                    ->helperText('إظهار الملصق في المكتبة')
                                    ->default(true)
                                    ->columnSpan(1),

                                TextInput::make('usage_count')
                                    ->label('عدد الاستخدامات')
                                    ->numeric()
                                    ->default(0)
                                    ->disabled()
                                    ->dehydrated(false)
                                    ->columnSpan(1),
                            ]),
                    ])
                    ->collapsible(),
            ]);
    }
}
