<?php

namespace App\Filament\Resources\ShippingCompanies\Tables;

use App\Models\ShippingCompany;
use Filament\Actions\ActionGroup;
use Filament\Actions\BulkAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;

class ShippingCompaniesTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->label('اسم الشركة')
                    ->searchable()
                    ->sortable()
                    ->weight('bold')
                    ->description(fn (ShippingCompany $record) => $record->code),

                TextColumn::make('phone')
                    ->label('الهاتف')
                    ->searchable()
                    ->icon('heroicon-o-phone'),

                TextColumn::make('email')
                    ->label('البريد الإلكتروني')
                    ->searchable()
                    ->icon('heroicon-o-envelope')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('contact_person')
                    ->label('المسؤول')
                    ->searchable()
                    ->toggleable(),

                TextColumn::make('rates_count')
                    ->label('المناطق')
                    ->counts('rates')
                    ->badge()
                    ->color('info')
                    ->alignCenter(),

                TextColumn::make('pending_shipments_count')
                    ->label('شحنات قيد التوصيل')
                    ->state(fn (ShippingCompany $record) => $record->pending_shipments_count)
                    ->badge()
                    ->color(fn ($state) => $state > 0 ? 'warning' : 'gray')
                    ->alignCenter(),

                TextColumn::make('delivered_shipments_count')
                    ->label('شحنات تم توصيلها')
                    ->state(fn (ShippingCompany $record) => $record->delivered_shipments_count)
                    ->badge()
                    ->color('success')
                    ->alignCenter()
                    ->toggleable(isToggledHiddenByDefault: true),

                IconColumn::make('is_active')
                    ->label('مفعّل')
                    ->boolean()
                    ->alignCenter()
                    ->sortable(),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                TernaryFilter::make('is_active')
                    ->label('الحالة')
                    ->placeholder('الكل')
                    ->trueLabel('مفعّل')
                    ->falseLabel('معطّل'),
            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make(),
                    DeleteAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف شركة الشحن')
                        ->modalDescription('هل أنت متأكد من حذف هذه الشركة؟'),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف الشركات المحددة')
                        ->modalDescription('هل أنت متأكد من حذف الشركات المحددة؟'),

                    BulkAction::make('activate')
                        ->label('تفعيل')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->requiresConfirmation()
                        ->action(fn ($records) => $records->each->update(['is_active' => true]))
                        ->deselectRecordsAfterCompletion(),

                    BulkAction::make('deactivate')
                        ->label('إلغاء التفعيل')
                        ->icon('heroicon-o-x-circle')
                        ->color('danger')
                        ->requiresConfirmation()
                        ->action(fn ($records) => $records->each->update(['is_active' => false]))
                        ->deselectRecordsAfterCompletion(),
                ]),
            ])
            ->defaultSort('name', 'asc')
            ->emptyStateHeading('لا توجد شركات شحن')
            ->emptyStateDescription('ابدأ بإضافة شركة شحن جديدة')
            ->emptyStateIcon('heroicon-o-truck')
            ->striped();
    }
}
