<?php

namespace App\Filament\Resources\ShippingCompanies\Schemas;

use App\Models\City;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;

class ShippingCompanyForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            self::sectionBasicInfo(),
            self::sectionContact(),
            self::sectionRates(),
        ]);
    }

    /* ================== المعلومات الأساسية ================== */

    protected static function sectionBasicInfo(): Section
    {
        return Section::make('معلومات الشركة')
            ->description('البيانات الأساسية لشركة الشحن')
            ->icon('heroicon-o-truck')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(2)->schema([
                    TextInput::make('name')
                        ->label('اسم الشركة')
                        ->required()
                        ->maxLength(255)
                        ->placeholder('مثال: شركة التوصيل السريع')
                        ->prefixIcon('heroicon-o-building-office'),

                    TextInput::make('code')
                        ->label('كود الشركة')
                        ->maxLength(50)
                        ->placeholder('مثال: FAST')
                        ->helperText('كود مختصر للشركة')
                        ->prefixIcon('heroicon-o-hashtag'),

                    Textarea::make('address')
                        ->label('العنوان')
                        ->rows(2)
                        ->placeholder('عنوان مقر الشركة...')
                        ->columnSpanFull(),

                    Textarea::make('notes')
                        ->label('ملاحظات')
                        ->rows(2)
                        ->placeholder('ملاحظات إضافية...')
                        ->columnSpanFull(),

                    Toggle::make('is_active')
                        ->label('مفعّل')
                        ->default(true)
                        ->inline(false),
                ]),
            ]);
    }

    /* ================== معلومات الاتصال ================== */

    protected static function sectionContact(): Section
    {
        return Section::make('معلومات الاتصال')
            ->description('بيانات التواصل مع الشركة')
            ->icon('heroicon-o-phone')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    TextInput::make('phone')
                        ->label('رقم الهاتف')
                        ->tel()
                        ->maxLength(20)
                        ->prefixIcon('heroicon-o-phone'),

                    TextInput::make('email')
                        ->label('البريد الإلكتروني')
                        ->email()
                        ->maxLength(255)
                        ->prefixIcon('heroicon-o-envelope'),

                    TextInput::make('contact_person')
                        ->label('الشخص المسؤول')
                        ->maxLength(255)
                        ->placeholder('اسم المسؤول')
                        ->prefixIcon('heroicon-o-user'),
                ]),
            ]);
    }

    /* ================== أسعار الشحن ================== */

    protected static function sectionRates(): Section
    {
        return Section::make('أسعار الشحن')
            ->description('تحديد أسعار الشحن لكل مدينة')
            ->icon('heroicon-o-banknotes')
            ->collapsible()
            ->collapsed()
            ->columnSpanFull()
            ->schema([
                Repeater::make('rates')
                    ->relationship()
                    ->label('أسعار الشحن حسب المدينة')
                    ->hiddenLabel()
                    ->collapsible()
                    ->defaultItems(0)
                    ->addActionLabel('+ إضافة سعر')
                    ->itemLabel(fn (array $state): ?string => City::find($state['city_id'] ?? null)?->name ?? 'مدينة جديدة')
                    ->schema([
                        Grid::make(4)->schema([
                            Select::make('city_id')
                                ->label('المدينة')
                                ->options(fn () => City::query()->orderBy('name')->pluck('name', 'id'))
                                ->required()
                                ->searchable()
                                ->preload()
                                ->prefixIcon('heroicon-o-map-pin'),

                            TextInput::make('shipping_fee')
                                ->label('رسوم الشحن')
                                ->numeric()
                                ->required()
                                ->default(0)
                                ->suffix('د.ع')
                                ->prefixIcon('heroicon-o-banknotes'),

                            TextInput::make('delivery_days')
                                ->label('أيام التوصيل')
                                ->numeric()
                                ->minValue(1)
                                ->placeholder('1-3')
                                ->helperText('عدد الأيام المتوقعة')
                                ->prefixIcon('heroicon-o-clock'),

                            Toggle::make('is_active')
                                ->label('مفعّل')
                                ->default(true)
                                ->inline(false),
                        ]),
                    ]),
            ]);
    }
}
