<?php

namespace App\Filament\Resources\Shipments\Tables;

use App\Enums\ShipmentStatus;
use App\Models\Shipment;
use Coolsam\Flatpickr\Forms\Components\Flatpickr;
use Filament\Actions\ActionGroup;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;

class ShipmentsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('tracking_number')
                    ->label('رقم التتبع')
                    ->searchable()
                    ->sortable()
                    ->weight('bold')
                    ->copyable()
                    ->copyMessage('تم نسخ رقم التتبع')
                    ->placeholder('-'),

                TextColumn::make('order.order_number')
                    ->label('رقم الطلب')
                    ->searchable()
                    ->sortable()
                    ->badge()
                    ->color('primary')
                    ->url(fn (Shipment $record) => route('filament.admin.resources.orders.edit', $record->order_id)),

                TextColumn::make('shippingCompany.name')
                    ->label('شركة الشحن')
                    ->searchable()
                    ->sortable()
                    ->badge()
                    ->color('info'),

                TextColumn::make('status')
                    ->label('الحالة')
                    ->badge()
                    ->formatStateUsing(fn (ShipmentStatus $state) => $state->label())
                    ->color(fn (ShipmentStatus $state) => $state->color())
                    ->icon(fn (ShipmentStatus $state) => $state->icon())
                    ->sortable(),

                TextColumn::make('cod_amount')
                    ->label('مبلغ COD')
                    ->formatStateUsing(fn ($state) => $state > 0 ? number_format($state, 0, '', ',').' د.ع' : '-')
                    ->badge()
                    ->color(fn ($state) => $state > 0 ? 'warning' : 'gray')
                    ->alignEnd(),

                IconColumn::make('cod_collected')
                    ->label('تم التحصيل')
                    ->boolean()
                    ->visible(fn (Shipment $record) => $record->cod_amount > 0)
                    ->alignCenter(),

                TextColumn::make('delivery_attempts')
                    ->label('المحاولات')
                    ->badge()
                    ->color(fn ($state) => match (true) {
                        $state === 0 => 'gray',
                        $state <= 2 => 'warning',
                        default => 'danger',
                    })
                    ->alignCenter(),

                TextColumn::make('shipped_at')
                    ->label('تاريخ الشحن')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->placeholder('-')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('delivered_at')
                    ->label('تاريخ التوصيل')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->placeholder('-')
                    ->toggleable(),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('status')
                    ->label('الحالة')
                    ->options(ShipmentStatus::options())
                    ->multiple(),

                SelectFilter::make('shipping_company_id')
                    ->label('شركة الشحن')
                    ->relationship('shippingCompany', 'name')
                    ->searchable()
                    ->preload()
                    ->multiple(),

                TernaryFilter::make('cod_collected')
                    ->label('تحصيل COD')
                    ->placeholder('الكل')
                    ->trueLabel('تم التحصيل')
                    ->falseLabel('لم يتم التحصيل')
                    ->queries(
                        true: fn (Builder $query) => $query->where('cod_collected', true),
                        false: fn (Builder $query) => $query->where('cod_amount', '>', 0)->where('cod_collected', false),
                    ),

                Filter::make('pending')
                    ->label('قيد التوصيل')
                    ->query(fn (Builder $query) => $query->pending()),

                Filter::make('today')
                    ->label('اليوم')
                    ->query(fn (Builder $query) => $query->whereDate('created_at', today())),

                Filter::make('date_range')
                    ->form([
                        Flatpickr::make('from_date')
                            ->label('من تاريخ')
                            ->displayFormat('Y-m-d')
                            ->native(false),
                        Flatpickr::make('to_date')
                            ->label('إلى تاريخ')
                            ->displayFormat('Y-m-d')
                            ->native(false),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['from_date'] ?? null,
                                fn (Builder $q, $date) => $q->whereDate('created_at', '>=', Carbon::parse($date))
                            )
                            ->when(
                                $data['to_date'] ?? null,
                                fn (Builder $q, $date) => $q->whereDate('created_at', '<=', Carbon::parse($date))
                            );
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];
                        if ($data['from_date'] ?? null) {
                            $indicators['from_date'] = 'من: '.$data['from_date'];
                        }
                        if ($data['to_date'] ?? null) {
                            $indicators['to_date'] = 'إلى: '.$data['to_date'];
                        }

                        return $indicators;
                    }),
            ])
            ->recordActions([
                ActionGroup::make([
                    ViewAction::make(),
                    EditAction::make(),
                    DeleteAction::make()
                        ->visible(fn (Shipment $record) => $record->status === ShipmentStatus::Preparing)
                        ->requiresConfirmation(),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->requiresConfirmation(),
                ]),
            ])
            ->defaultSort('created_at', 'desc')
            ->emptyStateHeading('لا توجد شحنات')
            ->emptyStateDescription('ابدأ بإضافة شحنة جديدة')
            ->emptyStateIcon('heroicon-o-truck')
            ->striped();
    }
}
