<?php

namespace App\Filament\Resources\Shipments\Schemas;

use App\Enums\ShipmentStatus;
use App\Models\Shipment;
use Filament\Infolists\Components\RepeatableEntry;
use Filament\Infolists\Components\TextEntry;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Schema;

class ShipmentInfolist
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            self::sectionBasicInfo(),
            self::sectionTracking(),
            self::sectionCod(),
            self::sectionStatusHistory(),
        ]);
    }

    /* ================== المعلومات الأساسية ================== */

    protected static function sectionBasicInfo(): Section
    {
        return Section::make('معلومات الشحنة')
            ->icon('heroicon-o-truck')
            ->columns(4)
            ->childComponents([
                TextEntry::make('tracking_number')
                    ->label('رقم التتبع')
                    ->weight('bold')
                    ->copyable()
                    ->placeholder('-'),

                TextEntry::make('order.order_number')
                    ->label('رقم الطلب')
                    ->badge()
                    ->color('primary')
                    ->url(fn (Shipment $record) => route('filament.admin.resources.orders.view', $record->order_id)),

                TextEntry::make('shippingCompany.name')
                    ->label('شركة الشحن')
                    ->badge()
                    ->color('info'),

                TextEntry::make('status')
                    ->label('الحالة')
                    ->badge()
                    ->formatStateUsing(fn (ShipmentStatus $state) => $state->label())
                    ->color(fn (ShipmentStatus $state) => $state->color())
                    ->icon(fn (ShipmentStatus $state) => $state->icon()),

                TextEntry::make('delivery_attempts')
                    ->label('محاولات التوصيل')
                    ->badge()
                    ->color(fn ($state) => $state > 0 ? 'warning' : 'gray'),
            ]);
    }

    /* ================== التتبع ================== */

    protected static function sectionTracking(): Section
    {
        return Section::make('تفاصيل التتبع')
            ->icon('heroicon-o-clock')
            ->columns(4)
            ->collapsed()
            ->childComponents([
                TextEntry::make('waybill_number')
                    ->label('رقم البوليصة')
                    ->placeholder('-'),

                TextEntry::make('shipped_at')
                    ->label('تاريخ الشحن')
                    ->dateTime('Y-m-d h:i A')
                    ->placeholder('-'),

                TextEntry::make('delivered_at')
                    ->label('تاريخ التوصيل')
                    ->dateTime('Y-m-d h:i A')
                    ->placeholder('-'),

                TextEntry::make('returned_at')
                    ->label('تاريخ الإرجاع')
                    ->dateTime('Y-m-d h:i A')
                    ->placeholder('-'),

                TextEntry::make('notes')
                    ->label('ملاحظات')
                    ->placeholder('لا توجد ملاحظات')
                    ->columnSpanFull(),
            ]);
    }

    /* ================== COD ================== */

    protected static function sectionCod(): Section
    {
        return Section::make('الدفع عند الاستلام')
            ->icon('heroicon-o-banknotes')
            ->columns(3)
            ->collapsed()
            ->visible(fn (Shipment $record) => $record->cod_amount > 0)
            ->childComponents([
                TextEntry::make('cod_amount')
                    ->label('المبلغ')
                    ->formatStateUsing(fn ($state) => number_format($state, 0, '', ',').' د.ع')
                    ->weight('bold')
                    ->color('success'),

                TextEntry::make('cod_collected')
                    ->label('حالة التحصيل')
                    ->badge()
                    ->formatStateUsing(fn ($state) => $state ? 'تم التحصيل' : 'لم يتم التحصيل')
                    ->color(fn ($state) => $state ? 'success' : 'warning')
                    ->icon(fn ($state) => $state ? 'heroicon-o-check-circle' : 'heroicon-o-clock'),

                TextEntry::make('cod_collected_at')
                    ->label('تاريخ التحصيل')
                    ->dateTime('Y-m-d h:i A')
                    ->placeholder('-'),
            ]);
    }

    /* ================== سجل الحالات ================== */

    protected static function sectionStatusHistory(): Section
    {
        return Section::make('سجل تتبع الشحنة')
            ->icon('heroicon-o-clock')
            ->collapsed()
            ->childComponents([
                RepeatableEntry::make('statusHistory')
                    ->label('التاريخ')
                    ->hiddenLabel()
                    ->schema([
                        Grid::make(4)->childComponents([
                            TextEntry::make('status')
                                ->label('الحالة')
                                ->badge()
                                ->formatStateUsing(fn ($state) => ShipmentStatus::tryFrom($state)?->label() ?? $state)
                                ->color(fn ($state) => ShipmentStatus::tryFrom($state)?->color() ?? 'gray'),

                            TextEntry::make('location')
                                ->label('الموقع')
                                ->placeholder('-'),

                            TextEntry::make('notes')
                                ->label('ملاحظات')
                                ->placeholder('-'),

                            TextEntry::make('changed_at')
                                ->label('التاريخ')
                                ->dateTime('Y-m-d h:i A'),
                        ]),
                    ]),
            ]);
    }
}
