<?php

namespace App\Filament\Resources\Shipments\Schemas;

use App\Enums\ShipmentStatus;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Schema;

class ShipmentForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            self::sectionBasicInfo(),
            self::sectionTracking(),
            self::sectionCod(),
        ]);
    }

    /* ================== Helpers ================== */

    protected static function moneyInput(string $name, string $label, bool $required = false): TextInput
    {
        $input = TextInput::make($name)
            ->label($label)
            ->numeric()
            ->default(0)
            ->suffix('د.ع')
            ->minValue(0);

        return $required ? $input->required() : $input;
    }

    /* ================== المعلومات الأساسية ================== */

    protected static function sectionBasicInfo(): Section
    {
        return Section::make('معلومات الشحنة')
            ->description('البيانات الأساسية للشحنة')
            ->icon('heroicon-o-truck')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    Select::make('order_id')
                        ->label('الطلب')
                        ->relationship('order', 'order_number')
                        ->required()
                        ->searchable()
                        ->preload()
                        ->prefixIcon('heroicon-o-shopping-cart'),

                    Select::make('shipping_company_id')
                        ->label('شركة الشحن')
                        ->relationship('shippingCompany', 'name')
                        ->required()
                        ->searchable()
                        ->preload()
                        ->prefixIcon('heroicon-o-building-office'),

                    Select::make('status')
                        ->label('الحالة')
                        ->options(ShipmentStatus::options())
                        ->default(ShipmentStatus::Preparing->value)
                        ->required()
                        ->prefixIcon('heroicon-o-flag'),
                ]),
            ]);
    }

    /* ================== التتبع ================== */

    protected static function sectionTracking(): Section
    {
        return Section::make('معلومات التتبع')
            ->description('أرقام التتبع والتواريخ')
            ->icon('heroicon-o-map')
            ->collapsible()
            ->collapsed()
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    TextInput::make('tracking_number')
                        ->label('رقم التتبع')
                        ->maxLength(100)
                        ->placeholder('رقم تتبع الشحنة')
                        ->prefixIcon('heroicon-o-hashtag'),

                    TextInput::make('waybill_number')
                        ->label('رقم بوليصة الشحن')
                        ->maxLength(100)
                        ->placeholder('رقم البوليصة')
                        ->prefixIcon('heroicon-o-document'),

                    TextInput::make('delivery_attempts')
                        ->label('محاولات التوصيل')
                        ->numeric()
                        ->default(0)
                        ->disabled()
                        ->dehydrated(false)
                        ->prefixIcon('heroicon-o-arrow-path'),

                    DateTimePicker::make('shipped_at')
                        ->label('تاريخ الشحن')
                        ->prefixIcon('heroicon-o-clock'),

                    DateTimePicker::make('delivered_at')
                        ->label('تاريخ التوصيل')
                        ->prefixIcon('heroicon-o-check'),

                    DateTimePicker::make('returned_at')
                        ->label('تاريخ الإرجاع')
                        ->prefixIcon('heroicon-o-arrow-uturn-left'),

                    Textarea::make('notes')
                        ->label('ملاحظات')
                        ->rows(2)
                        ->placeholder('ملاحظات عن الشحنة...')
                        ->columnSpanFull(),
                ]),
            ]);
    }

    /* ================== COD ================== */

    protected static function sectionCod(): Section
    {
        return Section::make('الدفع عند الاستلام (COD)')
            ->description('تفاصيل مبلغ COD')
            ->icon('heroicon-o-banknotes')
            ->collapsible()
            ->collapsed()
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    self::moneyInput('cod_amount', 'مبلغ COD'),

                    Toggle::make('cod_collected')
                        ->label('تم التحصيل')
                        ->default(false)
                        ->inline(false)
                        ->visibleOn('edit'),

                    DateTimePicker::make('cod_collected_at')
                        ->label('تاريخ التحصيل')
                        ->disabled()
                        ->dehydrated(false)
                        ->visible(fn (Get $get) => $get('cod_collected'))
                        ->visibleOn('edit'),
                ]),
            ]);
    }
}
