<?php

namespace App\Filament\Resources\Shipments\Pages;

use App\Enums\ShipmentStatus;
use App\Filament\Resources\Shipments\ShipmentResource;
use Filament\Actions\Action;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\ViewAction;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\EditRecord;

class EditShipment extends EditRecord
{
    protected static string $resource = ShipmentResource::class;

    protected function getHeaderActions(): array
    {
        $statusActions = [];

        // حالات الانتقال المتاحة
        $transitions = $this->getAvailableTransitions();

        foreach ($transitions as $transition) {
            $statusActions[] = Action::make('transition_'.$transition['status']->value)
                ->label($transition['label'])
                ->icon($transition['status']->icon())
                ->color($transition['color'])
                ->requiresConfirmation()
                ->modalHeading($transition['label'])
                ->form([
                    TextInput::make('location')
                        ->label('الموقع')
                        ->placeholder('الموقع الحالي...'),
                    Textarea::make('notes')
                        ->label('ملاحظات')
                        ->rows(2),
                ])
                ->action(function (array $data) use ($transition) {
                    $this->record->updateStatus(
                        $transition['status'],
                        $data['location'] ?? null,
                        $data['notes'] ?? null
                    );

                    Notification::make()
                        ->title('تم التحديث')
                        ->body('تم تحديث حالة الشحنة إلى: '.$transition['label'])
                        ->success()
                        ->send();

                    $this->refreshFormData(['status']);
                });
        }

        return [
            ViewAction::make(),

            ActionGroup::make($statusActions)
                ->label('تحديث الحالة')
                ->icon('heroicon-o-arrow-path')
                ->button()
                ->color('primary')
                ->visible(fn () => count($statusActions) > 0),

            Action::make('collect_cod')
                ->label('تحصيل COD')
                ->icon('heroicon-o-banknotes')
                ->color('success')
                ->visible(fn () => $this->record->has_pending_cod)
                ->requiresConfirmation()
                ->modalHeading('تأكيد تحصيل COD')
                ->modalDescription('هل تم تحصيل مبلغ '.number_format($this->record->cod_amount, 0, '', ',').' د.ع؟')
                ->action(function () {
                    $this->record->markCodCollected();

                    Notification::make()
                        ->title('تم التحصيل')
                        ->body('تم تسجيل تحصيل مبلغ COD بنجاح')
                        ->success()
                        ->send();

                    $this->refreshFormData(['cod_collected', 'cod_collected_at']);
                }),

            Action::make('record_attempt')
                ->label('تسجيل محاولة فاشلة')
                ->icon('heroicon-o-exclamation-triangle')
                ->color('warning')
                ->visible(fn () => $this->record->is_active)
                ->form([
                    Textarea::make('notes')
                        ->label('سبب الفشل')
                        ->required()
                        ->rows(2),
                ])
                ->action(function (array $data) {
                    $this->record->recordFailedAttempt($data['notes'] ?? null);

                    Notification::make()
                        ->title('تم التسجيل')
                        ->body('تم تسجيل محاولة التوصيل الفاشلة')
                        ->warning()
                        ->send();

                    $this->refreshFormData(['delivery_attempts']);
                }),

            DeleteAction::make()
                ->visible(fn () => $this->record->status === ShipmentStatus::Preparing),
        ];
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('index');
    }

    protected function getAvailableTransitions(): array
    {
        return match ($this->record->status) {
            ShipmentStatus::Preparing => [
                ['status' => ShipmentStatus::Picked, 'label' => 'تم الاستلام', 'color' => 'info'],
            ],
            ShipmentStatus::Picked => [
                ['status' => ShipmentStatus::InTransit, 'label' => 'في الطريق', 'color' => 'warning'],
            ],
            ShipmentStatus::InTransit => [
                ['status' => ShipmentStatus::OutForDelivery, 'label' => 'خرج للتوصيل', 'color' => 'primary'],
            ],
            ShipmentStatus::OutForDelivery => [
                ['status' => ShipmentStatus::Delivered, 'label' => 'تم التوصيل', 'color' => 'success'],
                ['status' => ShipmentStatus::Returned, 'label' => 'مرتجع', 'color' => 'danger'],
                ['status' => ShipmentStatus::Failed, 'label' => 'فشل', 'color' => 'danger'],
            ],
            default => [],
        };
    }
}
