<?php

namespace App\Filament\Resources\Roles\Tables;

use App\Enums\RoleEnum;
use Filament\Tables\Table;
use Illuminate\Support\Arr;
use Filament\Actions\EditAction;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use App\Support\PermissionsLabel;

class RolesTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->label('الدور')
                    ->badge()
                    ->separator(', ')
                    ->color(function ($state) {
                        $enum = RoleEnum::tryFrom($state);
                        return $enum?->getColor() ?? 'gray';
                    })
                    ->formatStateUsing(fn(string $state) => __("roles.$state")) // ⬅ ترجمة
                    ->searchable()
                    ->sortable(),

                TextColumn::make('permissions.name')
                    ->label('الصلاحيات')
                    ->badge()
                    ->color('danger')
                    ->limitList(3)
                    ->formatStateUsing(function (string $state) {
                        // يدعم مفاتيح بثلاثة أجزاء مثل: inventory.stock.transfer
                        $parts          = explode('.', $state);
                        $actionKey      = array_pop($parts);                  // آخر جزء = الفعل
                        $resourceKey    = implode('.', $parts);             // الباقي = المورد (قد يحتوي نقاط)

                        $resources = trans('permissions.resources');     // مصفوفة الموارد
                        $actions   = trans('permissions.actions');       // مصفوفة الأفعال

                        // جلب اسم المورد حتى لو كان متداخلًا (inventory.stock)
                        $resourceLabel = Arr::get($resources, $resourceKey, $resourceKey);
                        $actionLabel   = $actions[$actionKey] ?? $actionKey;

                        // تركيب النص النهائي حسب 'format'
                        return trans('permissions.format', [
                            'action'   => $actionLabel,
                            'resource' => $resourceLabel,
                        ]);
                    })
            ])
            ->filters([
                // مثال فلتر حسب صلاحية معيّنة
                SelectFilter::make('permissions')
                    ->label('الصلاحيات')
                    ->relationship('permissions', 'name')
                    ->multiple()
                    ->preload()
                    ->searchable(),
            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make()
                        ->label('تعديل')
                        ->icon('heroicon-m-pencil')
                        ->color('gray'),

                    DeleteAction::make()
                        ->label('حذف')
                        ->icon('heroicon-m-trash')
                        ->color('gray')
                        ->requiresConfirmation(),
                ])
                    ->label('') // نخليها أيقونة فقط
                    ->icon('heroicon-m-ellipsis-vertical')
                    ->iconButton()
                    ->color('gray')
                    ->size('lg')
                    ->extraAttributes(['class' => 'fi-dd-trigger']),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                ]),
            ]);
    }
}
