<?php

namespace App\Filament\Resources\Receipts\Tables;

use Filament\Tables\Table;
use App\Enums\ReceiptStatus;
use Filament\Actions\Action;
use Filament\Actions\EditAction;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Tables\Filters\Filter;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\ExportBulkAction;
use Filament\Tables\Filters\Indicator;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Columns\SelectColumn;
use Filament\Tables\Filters\SelectFilter;
use App\Filament\Resources\Receipts\ReceiptResource;
use Illuminate\Contracts\Database\Eloquent\Builder;

class ReceiptsTable
{
    /** توحيد عرض طريقة الدفع بالعربي */
    private const METHOD_LABELS = [
        'cash'     => 'نقدًا',
        'card'     => 'بطاقة',
        'transfer' => 'حوالة',
        'mixed'    => 'مختلطة',
    ];

    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('number')
                    ->label('رقم السند')
                    ->searchable()
                    ->copyable(),

                TextColumn::make('client.name')
                    ->label('العميل')
                    ->sortable()
                    ->searchable(),

                TextColumn::make('vault.name')
                    ->label('الخزنة')
                    ->toggleable(isToggledHiddenByDefault: true),

                // ── طريقة الدفع بالعربي كبادج
                TextColumn::make('method')
                    ->label('الطريقة')
                    ->badge()
                    ->formatStateUsing(fn($state) => self::METHOD_LABELS[$state] ?? '—')
                    ->colors([
                        'success' => ['cash'],
                        'info'    => ['card'],
                        'warning' => ['transfer'],
                        'gray'    => ['mixed'],
                    ]),

                // ── مبالغ مع فواصل + لاحقة العملة
                TextColumn::make('client_debt_before')
                    ->label('الديون السابقة')
                    ->formatStateUsing(fn($state) => self::money($state))
                    ->suffix(' د.ع')
                    ->badge()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('total_amount')
                    ->label('المبلغ')
                    ->formatStateUsing(fn($state) => self::money($state))
                    ->suffix(' د.ع')
                    ->badge()
                    ->color('info')
                    ->sortable(),

                TextColumn::make('client_debt_after')
                    ->label('المتبقي')
                    ->formatStateUsing(fn($state) => self::money($state))
                    ->suffix(' د.ع')
                    ->color(fn($record) => $record->client_debt_after > 0 ? 'danger' : 'success'),

                TextColumn::make('issue_date')
                    ->label('التاريخ')
                    ->date('Y-m-d')
                    ->sortable(),

                TextColumn::make('user.name')
                    ->label('المستخدم')
                    ->toggleable(isToggledHiddenByDefault: true),

                // ── تغيير الحالة مباشرة من الجدول (بدون formatStateUsing على SelectColumn)
                SelectColumn::make('status')
                    ->label('الحالة')
                    ->options(
                        fn() => collect(ReceiptStatus::cases())
                            ->mapWithKeys(fn(ReceiptStatus $s) => [$s->value => $s->label()])
                            ->toArray()
                    )
                    ->rules(['required'])
                    ->sortable()
                    ->selectablePlaceholder(false),
            ])
            ->defaultSort('issue_date', 'desc')
            ->filters([
                Filter::make('created_range')
                    ->label('تاريخ')
                    ->schema([
                        DatePicker::make('from')->label('من'),
                        DatePicker::make('to')->label('إلى'),
                    ])
                    ->query(function (Builder $query, array $data) {
                        return $query
                            ->when($data['from'] ?? null, fn($q, $d) => $q->whereDate('issue_date', '>=', $d))
                            ->when($data['to'] ?? null, fn($q, $d)   => $q->whereDate('issue_date', '<=', $d));
                    })
                    ->indicateUsing(function (array $data): ?Indicator {
                        if (!($data['from'] ?? null) && !($data['to'] ?? null)) return null;
                        $label = 'التاريخ';
                        if ($data['from'] ?? null) $label .= " من {$data['from']}";
                        if ($data['to']   ?? null) $label .= " إلى {$data['to']}";
                        return Indicator::make($label);
                    }),

                SelectFilter::make('method')
                    ->label('طريقة الدفع')
                    ->options(self::METHOD_LABELS),

                SelectFilter::make('status')
                    ->label('الحالة')
                    ->options(
                        fn() => collect(ReceiptStatus::cases())
                            ->mapWithKeys(fn(ReceiptStatus $s) => [$s->value => $s->label()])
                            ->toArray()
                    ),
            ])
            ->recordActions([
                ActionGroup::make([
                    Action::make('print')
                        ->label('طباعة')
                        ->icon('heroicon-o-printer')
                        ->color('success')
                        ->url(fn($record) => ReceiptResource::getUrl('print', ['record' => $record]))
                        ->openUrlInNewTab(),
                    EditAction::make()->label('تعديل')->icon('heroicon-m-pencil')->color('info'),
                    DeleteAction::make()->label('حذف')->icon('heroicon-m-trash')->color('danger')->requiresConfirmation(),
                ])
                    ->label('')
                    ->icon('heroicon-m-ellipsis-vertical')
                    ->iconButton()
                    ->color('gray')
                    ->size('lg')
                    ->extraAttributes(['class' => 'fi-dd-trigger']),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    ExportBulkAction::make()->label('تصدير'),
                    DeleteBulkAction::make(),
                ]),
            ]);
    }

    /** تنسيق مبالغ بفواصل (بدون كسور) مع fallback لـ — */
    private static function money($value): string
    {
        if ($value === null || $value === '') {
            return '—';
        }
        $digits = is_numeric($value) ? (int) $value : (int) preg_replace('/\D+/', '', (string) $value);
        return number_format($digits, 0, '.', ',');
    }
}
