<?php

namespace App\Filament\Resources\Receipts\Schemas;

use App\Models\Invoice;
use App\Models\Receipt;
use Filament\Schemas\Schema;
use Illuminate\Validation\Rule;
use Filament\Forms\Components\Select;
use Filament\Schemas\Components\Grid;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Fieldset;
use Illuminate\Database\Eloquent\Builder;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Coolsam\Flatpickr\Forms\Components\Flatpickr;

class ReceiptForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            Section::make()
                ->schema([
                    Grid::make(12)->schema([
                        Flatpickr::make('issue_date')
                            ->label('التاريخ')
                            ->format('Y-m-d')
                            ->displayFormat('Y-m-d')
                            ->locale('ar')                 // ← واجهة عربية
                            ->minDate(fn() => now('Asia/Baghdad')->startOfDay())
                            ->default(fn() => now('Asia/Baghdad')->toDateString())
                            ->timezone(null)
                            ->allowInput()
                            ->disableMobile(true)
                            ->columnSpan(4),

                        TextInput::make('number')
                            ->label('رقم سند القبض')
                            ->placeholder('—')
                            // نُظهر الرقم المُتولد للمستخدم
                            ->default(fn() => Receipt::generateNumber((int) (user_info('branch_id') ?? user_info()?->subscription?->id)))
                            ->readOnly()           // يبقى مُرسَل/مُخزَّن
                            ->required()
                            ->rules([
                                Rule::unique('receipts', 'number')
                                    ->where(fn($q) => $q->where('branch_id', (int) (user_info('branch_id') ?? user_info()?->subscription?->id))),
                            ])
                            ->columnSpan(4),

                        Select::make('method')
                            ->label('طريقة الدفع')
                            ->options([
                                'نقداً'       => 'نقداً',
                                'تحويل بنكي' => 'تحويل بنكي',
                                'أخرى'       => 'أخرى',
                            ])
                            ->default('نقداً')
                            ->required()
                            ->columnSpan(4),
                    ]),

                    /* اختيار العميل + الملخصات */
                    Grid::make(12)->schema([
                        Select::make('client_id')
                            ->label('اختيار العميل')
                            ->placeholder('اختر العميل')
                            ->relationship(
                                name: 'client',
                                titleAttribute: 'name',
                                modifyQueryUsing: fn(Builder $query) => $query->where(
                                    'branch_id',
                                    user_info('branch_id') ?? user_info()?->subscription?->id
                                )
                            )
                            ->searchable()
                            ->preload()
                            ->live()
                            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                if (blank($state)) {
                                    foreach (
                                        [
                                            'orders_total_view',
                                            'paid_total_view',
                                            'debts_total_view',
                                            'paid_total_before_view',
                                            'client_debt_before_view',
                                            'client_debt_after_view',
                                            'client_debt_before_raw',
                                            'client_debt_after_raw',
                                            'client_debt_before',
                                            'client_debt_after',
                                        ] as $k
                                    ) {
                                        $set($k, null);
                                    }
                                    return;
                                }

                                $subscriptionId = (int) (user_info('branch_id') ?? user_info()?->subscription?->id);
                                $totals = self::getClientTotals((int) $state, $subscriptionId);

                                // عرض منسّق
                                $set('orders_total_view',      self::formatCurrency($totals['orders_total']));
                                $set('paid_total_view',        self::formatCurrency($totals['paid_total']));
                                $set('debts_total_view',       self::formatCurrency($totals['debt_total']));
                                $set('paid_total_before_view', self::formatCurrency($totals['paid_total']));

                                // خام + حقول فعلية تُحفظ
                                $before = (int) $totals['debt_total'];
                                $set('client_debt_before_raw', $before);
                                $set('client_debt_before',     $before);
                                $set('client_debt_before_view', self::formatCurrency($before));

                                // المتبقي الحالي
                                $currentPaidRaw = (string) ($get('total_amount') ?? '0');
                                $currentPaid = (int) preg_replace('/\D+/', '', $currentPaidRaw);
                                $after = max(0, $before - $currentPaid);

                                $set('client_debt_after_raw',  $after);
                                $set('client_debt_after',      $after); // للحفظ
                                $set('client_debt_after_view', self::formatCurrency($after));
                            })
                            ->columnSpan(6)
                            ->rules([
                                Rule::exists('clients', 'id')->where(function ($q) {
                                    $q->where('branch_id', user_info('branch_id') ?? user_info()?->subscription?->id);
                                }),
                            ]),

                        TextInput::make('orders_total_view')
                            ->label('مجموع الطلبات')
                            ->suffix('د.ع')
                            ->dehydrated(false)->disabled()
                            ->afterStateHydrated(fn($state, Set $set) => $set('orders_total_view', $state === null ? null : self::formatCurrency($state)))
                            ->formatStateUsing(fn($get, $state) => $state === null ? '—' : (is_numeric($state) ? self::formatCurrency($state) : $state))
                            ->columnSpan(6),

                        TextInput::make('paid_total_view')
                            ->label('مجموع المدفوع')
                            ->suffix('د.ع')
                            ->dehydrated(false)->disabled()
                            ->afterStateHydrated(fn($state, Set $set) => $set('paid_total_view', $state === null ? null : self::formatCurrency($state)))
                            ->formatStateUsing(fn($get, $state) => $state === null ? '—' : (is_numeric($state) ? self::formatCurrency($state) : $state))
                            ->columnSpan(6),

                        TextInput::make('debts_total_view')
                            ->label('مجموع الديون')
                            ->suffix('د.ع')
                            ->dehydrated(false)->disabled()
                            ->afterStateHydrated(fn($state, Set $set) => $set('debts_total_view', $state === null ? null : self::formatCurrency($state)))
                            ->formatStateUsing(fn($get, $state) => $state === null ? '—' : (is_numeric($state) ? self::formatCurrency($state) : $state))
                            ->columnSpan(6),
                    ]),

                    /* القسط الحالي + السابق */
                    Grid::make(12)->schema([
                        Fieldset::make('معلومات القسط الحالي')
                            ->schema([
                                TextInput::make('total_amount')
                                    ->label('المبلغ المدفوع')
                                    ->suffix('د.ع')
                                    ->required()
                                    ->live(onBlur: false)
                                    ->afterStateHydrated(function ($state, Set $set) {
                                        if ($state !== null && $state !== '') {
                                            $set('total_amount', self::formatCurrency($state));
                                        }
                                    })
                                    ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                        $raw = preg_replace('/\D+/', '', (string) $state);
                                        $set('total_amount', $raw === '' ? '' : self::formatCurrency($raw));

                                        $paid   = (int) ($raw ?: 0);
                                        $before = (int) ($get('client_debt_before_raw') ?? 0);
                                        $after  = max(0, $before - $paid);

                                        // خام + حفظ + عرض
                                        $set('client_debt_after_raw',  $after);
                                        $set('client_debt_after',      $after); // للحفظ
                                        $set('client_debt_after_view', self::formatCurrency($after));
                                    })
                                    ->dehydrateStateUsing(fn($state) => (int) preg_replace('/\D+/', '', (string) $state))
                                    ->extraAttributes(['inputmode' => 'numeric'])
                                    ->columnSpan(6),

                                // المتبقي الحالي (عرض فقط)
                                TextInput::make('client_debt_after_view')
                                    ->label('المبلغ المتبقي')
                                    ->suffix('د.ع')
                                    ->dehydrated(false)->disabled()
                                    ->afterStateHydrated(fn($state, Set $set) => $set('client_debt_after_view', self::formatCurrency((int) ($state ?? 0))))
                                    ->formatStateUsing(fn($get, $state) => self::formatCurrency((int) preg_replace('/\D+/', '', (string) ($state ?? 0))))
                                    ->columnSpan(6),

                                Textarea::make('notes')
                                    ->label('الملاحظات')
                                    ->rows(2)
                                    ->dehydrated(false)
                                    ->columnSpan(6),
                            ])
                            ->columnSpan(6),

                        Fieldset::make('معلومات القسط السابق')
                            ->schema([
                                // المتبقي السابق (عرض فقط)
                                TextInput::make('client_debt_before_view')
                                    ->label('المبلغ المتبقي')
                                    ->suffix('د.ع')
                                    ->dehydrated(false)->disabled()
                                    ->afterStateHydrated(fn($state, Set $set) => $set('client_debt_before_view', self::formatCurrency((int) ($state ?? 0))))
                                    ->formatStateUsing(fn($get, $state) => self::formatCurrency((int) preg_replace('/\D+/', '', (string) ($state ?? 0))))
                                    ->columnSpan(6),

                                TextInput::make('paid_total_before_view')
                                    ->label('المبلغ المدفوع')
                                    ->dehydrated(false)->disabled()
                                    ->suffix('د.ع')
                                    ->afterStateHydrated(fn($state, Set $set) => $set('paid_total_before_view', self::formatCurrency($state ?? 0)))
                                    ->formatStateUsing(fn($get, $state) => self::formatCurrency((int) preg_replace('/\D+/', '', (string) ($state ?? 0))))
                                    ->columnSpan(6),

                                Textarea::make('notes_before_view')
                                    ->label('الملاحظات')
                                    ->rows(2)
                                    ->dehydrated(false)
                                    ->disabled()
                                    ->columnSpan(6),
                            ])
                            ->columnSpan(6),
                    ]),
                ])->columnSpanFull(),

            /* قيَم خام مساعدة (لا تُخزَّن) */
            TextInput::make('client_debt_before_raw')->hidden()->dehydrated(false),
            TextInput::make('client_debt_after_raw')->hidden()->dehydrated(false),

            /* حقول فعلية تُحفَظ في الجدول (مخفية) */
            TextInput::make('client_debt_before')->hidden()->default(0)->numeric(),
            TextInput::make('client_debt_after')->hidden()->default(0)->numeric(),

            /* قيَم خفية عامة */
            TextInput::make('branch_id')
                ->default(fn() => user_info('branch_id') ?? user_info()?->subscription?->id)
                ->hidden()
                ->numeric(),

            TextInput::make('user_id')
                ->default(fn() => user_info('id'))
                ->hidden()
                ->numeric(),
        ]);
    }

    /** يجلب مجموع الطلبات، مجموع المدفوع، والمتبقي */
    private static function getClientTotals(int $clientId, int $subscriptionId): array
    {
        $ordersTotal = (int) Invoice::query()
            ->where('client_id', $clientId)
            ->where('branch_id', $subscriptionId)
            ->sum('total_price');

        $paidTotal = (int) Receipt::query()
            ->where('client_id', $clientId)
            ->where('branch_id', $subscriptionId)
            ->whereNull('canceled_at')
            ->sum('total_amount');

        $debtTotal = max(0, $ordersTotal - $paidTotal);

        return [
            'orders_total' => $ordersTotal,
            'paid_total'   => $paidTotal,
            'debt_total'   => $debtTotal,
        ];
    }

    /** تنسيق أرقام بفواصل */
    private static function formatCurrency($amount): string
    {
        if ($amount === null || $amount === '') {
            return '0';
        }
        $digits = is_numeric($amount) ? (string) $amount : preg_replace('/\D+/', '', (string) $amount);
        if ($digits === '') {
            return '0';
        }
        return number_format((int) $digits, 0, '.', ',');
    }
}
