<?php

namespace App\Filament\Resources\Purchases\SupplierPayments\Tables;

use Filament\Tables\Table;
use App\Models\PurchaseInvoice;
use App\Models\SupplierPayment;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\RestoreAction;
use Filament\Tables\Filters\Filter;
use Illuminate\Contracts\View\View;
use Filament\Actions\BulkActionGroup;
use Filament\Forms\Components\Select;
use Filament\Actions\DeleteBulkAction;
use Filament\Support\Enums\FontWeight;
use Filament\Actions\ForceDeleteAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;

class SupplierPaymentsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns(self::getColumns())
            ->filters(self::getFilters())
            ->recordActions(self::getRecordActions())
            ->toolbarActions(self::getToolbarActions())
            ->defaultSort('payment_date', 'desc')
            ->striped()
            ->paginated([10, 25, 50, 100])
            ->emptyStateHeading('لا توجد دفعات')
            ->emptyStateDescription('قم بإنشاء دفعة جديدة للمورد');
    }

    // ══════════════════════════════════════════════════════════════
    //  إحصائيات الهيدر
    // ══════════════════════════════════════════════════════════════

    protected static function getHeaderStats(): array
    {
        $totals = PurchaseInvoice::selectRaw('
            COALESCE(SUM(total), 0) as total_invoices,
            COALESCE(SUM(paid), 0) as total_paid
        ')->first();

        return [
            'total_invoices' => (int) $totals->total_invoices,
            'total_paid'     => (int) $totals->total_paid,
            'total_due'      => max(0, $totals->total_invoices - $totals->total_paid),
        ];
    }

    // ══════════════════════════════════════════════════════════════
    //  الأعمدة
    // ══════════════════════════════════════════════════════════════

    protected static function getColumns(): array
    {
        return [
            TextColumn::make('payment_number')
                ->label('رقم الدفعة')
                ->searchable()
                ->sortable()
                ->copyable()
                ->weight('bold')
                ->color('primary'),

            TextColumn::make('supplier.name')
                ->label('المورد')
                ->searchable()
                ->sortable()
                ->weight('bold'),

            TextColumn::make('payment_date')
                ->label('التاريخ')
                ->date('Y-m-d')
                ->weight(FontWeight::Bold)
                ->sortable(),

            // الرصيد قبل الدفعة
            TextColumn::make('balance_before')
                ->label('المبلغ الاجمالي')
                ->weight(FontWeight::Bold)
                ->state(function ($record) {
                    return self::getBalanceBefore($record);
                })
                ->formatStateUsing(fn($state) => number_format($state))
                ->color('warning'),

            TextColumn::make('amount')
                ->label('المدفوع')
                ->sortable()
                ->weight(FontWeight::Bold)
                ->color('success')
                ->formatStateUsing(fn($state) => number_format($state)),

            // الرصيد بعد الدفعة
            TextColumn::make('balance_after')
                ->label('المتبقي')
                ->weight(FontWeight::Bold)
                ->state(function ($record) {
                    $before = self::getBalanceBefore($record);
                    return max(0, $before - $record->amount);
                })
                ->formatStateUsing(fn($state) => number_format($state))
                ->color(fn($state) => $state > 0 ? 'danger' : 'success'),

            TextColumn::make('payment_method')
                ->label('طريقة الدفع')
                ->badge()
                ->formatStateUsing(fn(string $state): string => match ($state) {
                    'cash'          => 'نقدي',
                    'bank_transfer' => 'تحويل بنكي',
                    'check'         => 'شيك',
                    'card'          => 'بطاقة',
                    default         => $state,
                })
                ->color(fn(string $state): string => match ($state) {
                    'cash'          => 'success',
                    'bank_transfer' => 'info',
                    'check'         => 'warning',
                    'card'          => 'primary',
                    default         => 'gray',
                }),

            TextColumn::make('branch.name')
                ->label('الفرع')
                ->sortable()
                ->visible(fn() => user_info()->hasRole('admin'))
                ->toggleable(isToggledHiddenByDefault: true),

            TextColumn::make('user.name')
                ->label('الموظف')
                ->sortable()
                ->toggleable(isToggledHiddenByDefault: true),

            TextColumn::make('created_at')
                ->label('تاريخ الإنشاء')
                ->dateTime('Y-m-d h:i A')
                ->sortable()
                ->toggleable(isToggledHiddenByDefault: true),
        ];
    }

    // ══════════════════════════════════════════════════════════════
    //  حساب الرصيد قبل الدفعة
    // ══════════════════════════════════════════════════════════════

    protected static function getBalanceBefore($record): int
    {
        // إجمالي فواتير المورد
        $totalInvoices = (int) PurchaseInvoice::where('supplier_id', $record->supplier_id)
            ->sum('total');

        // إجمالي الدفعات قبل هذه الدفعة (بناءً على التاريخ والـ ID)
        $paidBefore = (int) SupplierPayment::where('supplier_id', $record->supplier_id)
            ->where(function ($query) use ($record) {
                $query->where('payment_date', '<', $record->payment_date)
                    ->orWhere(function ($q) use ($record) {
                        $q->where('payment_date', '=', $record->payment_date)
                            ->where('id', '<', $record->id);
                    });
            })
            ->sum('amount');

        return max(0, $totalInvoices - $paidBefore);
    }

    // ══════════════════════════════════════════════════════════════
    //  الفلاتر
    // ══════════════════════════════════════════════════════════════

    protected static function getFilters(): array
    {
        return [
            Filter::make('payment_period')
                ->schema([
                    Select::make('period')
                        ->label('الفترة الزمنية')
                        ->options([
                            'today'      => 'اليوم',
                            'yesterday'  => 'أمس',
                            'this_week'  => 'الأسبوع الحالي',
                            'last_week'  => 'الأسبوع الماضي',
                            'this_month' => 'الشهر الحالي',
                            'last_month' => 'الشهر الماضي',
                            'custom'     => 'مخصص',
                        ])
                        ->live()
                        ->afterStateUpdated(fn(Set $set) => $set('from', null) & $set('to', null)),

                    DatePicker::make('from')
                        ->label('من تاريخ')
                        ->native(false)
                        ->visible(fn(Get $get) => $get('period') === 'custom'),

                    DatePicker::make('to')
                        ->label('إلى تاريخ')
                        ->native(false)
                        ->visible(fn(Get $get) => $get('period') === 'custom'),
                ])
                ->query(function (Builder $query, array $data): Builder {
                    $period = $data['period'] ?? null;

                    return match ($period) {
                        'today'      => $query->whereDate('payment_date', today()),
                        'yesterday'  => $query->whereDate('payment_date', today()->subDay()),
                        'this_week'  => $query->whereBetween('payment_date', [now()->startOfWeek(), now()->endOfWeek()]),
                        'last_week'  => $query->whereBetween('payment_date', [now()->subWeek()->startOfWeek(), now()->subWeek()->endOfWeek()]),
                        'this_month' => $query->whereBetween('payment_date', [now()->startOfMonth(), now()->endOfMonth()]),
                        'last_month' => $query->whereBetween('payment_date', [now()->subMonth()->startOfMonth(), now()->subMonth()->endOfMonth()]),
                        'custom'     => $query
                            ->when($data['from'] ?? null, fn($q, $d) => $q->whereDate('payment_date', '>=', $d))
                            ->when($data['to'] ?? null, fn($q, $d) => $q->whereDate('payment_date', '<=', $d)),
                        default      => $query,
                    };
                })
                ->indicateUsing(function (array $data): ?string {
                    return match ($data['period'] ?? null) {
                        'today'      => 'اليوم',
                        'yesterday'  => 'أمس',
                        'this_week'  => 'الأسبوع الحالي',
                        'last_week'  => 'الأسبوع الماضي',
                        'this_month' => 'الشهر الحالي',
                        'last_month' => 'الشهر الماضي',
                        'custom'     => ($data['from'] || $data['to']) ? 'من: ' . ($data['from'] ?? '...') . ' إلى: ' . ($data['to'] ?? '...') : null,
                        default      => null,
                    };
                }),

            SelectFilter::make('supplier_id')
                ->label('المورد')
                ->relationship('supplier', 'name')
                ->searchable()
                ->preload(),

            SelectFilter::make('branch_id')
                ->label('الفرع')
                ->searchable()
                ->preload()
                ->visible(fn() => user_info()->hasRole('admin'))
                ->relationship('branch', 'name'),

            SelectFilter::make('payment_method')
                ->label('طريقة الدفع')
                ->options([
                    'cash'          => 'نقدي',
                    'bank_transfer' => 'تحويل بنكي',
                    'check'         => 'شيك',
                    'card'          => 'بطاقة',
                ]),

            TrashedFilter::make()->label('حالة السجل'),
        ];
    }

    // ══════════════════════════════════════════════════════════════
    //  الإجراءات
    // ══════════════════════════════════════════════════════════════

    protected static function getRecordActions(): array
    {
        return [
            ActionGroup::make([
                ViewAction::make()->label('عرض'),
                EditAction::make()->label('تعديل'),
                DeleteAction::make()->label('حذف'),
                ForceDeleteAction::make()->label('حذف نهائي'),
                RestoreAction::make()->label('استعادة'),
            ])->tooltip('الإجراءات'),
        ];
    }

    protected static function getToolbarActions(): array
    {
        return [
            BulkActionGroup::make([
                DeleteBulkAction::make()->label('حذف المحدد'),
                ForceDeleteBulkAction::make()->label('حذف نهائي'),
                RestoreBulkAction::make()->label('استعادة المحدد'),
            ]),
        ];
    }
}
