<?php

namespace App\Filament\Resources\Purchases\SupplierPayments\Schemas;

use Filament\Schemas\Schema;
use App\Models\PurchaseInvoice;
use App\Models\SupplierPayment;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Select;
use Filament\Schemas\Components\Grid;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Coolsam\Flatpickr\Forms\Components\Flatpickr;

class SupplierPaymentForm
{
    // ══════════════════════════════════════════════════════════════
    //  تنسيق الأرقام
    // ══════════════════════════════════════════════════════════════

    protected static function nf(int|string|null $v): string
    {
        if ($v === null || $v === '') {
            return '0';
        }

        if (is_int($v)) {
            return number_format($v);
        }

        $cleaned = str_replace([',', ' ', '،'], '', strtr((string) $v, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]));

        return ctype_digit($cleaned) ? number_format((int) $cleaned) : '0';
    }

    protected static function toInt(mixed $v): int
    {
        if ($v === null || $v === '') {
            return 0;
        }

        if (is_int($v)) {
            return $v;
        }

        $cleaned = str_replace([',', ' ', '،'], '', strtr((string) $v, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]));

        return (int) $cleaned;
    }

    // ══════════════════════════════════════════════════════════════
    //  جلب بيانات المورد
    // ══════════════════════════════════════════════════════════════

    protected static function getSupplierData(int $supplierId): array
    {
        $invoiceData = PurchaseInvoice::where('supplier_id', $supplierId)
            ->selectRaw('COALESCE(SUM(total), 0) as total_invoices, COALESCE(SUM(paid), 0) as total_paid')
            ->first();

        $totalInvoices = (int) $invoiceData->total_invoices;
        $totalPaid = (int) $invoiceData->total_paid;

        $lastPayment = SupplierPayment::where('supplier_id', $supplierId)
            ->latest('payment_date')
            ->first(['amount', 'notes', 'payment_date']);

        return [
            'total_invoices'         => $totalInvoices,
            'total_paid'             => $totalPaid,
            'total_due'              => max(0, $totalInvoices - $totalPaid),
            'prev_payment_amount'    => (int) ($lastPayment?->amount ?? 0),
            'prev_payment_remaining' => max(0, $totalInvoices - $totalPaid),
            'prev_payment_notes'     => $lastPayment?->notes,
            'prev_payment_date'      => $lastPayment?->payment_date?->format('Y-m-d'),
        ];
    }

    // ══════════════════════════════════════════════════════════════
    //  الفورم
    // ══════════════════════════════════════════════════════════════

    public static function configure(Schema $schema): Schema
    {
        return $schema
            ->components([
                Section::make('معلومات الدفعة')
                    ->description('البيانات الأساسية للدفعة وحالة ديون المورد')
                    ->icon('heroicon-o-document-text')
                    ->schema([
                        Grid::make(3)->schema([
                            Select::make('supplier_id')
                                ->label('المورد')
                                ->relationship('supplier', 'name')
                                ->searchable()
                                ->preload()
                                ->required()
                                ->live(debounce: 500)
                                ->afterStateUpdated(fn(?string $state, Set $set) => self::onSupplierChange($state, $set))
                                ->rules([
                                    fn(): \Closure => function (string $attribute, $value, \Closure $fail) {
                                        if ($value) {
                                            $data = self::getSupplierData((int) $value);
                                            if ($data['total_due'] <= 0) {
                                                $fail('هذا المورد ليس عليه أي ديون مستحقة');
                                            }
                                        }
                                    },
                                ]),

                            Flatpickr::make('payment_date')
                                ->label('تاريخ الدفعة')
                                ->default(now()->format('Y-m-d'))
                                ->required(),

                            Select::make('payment_method')
                                ->label('طريقة الدفع')
                                ->options([
                                    'cash'          => 'نقدي',
                                    'bank_transfer' => 'تحويل بنكي',
                                    'check'         => 'شيك',
                                    'card'          => 'بطاقة',
                                ])
                                ->default('cash')
                                ->required(),
                        ]),

                        Grid::make(3)->schema([
                            TextInput::make('total_invoices')
                                ->label('إجمالي الفواتير')
                                ->suffix('د.ع')
                                ->readOnly()
                                ->dehydrated(false)
                                ->default('0')
                                ->formatStateUsing(fn($v) => self::nf($v)),

                            TextInput::make('total_paid')
                                ->label('المدفوع')
                                ->suffix('د.ع')
                                ->readOnly()
                                ->dehydrated(false)
                                ->default('0')
                                ->extraAttributes(['class' => 'text-success-600 dark:text-success-400'])
                                ->formatStateUsing(fn($v) => self::nf($v)),

                            TextInput::make('total_due')
                                ->label('المتبقي')
                                ->suffix('د.ع')
                                ->readOnly()
                                ->dehydrated(false)
                                ->default('0')
                                ->extraAttributes(['class' => 'text-danger-600 dark:text-danger-400'])
                                ->formatStateUsing(fn($v) => self::nf($v)),
                        ]),
                    ])
                    ->columnSpanFull()
                    ->collapsible(),

                Grid::make(2)
                    ->schema([
                        Section::make('الدفعة الحالية')
                            ->description('المبلغ المراد تسديده')
                            ->icon('heroicon-o-banknotes')
                            ->schema([
                                TextInput::make('amount')
                                    ->label('المبلغ')
                                    ->suffix('د.ع')
                                    ->required()
                                    ->default(0)
                                    ->live(debounce: 500)
                                    ->disabled(function (Get $get) {
                                        $supplierId = $get('supplier_id');
                                        if (! $supplierId) {
                                            return true;
                                        }
                                        $data = self::getSupplierData((int) $supplierId);
                                        return $data['total_due'] <= 0;
                                    })
                                    ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                        $paid = self::toInt($state);
                                        $due = self::toInt($get('total_due'));
                                        $set('current_remaining', self::nf(max(0, $due - $paid)));
                                    })
                                    ->dehydrateStateUsing(fn($state) => self::toInt($state))
                                    ->formatStateUsing(fn($v) => self::nf($v))
                                    ->extraInputAttributes([
                                        'inputmode' => 'numeric',
                                        'onkeyup'   => "this.value = this.value.replace(/[^0-9]/g,'').replace(/\B(?=(\d{3})+(?!\d))/g,',')",
                                    ]),

                                TextInput::make('current_remaining')
                                    ->label('المتبقي بعد الدفعة')
                                    ->suffix('د.ع')
                                    ->readOnly()
                                    ->dehydrated(false)
                                    ->default('0')
                                    ->extraAttributes(['class' => 'text-warning-600 dark:text-warning-400'])
                                    ->formatStateUsing(fn($v) => self::nf($v)),

                                Textarea::make('notes')
                                    ->label('ملاحظات')
                                    ->rows(2)
                                    ->placeholder('ملاحظات إضافية...'),
                            ]),

                        Section::make('الدفعة السابقة')
                            ->description(fn(Get $get) => $get('prev_payment_date')
                                ? 'بتاريخ ' . $get('prev_payment_date')
                                : 'لا توجد دفعات سابقة')
                            ->icon('heroicon-o-clock')
                            ->schema([
                                TextInput::make('prev_payment_amount')
                                    ->label('المبلغ')
                                    ->suffix('د.ع')
                                    ->readOnly()
                                    ->dehydrated(false)
                                    ->default('0')
                                    ->extraAttributes(['class' => 'text-success-600 dark:text-success-400'])
                                    ->formatStateUsing(fn($v) => self::nf($v)),

                                TextInput::make('prev_payment_remaining')
                                    ->label('المتبقي بعدها')
                                    ->suffix('د.ع')
                                    ->readOnly()
                                    ->dehydrated(false)
                                    ->default('0')
                                    ->formatStateUsing(fn($v) => self::nf($v)),

                                Hidden::make('prev_payment_date'),

                                Textarea::make('prev_payment_notes')
                                    ->label('ملاحظات')
                                    ->rows(2)
                                    ->readOnly()
                                    ->dehydrated(false)
                                    ->placeholder('لا توجد'),
                            ]),

                    ])->columnSpanFull(),

            ]);
    }

    // ══════════════════════════════════════════════════════════════
    //  عند تغيير المورد
    // ══════════════════════════════════════════════════════════════

    protected static function onSupplierChange(?string $supplierId, Set $set): void
    {
        $defaults = [
            'total_invoices'         => '0',
            'total_paid'             => '0',
            'total_due'              => '0',
            'prev_payment_amount'    => '0',
            'prev_payment_remaining' => '0',
            'prev_payment_notes'     => null,
            'prev_payment_date'      => null,
            'current_remaining'      => '0',
            'amount'                 => '0',
        ];

        foreach ($defaults as $field => $value) {
            $set($field, $value);
        }

        if (! $supplierId) {
            return;
        }

        $data = self::getSupplierData((int) $supplierId);

        // إظهار إشعار إذا لا توجد ديون
        if ($data['total_due'] <= 0) {
            Notification::make()
                ->title('لا توجد ديون')
                ->body('هذا المورد ليس عليه أي ديون مستحقة')
                ->warning()
                ->duration(5000)
                ->send();
        }

        $set('total_invoices', self::nf($data['total_invoices']));
        $set('total_paid', self::nf($data['total_paid']));
        $set('total_due', self::nf($data['total_due']));
        $set('prev_payment_amount', self::nf($data['prev_payment_amount']));
        $set('prev_payment_remaining', self::nf($data['prev_payment_remaining']));
        $set('prev_payment_notes', $data['prev_payment_notes']);
        $set('prev_payment_date', $data['prev_payment_date']);
    }
}
