<?php

namespace App\Filament\Resources\Purchases\PurchaseReturns\Tables;

use Filament\Tables\Table;
use Filament\Actions\EditAction;
use Filament\Actions\ViewAction;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\RestoreAction;
use Filament\Tables\Filters\Filter;
use Filament\Actions\BulkActionGroup;
use Filament\Forms\Components\Select;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\ForceDeleteAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Forms\Components\DatePicker;
use Filament\Tables\Filters\SelectFilter;
use Illuminate\Database\Eloquent\Builder;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;

class PurchaseReturnsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns(self::getColumns())
            ->filters(self::getFilters())
            ->recordActions(self::getRecordActions())
            ->toolbarActions(self::getToolbarActions())
            ->defaultSort('return_date', 'desc')
            ->striped()
            ->paginated([10, 25, 50, 100])
            ->emptyStateHeading('لا توجد مرتجعات')
            ->emptyStateDescription('قم بإنشاء مرتجع جديد');
    }

    protected static function getColumns(): array
    {
        return [
            TextColumn::make('return_number')
                ->label('رقم المرتجع')
                ->searchable()
                ->sortable()
                ->copyable()
                ->weight('bold')
                ->color('primary'),

            TextColumn::make('invoice.invoice_number')
                ->label('رقم الفاتورة')
                ->searchable()
                ->sortable(),

            TextColumn::make('supplier.name')
                ->label('المورد')
                ->searchable()
                ->sortable()
                ->weight('bold'),

            TextColumn::make('return_date')
                ->label('التاريخ')
                ->date('Y-m-d')
                ->sortable(),

            TextColumn::make('total')
                ->label('المبلغ')
                ->sortable()
                ->weight('bold')
                ->color('danger')
                ->formatStateUsing(fn($state) => number_format($state)),

            TextColumn::make('items_count')
                ->label('العناصر')
                ->counts('items')
                ->badge()
                ->color('gray'),

            TextColumn::make('status')
                ->label('الحالة')
                ->badge()
                ->formatStateUsing(fn(string $state): string => match ($state) {
                    'pending'   => 'قيد الانتظار',
                    'approved'  => 'معتمد',
                    'completed' => 'مكتمل',
                    'cancelled' => 'ملغي',
                    default     => $state,
                })
                ->color(fn(string $state): string => match ($state) {
                    'pending'   => 'warning',
                    'approved'  => 'info',
                    'completed' => 'success',
                    'cancelled' => 'danger',
                    default     => 'gray',
                }),

            TextColumn::make('refund_method')
                ->label('طريقة الاسترداد')
                ->badge()
                ->formatStateUsing(fn(?string $state): string => match ($state) {
                    'credit_note'   => 'إشعار دائن',
                    'cash'          => 'نقدي',
                    'bank_transfer' => 'تحويل بنكي',
                    'replacement'   => 'استبدال',
                    default         => $state ?? '-',
                })
                ->color(fn(?string $state): string => match ($state) {
                    'credit_note'   => 'info',
                    'cash'          => 'success',
                    'bank_transfer' => 'primary',
                    'replacement'   => 'warning',
                    default         => 'gray',
                })
                ->toggleable(isToggledHiddenByDefault: true),

            TextColumn::make('user.name')
                ->label('الموظف')
                ->sortable()
                ->toggleable(isToggledHiddenByDefault: true),

            TextColumn::make('created_at')
                ->label('تاريخ الإنشاء')
                ->dateTime('Y-m-d h:i A')
                ->sortable()
                ->toggleable(isToggledHiddenByDefault: true),
        ];
    }

    protected static function getFilters(): array
    {
        return [
            Filter::make('return_period')
                ->schema([
                    Select::make('period')
                        ->label('الفترة الزمنية')
                        ->options([
                            'today'      => 'اليوم',
                            'yesterday'  => 'أمس',
                            'this_week'  => 'الأسبوع الحالي',
                            'last_week'  => 'الأسبوع الماضي',
                            'this_month' => 'الشهر الحالي',
                            'last_month' => 'الشهر الماضي',
                            'custom'     => 'مخصص',
                        ])
                        ->live()
                        ->afterStateUpdated(fn(Set $set) => $set('from', null) & $set('to', null)),

                    DatePicker::make('from')
                        ->label('من تاريخ')
                        ->native(false)
                        ->visible(fn(Get $get) => $get('period') === 'custom'),

                    DatePicker::make('to')
                        ->label('إلى تاريخ')
                        ->native(false)
                        ->visible(fn(Get $get) => $get('period') === 'custom'),
                ])
                ->query(function (Builder $query, array $data): Builder {
                    $period = $data['period'] ?? null;

                    return match ($period) {
                        'today'      => $query->whereDate('return_date', today()),
                        'yesterday'  => $query->whereDate('return_date', today()->subDay()),
                        'this_week'  => $query->whereBetween('return_date', [now()->startOfWeek(), now()->endOfWeek()]),
                        'last_week'  => $query->whereBetween('return_date', [now()->subWeek()->startOfWeek(), now()->subWeek()->endOfWeek()]),
                        'this_month' => $query->whereBetween('return_date', [now()->startOfMonth(), now()->endOfMonth()]),
                        'last_month' => $query->whereBetween('return_date', [now()->subMonth()->startOfMonth(), now()->subMonth()->endOfMonth()]),
                        'custom'     => $query
                            ->when($data['from'] ?? null, fn($q, $d) => $q->whereDate('return_date', '>=', $d))
                            ->when($data['to'] ?? null, fn($q, $d) => $q->whereDate('return_date', '<=', $d)),
                        default      => $query,
                    };
                }),

            SelectFilter::make('supplier_id')
                ->label('المورد')
                ->relationship('supplier', 'name')
                ->searchable()
                ->preload(),

            SelectFilter::make('status')
                ->label('الحالة')
                ->options([
                    'pending'   => 'قيد الانتظار',
                    'approved'  => 'معتمد',
                    'completed' => 'مكتمل',
                    'cancelled' => 'ملغي',
                ]),

            SelectFilter::make('refund_method')
                ->label('طريقة الاسترداد')
                ->options([
                    'credit_note'   => 'إشعار دائن',
                    'cash'          => 'نقدي',
                    'bank_transfer' => 'تحويل بنكي',
                    'replacement'   => 'استبدال',
                ]),

            TrashedFilter::make()->label('حالة السجل'),
        ];
    }

    protected static function getRecordActions(): array
    {
        return [
            ActionGroup::make([
                ViewAction::make()->label('عرض'),
                EditAction::make()->label('تعديل'),
                DeleteAction::make()->label('حذف'),
                ForceDeleteAction::make()->label('حذف نهائي'),
                RestoreAction::make()->label('استعادة'),
            ])->tooltip('الإجراءات'),
        ];
    }

    protected static function getToolbarActions(): array
    {
        return [
            BulkActionGroup::make([
                DeleteBulkAction::make()->label('حذف المحدد'),
                ForceDeleteBulkAction::make()->label('حذف نهائي'),
                RestoreBulkAction::make()->label('استعادة المحدد'),
            ]),
        ];
    }
}
