<?php

namespace App\Filament\Resources\Purchases\PurchaseReturns\Schemas;

use Closure;
use Filament\Schemas\Schema;
use App\Models\PurchaseInvoice;
use Illuminate\Support\HtmlString;
use App\Models\PurchaseInvoiceItem;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Select;
use Filament\Schemas\Components\Grid;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Notifications\Notification;
use Filament\Schemas\Components\Section;
use Filament\Infolists\Components\TextEntry;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Coolsam\Flatpickr\Forms\Components\Flatpickr;

class PurchaseReturnForm
{
    // ══════════════════════════════════════════════════════════════
    //  تنسيق الأرقام
    // ══════════════════════════════════════════════════════════════

    protected static function nf(int|string|null $v): string
    {
        if ($v === null || $v === '') {
            return '0';
        }

        return number_format((int) $v);
    }

    protected static function toInt(mixed $v): int
    {
        if ($v === null || $v === '') {
            return 0;
        }

        if (is_int($v)) {
            return $v;
        }

        return (int) str_replace([',', ' '], '', (string) $v);
    }

    // ══════════════════════════════════════════════════════════════
    //  جلب بيانات الفاتورة
    // ══════════════════════════════════════════════════════════════

    protected static function getInvoiceData(int $invoiceId): array
    {
        $invoice = PurchaseInvoice::with(['supplier', 'items.variant.product'])->find($invoiceId);

        if (! $invoice) {
            return [];
        }

        return [
            'invoice'      => $invoice,
            'supplier_id'  => $invoice->supplier_id,
            'supplier_name' => $invoice->supplier?->name,
            'warehouse_id' => $invoice->warehouse_id,
            'total'        => $invoice->total,
            'items'        => $invoice->items->map(function ($item) {
                $returned = $item->returnItems()->sum('quantity_returned');
                $returnable = $item->quantity_received - $returned;

                return [
                    'id'                 => $item->id,
                    'product_variant_id' => $item->product_variant_id,
                    'variant_name'       => $item->variant?->full_name ?? $item->variant?->sku,
                    'quantity_received'  => $item->quantity_received,
                    'already_returned'   => $returned,
                    'returnable'         => $returnable,
                    'unit_cost'          => $item->unit_cost,
                    'batch_number'       => $item->batch_number,
                ];
            })->filter(fn($item) => $item['returnable'] > 0)->values()->toArray(),
        ];
    }

    // ══════════════════════════════════════════════════════════════
    //  الفورم
    // ══════════════════════════════════════════════════════════════

    public static function configure(Schema $schema): Schema
    {
        return $schema->components([

            // ─────────────────────────────────────────────────────
            //  معلومات المرتجع
            // ─────────────────────────────────────────────────────

            Section::make('معلومات المرتجع')
                ->description('اختر الفاتورة لإرجاع عناصر منها')
                ->icon('heroicon-o-arrow-uturn-left')
                ->schema([
                    Grid::make(3)->schema([
                        Select::make('purchase_invoice_id')
                            ->label('فاتورة الشراء')
                            ->options(function () {
                                return PurchaseInvoice::with('supplier')
                                    ->whereIn('status', ['partial', 'received'])
                                    ->where('status', '!=', 'returned')
                                    ->get()
                                    ->mapWithKeys(fn($inv) => [
                                        $inv->id => "{$inv->invoice_number}"
                                    ]);
                            })
                            ->searchable()
                            ->preload()
                            ->required()
                            ->live(debounce: 500)
                            ->afterStateUpdated(fn(?string $state, Set $set) => self::onInvoiceChange($state, $set)),

                        Flatpickr::make('return_date')
                            ->label('تاريخ المرتجع')
                            ->default(now())
                            ->required(),

                        Select::make('refund_method')
                            ->label('طريقة الاسترداد')
                            ->options([
                                'credit_note'   => 'إشعار دائن (خصم من الديون)',
                                'cash'          => 'استرداد نقدي',
                                'bank_transfer' => 'تحويل بنكي',
                                'replacement'   => 'استبدال',
                            ])
                            ->default('credit_note')
                            ->required(),
                    ]),

                    // معلومات الفاتورة المختارة
                    Grid::make(3)->schema([
                        TextInput::make('supplier_name')
                            ->label('المورد')
                            ->readOnly()
                            ->dehydrated(false)
                            ->placeholder('سيتم تحديده من الفاتورة'),

                        TextInput::make('invoice_total')
                            ->label('إجمالي الفاتورة')
                            ->suffix('د.ع')
                            ->readOnly()
                            ->dehydrated(false)
                            ->default('0')
                            ->formatStateUsing(fn($v) => self::nf($v)),

                        TextInput::make('return_total')
                            ->label('إجمالي المرتجع')
                            ->suffix('د.ع')
                            ->readOnly()
                            ->dehydrated(false)
                            ->default('0')
                            ->extraAttributes(['class' => 'text-danger-600 dark:text-danger-400 font-bold'])
                            ->formatStateUsing(fn($v) => self::nf($v)),
                    ]),

                    // حقول مخفية
                    Hidden::make('supplier_id'),
                    Hidden::make('warehouse_id'),

                    // تحذير لا توجد عناصر
                    TextEntry::make('no_items_alert')
                        ->label('')
                        ->state(new HtmlString('
                            <div class="flex items-center gap-2 rounded-lg bg-warning-50 p-4 text-warning-700 dark:bg-warning-950 dark:text-warning-400">
                                <svg class="h-5 w-5" fill="currentColor" viewBox="0 0 20 20">
                                    <path fill-rule="evenodd" d="M8.257 3.099c.765-1.36 2.722-1.36 3.486 0l5.58 9.92c.75 1.334-.213 2.98-1.742 2.98H4.42c-1.53 0-2.493-1.646-1.743-2.98l5.58-9.92zM11 13a1 1 0 11-2 0 1 1 0 012 0zm-1-8a1 1 0 00-1 1v3a1 1 0 002 0V6a1 1 0 00-1-1z" clip-rule="evenodd"/>
                                </svg>
                                <span class="font-medium">لا توجد عناصر قابلة للإرجاع في هذه الفاتورة</span>
                            </div>
                        '))
                        ->visible(function (Get $get) {
                            $invoiceId = $get('purchase_invoice_id');
                            if (! $invoiceId) return false;

                            $data = self::getInvoiceData((int) $invoiceId);
                            return empty($data['items']);
                        })
                        ->columnSpanFull(),
                ])
                ->columnSpanFull()
                ->collapsible(),

            // ─────────────────────────────────────────────────────
            //  عناصر المرتجع
            // ─────────────────────────────────────────────────────

            Section::make('عناصر المرتجع')
                ->description('حدد العناصر والكميات المراد إرجاعها')
                ->icon('heroicon-o-cube')
                ->schema([
                    Repeater::make('items')
                        ->label('')
                        ->hiddenLabel()
                        ->schema([
                            Grid::make(6)->schema([
                                Select::make('purchase_invoice_item_id')
                                    ->label('المنتج')
                                    ->options(function (Get $get) {
                                        $invoiceId = $get('../../purchase_invoice_id');
                                        if (! $invoiceId) return [];

                                        $data = self::getInvoiceData((int) $invoiceId);
                                        if (empty($data['items'])) return [];

                                        return collect($data['items'])->mapWithKeys(fn($item) => [
                                            $item['id'] => "{$item['variant_name']} (متاح: {$item['returnable']})"
                                        ])->toArray();
                                    })
                                    ->required()
                                    ->preload()
                                    ->searchable()
                                    ->live(debounce: 300)
                                    ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                        if (! $state) return;

                                        $invoiceId = $get('../../purchase_invoice_id');
                                        if (! $invoiceId) return;

                                        $data = self::getInvoiceData((int) $invoiceId);
                                        $item = collect($data['items'])->firstWhere('id', (int) $state);

                                        if ($item) {
                                            $set('product_variant_id', $item['product_variant_id']);
                                            $set('max_quantity', $item['returnable']);
                                            $set('unit_cost', $item['unit_cost']);
                                            $set('batch_number', $item['batch_number']);
                                            $set('quantity_returned', 1);
                                            $set('line_total', $item['unit_cost']);
                                        }
                                    })
                                    ->columnSpan(2),

                                Hidden::make('product_variant_id'),

                                TextInput::make('quantity_returned')
                                    ->label('الكمية')
                                    ->numeric()
                                    ->required()
                                    ->default(1)
                                    ->minValue(1)
                                    ->live(debounce: 300)
                                    ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                        $qty = (int) $state;
                                        $cost = self::toInt($get('unit_cost'));
                                        $set('line_total', $qty * $cost);

                                        // تحديث الإجمالي
                                        self::updateReturnTotal($get, $set);
                                    })
                                    ->rules([
                                        fn(Get $get): Closure => function (string $attribute, $value, Closure $fail) use ($get) {
                                            $max = (int) $get('max_quantity');
                                            if ($max > 0 && (int) $value > $max) {
                                                $fail("الكمية القصوى المتاحة للإرجاع: {$max}");
                                            }
                                        },
                                    ])
                                    ->columnSpan(1),

                                TextInput::make('max_quantity')
                                    ->label('المتاح')
                                    ->readOnly()
                                    ->dehydrated(false)
                                    ->default(0)
                                    ->columnSpan(1),

                                TextInput::make('unit_cost')
                                    ->label('سعر الوحدة')
                                    ->suffix('د.ع')
                                    ->readOnly()
                                    ->default(0)
                                    ->formatStateUsing(fn($v) => self::nf($v))
                                    ->columnSpan(1),

                                TextInput::make('line_total')
                                    ->label('الإجمالي')
                                    ->suffix('د.ع')
                                    ->readOnly()
                                    ->dehydrated(false)
                                    ->default(0)
                                    ->extraAttributes(['class' => 'font-bold'])
                                    ->formatStateUsing(fn($v) => self::nf($v))
                                    ->columnSpan(1),

                                Hidden::make('batch_number'),
                            ]),

                            Textarea::make('notes')
                                ->label('ملاحظات')
                                ->rows(1)
                                ->placeholder('سبب الإرجاع...')
                                ->columnSpanFull(),
                        ])
                        ->defaultItems(0)
                        ->addActionLabel('إضافة عنصر')
                        ->reorderable(false)
                        ->collapsible()
                        ->itemLabel(
                            fn(array $state): ?string =>
                            isset($state['purchase_invoice_item_id'])
                                ? "عنصر #{$state['purchase_invoice_item_id']}"
                                : 'عنصر جديد'
                        )
                        ->live()
                        ->afterStateUpdated(fn(Get $get, Set $set) => self::updateReturnTotal($get, $set))
                        ->columnSpanFull(),
                ])
                ->columnSpanFull()
                ->visible(fn(Get $get) => (bool) $get('purchase_invoice_id'))
                ->collapsible(),

            // ─────────────────────────────────────────────────────
            //  ملاحظات
            // ─────────────────────────────────────────────────────

            Section::make('ملاحظات')
                ->schema([
                    Textarea::make('notes')
                        ->label('ملاحظات عامة')
                        ->rows(3)
                        ->placeholder('أي ملاحظات إضافية على المرتجع...')
                        ->columnSpanFull(),
                ])
                ->columnSpanFull()
                ->collapsed(),

        ]);
    }

    // ══════════════════════════════════════════════════════════════
    //  عند تغيير الفاتورة
    // ══════════════════════════════════════════════════════════════

    protected static function onInvoiceChange(?string $invoiceId, Set $set): void
    {
        // إعادة تعيين
        $set('supplier_id', null);
        $set('supplier_name', null);
        $set('warehouse_id', null);
        $set('invoice_total', '0');
        $set('return_total', '0');
        $set('items', []);

        if (! $invoiceId) {
            return;
        }

        $data = self::getInvoiceData((int) $invoiceId);

        if (empty($data)) {
            Notification::make()
                ->title('خطأ')
                ->body('لم يتم العثور على الفاتورة')
                ->danger()
                ->send();
            return;
        }

        $set('supplier_id', $data['supplier_id']);
        $set('supplier_name', $data['supplier_name']);
        $set('warehouse_id', $data['warehouse_id']);
        $set('invoice_total', self::nf($data['total']));

        if (empty($data['items'])) {
            Notification::make()
                ->title('تنبيه')
                ->body('لا توجد عناصر قابلة للإرجاع في هذه الفاتورة')
                ->warning()
                ->send();
        }
    }

    // ══════════════════════════════════════════════════════════════
    //  تحديث إجمالي المرتجع
    // ══════════════════════════════════════════════════════════════

    protected static function updateReturnTotal(Get $get, Set $set): void
    {
        $items = $get('items') ?? [];

        $total = collect($items)->sum(function ($item) {
            $qty = (int) ($item['quantity_returned'] ?? 0);
            $cost = self::toInt($item['unit_cost'] ?? 0);
            return $qty * $cost;
        });

        $set('return_total', self::nf($total));
    }
}
