<?php

namespace App\Filament\Resources\Purchases\PurchaseReturns\Pages;

use App\Filament\Resources\Purchases\PurchaseReturns\PurchaseReturnResource;
use App\Services\Purchases\PurchaseReturnService;
use Filament\Actions\Action;
use Filament\Actions\DeleteAction;
use Filament\Actions\ForceDeleteAction;
use Filament\Actions\RestoreAction;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\EditRecord;
use Illuminate\Database\Eloquent\Model;

class EditPurchaseReturn extends EditRecord
{
    protected static string $resource = PurchaseReturnResource::class;

    protected function mutateFormDataBeforeFill(array $data): array
    {
        // تحميل العناصر
        $return = $this->record->load(['items', 'invoice.supplier']);

        $data['supplier_name'] = $return->invoice?->supplier?->name;
        $data['invoice_total'] = $return->invoice?->total ?? 0;
        $data['return_total'] = $return->total;

        // تحميل العناصر للـ Repeater
        $data['items'] = $return->items->map(function ($item) {
            return [
                'purchase_invoice_item_id' => $item->purchase_invoice_item_id,
                'product_variant_id'       => $item->product_variant_id,
                'quantity_returned'        => $item->quantity_returned,
                'unit_cost'                => $item->unit_cost,
                'line_total'               => $item->line_total,
                'batch_number'             => $item->batch_number,
                'notes'                    => $item->notes,
                'max_quantity'             => $this->getMaxQuantity($item),
            ];
        })->toArray();

        return $data;
    }

    protected function getMaxQuantity($item): int
    {
        if (! $item->purchase_invoice_item_id) {
            return 999;
        }

        $invoiceItem = $item->invoiceItem;
        if (! $invoiceItem) {
            return 999;
        }

        $otherReturned = $invoiceItem->returnItems()
            ->where('id', '!=', $item->id)
            ->sum('quantity_returned');

        return $invoiceItem->quantity_received - $otherReturned;
    }

    protected function handleRecordUpdate(Model $record, array $data): Model
    {
        // التحقق من الحالة
        if ($record->status === 'completed') {
            Notification::make()
                ->title('خطأ')
                ->body('لا يمكن تعديل مرتجع مكتمل')
                ->danger()
                ->send();

            $this->halt();
        }

        // استخراج العناصر
        $items = $data['items'] ?? [];
        unset($data['items']);

        // إزالة الحقول غير المطلوبة
        unset(
            $data['supplier_name'],
            $data['invoice_total'],
            $data['return_total']
        );

        // تنظيف العناصر
        $cleanItems = collect($items)->map(function ($item) {
            return [
                'purchase_invoice_item_id' => $item['purchase_invoice_item_id'],
                'product_variant_id'       => $item['product_variant_id'],
                'quantity_returned'        => (int) $item['quantity_returned'],
                'unit_cost'                => $this->cleanAmount($item['unit_cost'] ?? 0),
                'batch_number'             => $item['batch_number'] ?? null,
                'notes'                    => $item['notes'] ?? null,
            ];
        })->toArray();

        try {
            return app(PurchaseReturnService::class)->update($record, $data, $cleanItems);
        } catch (\InvalidArgumentException $e) {
            Notification::make()
                ->title('خطأ')
                ->body($e->getMessage())
                ->danger()
                ->send();

            $this->halt();
        }
    }

    protected function cleanAmount(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }

        if (is_int($value)) {
            return $value;
        }

        return (int) str_replace([',', ' '], '', (string) $value);
    }

    protected function getHeaderActions(): array
    {
        return [
            // اعتماد المرتجع
            Action::make('approve')
                ->label('اعتماد')
                ->icon('heroicon-o-check-circle')
                ->color('success')
                ->visible(fn() => $this->record->status === 'pending')
                ->requiresConfirmation()
                ->modalHeading('اعتماد المرتجع')
                ->modalDescription('هل أنت متأكد من اعتماد هذا المرتجع؟')
                ->action(function () {
                    $this->record->update(['status' => 'approved']);

                    Notification::make()
                        ->title('تم الاعتماد')
                        ->success()
                        ->send();
                }),

            // إكمال المرتجع
            Action::make('complete')
                ->label('إكمال')
                ->icon('heroicon-o-check-badge')
                ->color('primary')
                ->visible(fn() => $this->record->status === 'approved')
                ->requiresConfirmation()
                ->modalHeading('إكمال المرتجع')
                ->modalDescription('هل أنت متأكد من إكمال هذا المرتجع؟ لن يمكن التعديل عليه بعد ذلك.')
                ->action(function () {
                    $this->record->update(['status' => 'completed']);

                    Notification::make()
                        ->title('تم الإكمال')
                        ->success()
                        ->send();
                }),

            DeleteAction::make()
                ->visible(fn() => $this->record->status !== 'completed')
                ->requiresConfirmation()
                ->action(function () {
                    try {
                        app(PurchaseReturnService::class)->delete($this->record);

                        Notification::make()
                            ->title('تم الحذف')
                            ->success()
                            ->send();

                        $this->redirect($this->getResource()::getUrl('index'));
                    } catch (\InvalidArgumentException $e) {
                        Notification::make()
                            ->title('خطأ')
                            ->body($e->getMessage())
                            ->danger()
                            ->send();
                    }
                }),

            ForceDeleteAction::make()
                ->visible(fn() => $this->record->trashed()),

            RestoreAction::make(),
        ];
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('index');
    }

    protected function getSavedNotificationTitle(): ?string
    {
        return 'تم تحديث المرتجع بنجاح';
    }
}
