<?php

namespace App\Filament\Resources\Purchases\PurchaseReturns\Pages;

use App\Filament\Resources\Purchases\PurchaseReturns\PurchaseReturnResource;
use App\Services\Purchases\PurchaseReturnService;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\CreateRecord;
use Illuminate\Database\Eloquent\Model;

class CreatePurchaseReturn extends CreateRecord
{
    protected static string $resource = PurchaseReturnResource::class;

    protected function handleRecordCreation(array $data): Model
    {
        // استخراج العناصر
        $items = $data['items'] ?? [];
        unset($data['items']);

        // إزالة الحقول غير المطلوبة
        unset(
            $data['supplier_name'],
            $data['invoice_total'],
            $data['return_total']
        );

        // التحقق من وجود عناصر
        if (empty($items)) {
            Notification::make()
                ->title('خطأ')
                ->body('يجب إضافة عنصر واحد على الأقل للمرتجع')
                ->danger()
                ->send();

            $this->halt();
        }

        // تنظيف العناصر
        $cleanItems = collect($items)->map(function ($item) {
            return [
                'purchase_invoice_item_id' => $item['purchase_invoice_item_id'],
                'product_variant_id'       => $item['product_variant_id'],
                'quantity_returned'        => (int) $item['quantity_returned'],
                'unit_cost'                => $this->cleanAmount($item['unit_cost'] ?? 0),
                'batch_number'             => $item['batch_number'] ?? null,
                'notes'                    => $item['notes'] ?? null,
            ];
        })->toArray();

        try {
            return app(PurchaseReturnService::class)->create($data, $cleanItems);
        } catch (\InvalidArgumentException $e) {
            Notification::make()
                ->title('خطأ')
                ->body($e->getMessage())
                ->danger()
                ->send();

            $this->halt();
        }
    }

    protected function cleanAmount(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }

        if (is_int($value)) {
            return $value;
        }

        return (int) str_replace([',', ' '], '', (string) $value);
    }

    protected function getRedirectUrl(): string
    {
        return $this->getResource()::getUrl('edit', ['record' => $this->record]);
    }

    protected function getCreatedNotificationTitle(): ?string
    {
        return 'تم إنشاء المرتجع بنجاح';
    }
}
