<?php

namespace App\Filament\Resources\Purchases\PurchaseInvoices\Tables;

use Filament\Actions\ActionGroup;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Actions\ViewAction;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\Select;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TrashedFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class PurchaseInvoicesTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('invoice_number')
                    ->label('رقم الفاتورة')
                    ->searchable()
                    ->copyable()
                    ->sortable(),

                TextColumn::make('supplier.name')
                    ->label('المورد')
                    ->searchable()
                    ->toggleable(),

                TextColumn::make('branch.name')
                    ->label('الفرع')
                    ->visible(user_info()->hasRole('admin'))
                    ->toggleable(),

                TextColumn::make('warehouse.name')
                    ->label('المستودع')
                    ->toggleable(),

                TextColumn::make('total')
                    ->label('الإجمالي')
                    ->sortable()
                    ->suffix(' د.ع')
                    ->color('success')
                    ->weight('bold')
                    ->formatStateUsing(fn ($state) => number_format((int) $state)),

                TextColumn::make('paid')
                    ->label('المدفوع')
                    ->sortable()
                    ->suffix(' د.ع')
                    ->formatStateUsing(fn ($state) => number_format((int) $state))
                    ->weight('bold')
                    ->color('danger'),

                TextColumn::make('due')
                    ->label('المتبقي')
                    ->sortable()
                    ->suffix(' د.ع')
                    ->formatStateUsing(fn ($state) => number_format((int) $state))
                    ->weight('bold')
                    ->color('info'),

                TextColumn::make('payment_status')
                    ->label('حالة الدفع')
                    ->sortable()
                    ->badge()
                    ->colors([
                        'danger' => 'unpaid',
                        'warning' => 'partial',
                        'success' => 'paid',
                        'gray' => 'overpaid',
                    ])
                    ->formatStateUsing(fn ($state) => match ($state) {
                        'unpaid' => 'غير مدفوعة',
                        'partial' => 'مدفوعة جزئياً',
                        'paid' => 'مدفوعة بالكامل',
                        'overpaid' => 'دفع زائد',
                        default => $state,
                    }),

                TextColumn::make('status')
                    ->label('حالة الفاتورة')
                    ->sortable()
                    ->badge()
                    ->colors([
                        'gray' => 'draft',
                        'warning' => 'pending',
                        'info' => 'partial',
                        'success' => 'received',
                        'danger' => 'cancelled',
                    ])
                    ->formatStateUsing(fn ($state) => match ($state) {
                        'draft' => 'مسودة',
                        'pending' => 'قيد الانتظار',
                        'partial' => 'استلام جزئي',
                        'received' => 'مستلمة',
                        'cancelled' => 'ملغاة',
                        default => $state,
                    }),

                TextColumn::make('invoice_date')
                    ->label('تاريخ الفاتورة')
                    ->date('Y-m-d')
                    ->toggleable(isToggledHiddenByDefault: false)
                    ->sortable(),

                TextColumn::make('due_date')
                    ->label('تاريخ الاستحقاق')
                    ->date('Y-m-d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->icon(fn ($record) => ($record->due > 0 && $record->due_date && $record->due_date->isPast()) ? 'heroicon-m-exclamation-triangle' : null)
                    ->color(fn ($record) => ($record->due > 0 && $record->due_date && $record->due_date->isPast()) ? 'danger' : null),
                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d H:i A')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->sortable(),
                TextColumn::make('updated_at')
                    ->label('تاريخ التحديث')
                    ->dateTime('Y-m-d H:i A')
                    ->toggleable(isToggledHiddenByDefault: true)
                    ->sortable(),

            ])
            ->defaultSort('invoice_date', 'desc')
            ->filters([
                // سلة المحذوفات (Soft deletes)
                TrashedFilter::make(),
                // فلتر حالة الدفع
                SelectFilter::make('payment_status')
                    ->label('حالة الدفع')
                    ->options([
                        'unpaid' => 'غير مدفوعة',
                        'partial' => 'مدفوعة جزئياً',
                        'paid' => 'مدفوعة بالكامل',
                        'overpaid' => 'دفع زائد',
                    ])
                    ->searchable()
                    ->preload(),

                // فلتر الفرع
                SelectFilter::make('branch_id')
                    ->label('الفرع')
                    ->relationship('branch', 'name')
                    ->searchable()
                    ->visible(user_info()->hasRole('admin'))
                    ->preload(),

                // فلتر المستودع
                SelectFilter::make('warehouse_id')
                    ->label('المستودع')
                    ->relationship('warehouse', 'name')
                    ->searchable()
                    ->preload(),

                // بين تاريخين (تاريخ الفاتورة)
                // فلتر فترة الفاتورة المتقدم
                Filter::make('invoice_period')
                    ->schema([
                        Select::make('period')
                            ->label('الفترة الزمنية')
                            ->options([
                                'today' => '📅 اليوم',
                                'yesterday' => '📅 أمس',
                                'this_week' => '📅 الأسبوع الحالي',
                                'last_week' => '📅 الأسبوع الماضي',
                                'this_month' => '📅 الشهر الحالي',
                                'last_month' => '📅 الشهر الماضي',
                                'custom' => '🔧 مخصص',
                            ])
                            ->searchable()
                            ->preload()
                            ->live()
                            ->afterStateUpdated(fn (Set $set) => $set('from', null) & $set('to', null)),

                        DatePicker::make('from')
                            ->label('📅 من تاريخ')
                            ->displayFormat('Y-m-d')
                            ->native(false)
                            ->prefixIcon('heroicon-o-calendar-days')
                            ->visible(fn (Get $get) => $get('period') === 'custom'),

                        DatePicker::make('to')
                            ->label('📅 إلى تاريخ')
                            ->displayFormat('Y-m-d')
                            ->native(false)
                            ->prefixIcon('heroicon-o-calendar-days')
                            ->visible(fn (Get $get) => $get('period') === 'custom'),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        $period = $data['period'] ?? null;

                        return match ($period) {
                            'today' => $query->whereDate('invoice_date', today()),
                            'yesterday' => $query->whereDate('invoice_date', today()->subDay()),
                            'this_week' => $query->whereBetween('invoice_date', [
                                now()->startOfWeek(),
                                now()->endOfWeek(),
                            ]),
                            'last_week' => $query->whereBetween('invoice_date', [
                                now()->subWeek()->startOfWeek(),
                                now()->subWeek()->endOfWeek(),
                            ]),
                            'this_month' => $query->whereBetween('invoice_date', [
                                now()->startOfMonth(),
                                now()->endOfMonth(),
                            ]),
                            'last_month' => $query->whereBetween('invoice_date', [
                                now()->subMonth()->startOfMonth(),
                                now()->subMonth()->endOfMonth(),
                            ]),
                            'custom' => $query
                                ->when($data['from'] ?? null, fn ($q, $date) => $q->whereDate('invoice_date', '>=', $date))
                                ->when($data['to'] ?? null, fn ($q, $date) => $q->whereDate('invoice_date', '<=', $date)),
                            default => $query,
                        };
                    })
                    ->indicateUsing(function (array $data): ?string {
                        $period = $data['period'] ?? null;

                        return match ($period) {
                            'today' => 'الفترة: اليوم',
                            'yesterday' => 'الفترة: أمس',
                            'this_week' => 'الفترة: الأسبوع الحالي',
                            'last_week' => 'الفترة: الأسبوع الماضي',
                            'this_month' => 'الفترة: الشهر الحالي',
                            'last_month' => 'الفترة: الشهر الماضي',
                            'custom' => $data['from'] || $data['to']
                                ? 'من: '.($data['from'] ?? '...').' إلى: '.($data['to'] ?? '...')
                                : null,
                            default => null,
                        };
                    }),

                // فواتير متأخرة (متبقي > 0 وتاريخ الاستحقاق فائت)
                Filter::make('overdue')
                    ->label('متأخرة')
                    ->query(
                        fn (Builder $q) => $q
                            ->where('due', '>', 0)
                            ->whereNotNull('due_date')
                            ->whereDate('due_date', '<', now()->toDateString())
                    ),
            ])
            ->recordActions([
                ActionGroup::make([
                    ViewAction::make(),
                    EditAction::make(),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make(),
                    ForceDeleteBulkAction::make(),
                    RestoreBulkAction::make(),
                ]),
            ]);
    }
}
