<?php

namespace App\Filament\Resources\Purchases\PurchaseInvoices\Schemas;

use Filament\Schemas\Schema;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Infolists\Components\IconEntry;
use Filament\Infolists\Components\TextEntry;
use Filament\Infolists\Components\RepeatableEntry;

class PurchaseInvoiceInfolist
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            self::buildInvoiceInfoSection(),
            self::buildItemsSection(),
            self::buildTotalsSection(),
        ]);
    }

    /* =========================================
     * 1) معلومات الفاتورة
     * ======================================= */

    protected static function buildInvoiceInfoSection(): Section
    {
        return Section::make('📋 معلومات الفاتورة')
            ->icon('heroicon-o-document-text')
            ->schema([
                Grid::make(12)->schema([
                    TextEntry::make('supplier.name')
                        ->label('👨‍💼 المورد')
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextEntry::make('invoice_number')
                        ->label('🔖 رقم الفاتورة')
                        ->copyable()
                        ->copyMessage('تم نسخ رقم الفاتورة')
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextEntry::make('warehouse.name')
                        ->label('🏢 المستودع')
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextEntry::make('invoice_date')
                        ->label('📅 تاريخ الفاتورة')
                        ->date('Y-m-d')
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextEntry::make('due_date')
                        ->label('⏰ تاريخ الاستحقاق')
                        ->date('Y-m-d')
                        ->placeholder('-')
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextEntry::make('notes')
                        ->label('📝 ملاحظات')
                        ->placeholder('-')
                        ->columnSpanFull(),
                ]),
            ])
            ->collapsible()
            ->collapsed(false)
            ->columnSpanFull();
    }

    /* =========================================
     * 2) بنود الفاتورة (المنتجات)
     * ======================================= */

    protected static function buildItemsSection(): Section
    {
        return Section::make('📦 بنود الفاتورة')
            ->icon('heroicon-o-shopping-cart')
            ->schema([
                RepeatableEntry::make('items')
                    ->hiddenLabel()
                    ->schema([
                        Grid::make(12)->schema([
                            TextEntry::make('product.name')
                                ->label('المنتج')
                                ->columnSpan(['default' => 12, 'md' => 4]),

                            TextEntry::make('productVariant.full_name')
                                ->label('النسخة')
                                ->placeholder('افتراضية')
                                ->columnSpan(['default' => 12, 'md' => 4]),

                            TextEntry::make('quantity')
                                ->label('الكمية')
                                ->formatStateUsing(fn($state) => self::formatInt($state))
                                ->columnSpan(['default' => 12, 'md' => 4]),

                            TextEntry::make('unit_cost')
                                ->label('سعر الكلفة')
                                ->suffix(' د.ع')
                                ->formatStateUsing(fn($state) => self::formatInt($state))
                                ->columnSpan(['default' => 12, 'md' => 4]),

                            TextEntry::make('line_total')
                                ->label('إجمالي السطر')
                                ->suffix(' د.ع')
                                ->formatStateUsing(fn($state) => self::formatInt($state))
                                ->color('primary')
                                ->columnSpan(['default' => 12, 'md' => 4]),

                            TextEntry::make('batch_number')
                                ->label('رقم الدفعة')
                                ->placeholder('-')
                                ->columnSpan(['default' => 12, 'md' => 4]),

                            TextEntry::make('expiry_date')
                                ->label('تاريخ الانتهاء')
                                ->date('Y-m-d')
                                ->placeholder('-')
                                ->columnSpan(['default' => 12, 'md' => 4]),

                            TextEntry::make('notes')
                                ->label('ملاحظات السطر')
                                ->placeholder('-')
                                ->columnSpanFull(),
                        ]),
                    ])
                    ->columnSpanFull(),
            ])
            ->collapsible()
            ->collapsed(false)
            ->columnSpanFull();
    }

    /* =========================================
     * 3) المبالغ المالية والدفع
     * ======================================= */

    protected static function buildTotalsSection(): Section
    {
        return Section::make('💳 المبالغ المالية والدفع')
            ->icon('heroicon-o-banknotes')
            ->schema([
                Grid::make(12)->schema([
                    TextEntry::make('subtotal')
                        ->label('💎 مجموع العناصر')
                        ->suffix(' د.ع')
                        ->formatStateUsing(fn($state) => self::formatInt($state))
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextEntry::make('discount')
                        ->label('🎁 الخصم')
                        ->suffix(' د.ع')
                        ->formatStateUsing(fn($state) => self::formatInt($state))
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextEntry::make('freight')
                        ->label('🚚 أجور النقل')
                        ->suffix(' د.ع')
                        ->formatStateUsing(fn($state) => self::formatInt($state))
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextEntry::make('total')
                        ->label('💰 الإجمالي بعد الخصم والأجور')
                        ->suffix(' د.ع')
                        ->formatStateUsing(fn($state) => self::formatInt($state))
                        ->color('success')
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextEntry::make('paid')
                        ->label('✅ المبلغ المدفوع')
                        ->suffix(' د.ع')
                        ->formatStateUsing(fn($state) => self::formatInt($state))
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextEntry::make('due')
                        ->label('⏰ المتبقي (الدين)')
                        ->suffix(' د.ع')
                        ->formatStateUsing(fn($state) => self::formatInt($state))
                        ->color('warning')
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    IconEntry::make('receive_now')
                        ->label('📥 إدخال البضاعة إلى المخزن مباشرة')
                        ->boolean()
                        ->trueIcon('heroicon-o-check-circle')
                        ->falseIcon('heroicon-o-x-circle')
                        ->trueColor('success')
                        ->falseColor('gray')
                        ->columnSpan(['default' => 12, 'md' => 4]),

                    TextEntry::make('payment_status')
                        ->label('📊 حالة الدفع')
                        ->badge()
                        ->formatStateUsing(function ($state) {
                            return match ($state) {
                                'paid'    => 'مدفوع بالكامل',
                                'partial' => 'مدفوع جزئياً',
                                'unpaid', null => 'غير مدفوع',
                                default   => $state,
                            };
                        })
                        ->color(function ($state) {
                            return match ($state) {
                                'paid'    => 'success',
                                'partial' => 'warning',
                                default   => 'gray',
                            };
                        })
                        ->columnSpan(['default' => 12, 'md' => 4]),
                ]),
            ])
            ->collapsible()
            ->collapsed(false)
            ->columnSpanFull();
    }

    /* =========================================
     * Helpers
     * ======================================= */

    protected static function formatInt($value): string
    {
        $int = (int) ($value ?? 0);

        return number_format($int, 0, '.', ',');
    }
}
