<?php

namespace App\Filament\Resources\Purchases\PurchaseInvoices\Schemas;

use App\Enums\ProductUnit;
use App\Models\Attribute;
use App\Models\AttributeValue;
use App\Models\BranchWarehouse;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\ProductVariant;
use App\Models\PurchaseInvoice;
use App\Models\Warehouse;
use Filament\Actions\Action;
use Filament\Forms\Components\DatePicker;
use Filament\Forms\Components\FileUpload;
use Filament\Forms\Components\Hidden;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Notifications\Notification;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Group;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Filament\Schemas\Schema;
use Illuminate\Support\Facades\Cache;
use Illuminate\Support\Facades\Storage;
use Illuminate\Support\Str;

class PurchaseInvoiceForm
{
    private const MAX_RESULTS = 25;

    private const OPTIONS_LIMIT = 50;

    private const CACHE_TTL = 600;

    private const CACHE_TTL_LONG = 3600;

    /* ================================
     * Helpers: Integer فقط + تنسيق
     * ================================ */

    private static function toInt(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }

        $value = preg_replace_callback(
            '/[٠-٩]/u',
            fn ($m) => (string) mb_strpos('٠١٢٣٤٥٦٧٨٩', $m[0]),
            (string) $value
        );

        $clean = preg_replace('/[^\d]/u', '', (string) $value);

        return (int) ($clean ?: 0);
    }

    private static function fmt(mixed $value): string
    {
        return number_format(self::toInt($value), 0, '', ',');
    }

    private static function toFloat(mixed $value): float
    {
        if ($value === null || $value === '') {
            return 0.0;
        }

        $v = preg_replace_callback(
            '/[٠-٩]/u',
            fn ($m) => (string) mb_strpos('٠١٢٣٤٥٦٧٨٩', $m[0]),
            (string) $value
        );

        $v = preg_replace('/[^\d.]/u', '', (string) $v);

        if (substr_count($v, '.') > 1) {
            $parts = explode('.', $v);
            $v = $parts[0].'.'.implode('', array_slice($parts, 1));
        }

        return (float) ($v ?: 0);
    }

    /* ================================
     * Money Fields: تنسيق Live
     * ================================ */

    private static function moneyInput(string $name, string $label): TextInput
    {
        return TextInput::make($name)
            ->label($label)
            ->inputMode('numeric')
            ->step(1)
            ->default('0')
            ->live(debounce: 600)
            ->afterStateHydrated(function ($state, Set $set) use ($name) {
                $set($name, self::fmt($state));
            })
            ->afterStateUpdated(function ($state, Set $set) use ($name) {
                $formatted = self::fmt($state);
                if ((string) $state !== $formatted) {
                    $set($name, $formatted);
                }
            })
            ->dehydrateStateUsing(fn ($state) => self::toInt($state))
            ->rule('required')
            ->rule('regex:/^[\d,]+$/u');
    }

    private static function moneyDisplay(string $name, string $label, bool $dehydrated = true): TextInput
    {
        return TextInput::make($name)
            ->label($label)
            ->disabled()
            ->dehydrated($dehydrated)
            ->default('0')
            ->live()
            ->afterStateHydrated(function ($state, Set $set) use ($name) {
                $set($name, self::fmt($state));
            })
            ->dehydrateStateUsing(fn ($state) => self::toInt($state));
    }

    /* ================================
     * Pricing Defaults (ProductPrice)
     * ================================ */

    private static function fetchPricingDefaults(int $productId, int $variantId): array
    {
        // 1) سعر خاص للـ Variant
        $p = ProductPrice::query()
            ->active()
            ->where('product_variant_id', $variantId)
            ->first();

        // 2) وإلا سعر عام للمنتج (بدون variant)
        if (! $p) {
            $p = ProductPrice::query()
                ->active()
                ->where('product_id', $productId)
                ->whereNull('product_variant_id')
                ->first();
        }

        if (! $p) {
            return [
                'pricing_mode' => 'margin',
                'retail_margin' => '0',
                'wholesale_margin' => '0',
                'agent_margin' => '0',
                'retail_price' => '0',
                'wholesale_price' => '0',
                'agent_price' => '0',
            ];
        }

        $hasFixed =
            ((int) $p->retail_price > 0) ||
            ((int) $p->wholesale_price > 0) ||
            ((int) $p->agent_price > 0);

        return [
            'pricing_mode' => $hasFixed ? 'direct' : 'margin',

            // الهوامش (قد تكون كسور)
            'retail_margin' => rtrim(rtrim(number_format((float) $p->retail_margin, 2, '.', ''), '0'), '.'),
            'wholesale_margin' => rtrim(rtrim(number_format((float) $p->wholesale_margin, 2, '.', ''), '0'), '.'),
            'agent_margin' => rtrim(rtrim(number_format((float) $p->agent_margin, 2, '.', ''), '0'), '.'),

            // الأسعار (أرقام صحيحة منسّقة)
            'retail_price' => self::fmt((int) $p->retail_price),
            'wholesale_price' => self::fmt((int) $p->wholesale_price),
            'agent_price' => self::fmt((int) $p->agent_price),
        ];
    }

    private static function applyPricingDefaults(Set $set, int $productId, int $variantId): void
    {
        if ($productId <= 0 || $variantId <= 0) {
            return;
        }

        $d = self::fetchPricingDefaults($productId, $variantId);

        $set('pricing_mode', $d['pricing_mode']);

        $set('retail_margin', $d['retail_margin']);
        $set('wholesale_margin', $d['wholesale_margin']);
        $set('agent_margin', $d['agent_margin']);

        $set('retail_price', $d['retail_price']);
        $set('wholesale_price', $d['wholesale_price']);
        $set('agent_price', $d['agent_price']);
    }

    /* ================================
     * Schema
     * ================================ */

    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            self::invoiceInfo(),
            self::products(),
            self::financials(),
        ]);
    }

    private static function invoiceInfo(): Section
    {
        return Section::make('معلومات الفاتورة')
            ->description('البيانات الأساسية للمورد والمستودع والتواريخ')
            ->icon('heroicon-o-document-text')
            ->iconColor('primary')
            ->schema([
                Grid::make(12)->schema([
                    self::supplierField(),
                    self::invoiceNumberField(),
                    self::warehouseField(),
                    self::branchIdField(),
                    self::invoiceDateField(),
                    self::dueDateField(),
                    self::invoiceNotesField(),
                ]),
            ])
            ->collapsible()
            ->collapsed(false)
            ->columnSpanFull();
    }

    private static function products(): Section
    {
        return Section::make('منتجات الفاتورة')
            ->schema([
                Repeater::make('items')
                    ->label('')
                    ->hiddenLabel()
                    ->defaultItems(1)
                    ->minItems(1)

                    // إجبار إرسال items دائماً وبمفاتيح رقمية
                    ->dehydrated(true)
                    ->dehydrateStateUsing(function ($state) {
                        if (! is_array($state)) {
                            return [];
                        }

                        return array_values($state);
                    })

                    ->reorderable()
                    ->collapsible()
                    ->addActionLabel('إضافة منتج')
                    ->itemLabel(
                        fn (array $state): string => ! empty($state['product_id'])
                            ? (self::productName((int) $state['product_id']) ?? 'منتج')
                            : 'عنصر جديد'
                    )
                    ->afterStateHydrated(function ($state, Set $set, Get $get) {
                        $items = is_array($state) ? $state : [];

                        foreach ($items as $i => $item) {
                            $qty = max(1, (int) ($item['quantity_ordered'] ?? 1));
                            $cost = self::toInt($item['unit_cost'] ?? 0);

                            // line_total
                            $lineTotal = (int) ($item['line_total'] ?? ($qty * $cost));
                            $set("items.{$i}.unit_cost", self::fmt($cost));
                            $set("items.{$i}.line_total", $lineTotal);
                            $set("items.{$i}.line_total_display", self::fmt($lineTotal));

                            // إعادة بناء variant_rows وربط product_variant_id عند فتح التعديل
                            $variantRows = self::composeVariantRowsFromArray($item);
                            $set("items.{$i}.variant_rows", $variantRows);

                            $mainVariantId = (int) ($item['product_variant_id'] ?? 0);
                            if ($mainVariantId <= 0 && ! empty($variantRows)) {
                                $mainVariantId = (int) ($variantRows[0]['product_variant_id'] ?? 0);
                            }

                            $productId = (int) ($item['product_id'] ?? 0);
                            if ($productId > 0 && $mainVariantId <= 0) {
                                $mainVariantId = self::getOrCreateDefaultVariant($productId);
                            }

                            $set("items.{$i}.product_variant_id", $mainVariantId);

                            // التسعير: لو لم يُرسل من صفحة Edit أو كان صفراً، اجلبه من ProductPrice
                            $pricingMode = $item['pricing_mode'] ?? null;

                            $retailPrice = self::toInt($item['retail_price'] ?? 0);
                            $wholesalePrice = self::toInt($item['wholesale_price'] ?? 0);
                            $agentPrice = self::toInt($item['agent_price'] ?? 0);

                            $hasAnyPricing = ($retailPrice > 0) || ($wholesalePrice > 0) || ($agentPrice > 0);

                            if (! $pricingMode && ! $hasAnyPricing) {
                                $d = self::fetchPricingDefaults($productId, $mainVariantId);

                                $set("items.{$i}.pricing_mode", $d['pricing_mode']);
                                $set("items.{$i}.retail_margin", $d['retail_margin']);
                                $set("items.{$i}.wholesale_margin", $d['wholesale_margin']);
                                $set("items.{$i}.agent_margin", $d['agent_margin']);

                                $set("items.{$i}.retail_price", $d['retail_price']);
                                $set("items.{$i}.wholesale_price", $d['wholesale_price']);
                                $set("items.{$i}.agent_price", $d['agent_price']);
                            } else {
                                $set("items.{$i}.pricing_mode", $pricingMode ?: 'margin');
                                $set("items.{$i}.retail_margin", (string) ($item['retail_margin'] ?? '0'));
                                $set("items.{$i}.wholesale_margin", (string) ($item['wholesale_margin'] ?? '0'));
                                $set("items.{$i}.agent_margin", (string) ($item['agent_margin'] ?? '0'));

                                $set("items.{$i}.retail_price", self::fmt($item['retail_price'] ?? 0));
                                $set("items.{$i}.wholesale_price", self::fmt($item['wholesale_price'] ?? 0));
                                $set("items.{$i}.agent_price", self::fmt($item['agent_price'] ?? 0));
                            }
                        }

                        self::recalculateTotals($get, $set);
                    })
                    ->schema([
                        Hidden::make('id')->dehydrated(true),

                        // المتغير الرئيسي للسطر (يعتمد عليه الـ Service)
                        Hidden::make('product_variant_id')
                            ->default(0)
                            ->dehydrated(true)
                            ->dehydrateStateUsing(fn ($state) => (int) $state),

                        self::productField(),
                        self::attributesSection(),
                        self::itemDetailsGrid(),
                        self::autoPricingSection(),

                        self::lineTotalHidden(),
                        self::variantRowsHidden(),
                    ])
                    ->deleteAction(
                        fn ($action) => $action
                            ->requiresConfirmation()
                            ->color('danger')
                            ->after(fn (Set $set, Get $get) => self::recalculateTotals($get, $set))
                    )
                    ->columnSpanFull(),
            ])
            ->collapsible()
            ->columnSpanFull();
    }

    private static function financials(): Section
    {
        return Section::make('المدفوعات')
            ->schema([
                Grid::make(12)->schema([
                    self::subtotalField(),
                    self::discountField(),
                    self::freightField(),
                    self::totalField(),
                    self::paidField(),
                    self::dueField(),
                ]),
            ])
            ->collapsible()
            ->columnSpanFull();
    }

    /* ================================
     * Invoice Fields
     * ================================ */

    private static function supplierField(): Select
    {
        return Select::make('supplier_id')
            ->label('المورد')
            ->relationship('supplier', 'name')
            ->required()
            ->searchable()
            ->preload()
            ->native(false)
            ->prefixIcon('heroicon-o-user-group')
            ->createOptionForm(self::supplierCreateForm())
            ->createOptionAction(
                fn (Action $action) => $action
                    ->modalHeading('إضافة مورد جديد')
                    ->modalIcon('heroicon-o-user-plus')
                    ->modalIconColor('success')
                    ->modalSubmitActionLabel('حفظ')
                    ->color('success')
                    ->modalWidth('2xl')
            )
            ->columnSpan(['default' => 12, 'md' => 6, 'lg' => 4]);
    }

    private static function invoiceNumberField(): TextInput
    {
        return TextInput::make('invoice_number')
            ->label('رقم الفاتورة')
            ->placeholder('يُولَّد تلقائياً إذا تُرك فارغاً')
            ->default(fn () => self::generateInvoiceNumber())
            ->unique(table: PurchaseInvoice::class, ignorable: fn ($record) => $record)
            ->maxLength(100)
            ->prefixIcon('heroicon-o-hashtag')
            ->dehydrateStateUsing(function ($state) {
                $value = trim((string) $state);

                return $value !== '' ? $value : self::generateInvoiceNumber();
            })
            ->columnSpan(['default' => 12, 'md' => 6, 'lg' => 4]);
    }

    private static function warehouseField(): Select
    {
        return Select::make('warehouse_id')
            ->label('المستودع')
            ->required()
            ->searchable()
            ->native(false)
            ->options(function () {
                $branchId = user_info('branch_id');

                $warehouseIds = BranchWarehouse::query()
                    ->where('branch_id', $branchId)
                    ->whereNull('unassigned_at')
                    ->pluck('warehouse_id');

                return Warehouse::query()
                    ->whereIn('id', $warehouseIds)
                    ->orderBy('name')
                    ->pluck('name', 'id')
                    ->toArray();
            })
            ->prefixIcon('heroicon-o-building-storefront')
            ->columnSpan(['default' => 12, 'md' => 6, 'lg' => 4]);
    }

    private static function branchIdField(): Hidden
    {
        return Hidden::make('branch_id')->default(fn () => user_info('branch_id'));
    }

    private static function invoiceDateField(): DatePicker
    {
        return DatePicker::make('invoice_date')
            ->label('تاريخ الفاتورة')
            ->default(now())
            ->required()
            ->displayFormat('Y-m-d')
            ->native(false)
            ->prefixIcon('heroicon-o-calendar-days')
            ->columnSpan(['default' => 12, 'md' => 6, 'lg' => 4]);
    }

    private static function dueDateField(): DatePicker
    {
        return DatePicker::make('due_date')
            ->label('تاريخ الاستحقاق')
            ->displayFormat('Y-m-d')
            ->native(false)
            ->after('invoice_date')
            ->prefixIcon('heroicon-o-clock')
            ->columnSpan(['default' => 12, 'md' => 6, 'lg' => 4]);
    }

    private static function invoiceNotesField(): Textarea
    {
        return Textarea::make('notes')
            ->label('ملاحظات')
            ->rows(1)
            ->columnSpan(['default' => 12, 'lg' => 4]);
    }

    /* ================================
     * Product Field + Create Product + Image + Create Category/Brand
     * ================================ */

    private static function productField(): Select
    {
        return Select::make('product_id')
            ->label('المنتج')
            ->required()
            ->searchable()
            ->native(false)
            ->options(fn () => Cache::remember(
                'purchase_products_30',
                self::CACHE_TTL,
                fn () => Product::query()
                    ->select(['id', 'name'])
                    ->orderBy('name')
                    ->limit(30)
                    ->pluck('name', 'id')
                    ->toArray()
            ))
            ->getSearchResultsUsing(function (string $search) {
                $search = trim($search);
                if ($search === '') {
                    return [];
                }

                return Product::query()
                    ->select(['id', 'name'])
                    ->where('name', 'like', "%{$search}%")
                    ->orderBy('name')
                    ->limit(self::MAX_RESULTS)
                    ->pluck('name', 'id')
                    ->toArray();
            })
            ->getOptionLabelUsing(fn ($value) => $value ? self::productName((int) $value) : null)
            ->live()
            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                // reset
                $set('base_attributes', []);
                $set('color_quantities', []);
                $set('variant_rows', []);
                $set('quantity_ordered', 1);
                $set('unit_cost', '0');
                $set('line_total', 0);
                $set('line_total_display', self::fmt(0));

                // ensure default variant id immediately
                $productId = (int) ($state ?? 0);
                $mainVariantId = $productId > 0 ? self::getOrCreateDefaultVariant($productId) : 0;
                $set('product_variant_id', $mainVariantId);

                // بدل تصفير التسعير: اجلبه من ProductPrice
                if ($productId > 0 && $mainVariantId > 0) {
                    self::applyPricingDefaults($set, $productId, $mainVariantId);
                } else {
                    $set('pricing_mode', 'margin');
                    $set('retail_margin', '0');
                    $set('wholesale_margin', '0');
                    $set('agent_margin', '0');
                    $set('retail_price', '0');
                    $set('wholesale_price', '0');
                    $set('agent_price', '0');
                }

                self::updateLineTotal($set, $get);
                self::recalcPricesFromMargins($set, $get);

                // داخل items repeater => prefix للفاتورة هو ../../
                self::recalculateTotals($get, $set, '../../');
            })
            ->createOptionForm(self::productCreateForm())
            ->createOptionAction(
                fn (Action $action) => $action
                    ->modalHeading('إضافة منتج')
                    ->modalSubmitActionLabel('حفظ')
                    ->color('success')
                    ->modalWidth('3xl')
            )
            ->createOptionUsing(function (array $data) {
                $product = Product::create([
                    'name' => $data['name'],
                    'description' => $data['description'] ?? null,
                    'category_id' => $data['category_id'] ?? null,
                    'brand_id' => $data['brand_id'] ?? null,
                    'unit' => $data['product_unit'] ?? ProductUnit::PIECE->value,
                    'is_variable' => (bool) ($data['is_variable'] ?? false),
                    'is_active' => true,
                ]);

                $path = $data['image'] ?? null;

                if (is_string($path) && $path !== '') {
                    if (Storage::disk('upload')->exists($path)) {
                        $product
                            ->addMediaFromDisk($path, 'upload')
                            ->toMediaCollection(Product::MEDIA_COLLECTION_IMAGE, 'upload');

                        Storage::disk('upload')->delete($path);
                    }
                }

                Cache::forget('purchase_products_30');
                Notification::make()->success()->title('تمت إضافة المنتج')->send();

                return $product->id;
            })
            ->columnSpanFull();
    }

    private static function productCreateForm(): array
    {
        return [
            Grid::make(2)->schema([
                FileUpload::make('image')
                    ->label('الصورة')
                    ->disk('upload')
                    ->directory('tmp/products')
                    ->visibility('public')
                    ->image()
                    ->imageEditor()
                    ->maxFiles(1)
                    ->preserveFilenames()
                    ->downloadable()
                    ->openable()
                    ->columnSpan(2)
                    ->helperText('اختر صورة وسيتم ربطها تلقائياً بالمنتج بعد الحفظ'),

                TextInput::make('name')
                    ->label('اسم المنتج')
                    ->required()
                    ->maxLength(255)
                    ->columnSpan(2),

                Textarea::make('description')
                    ->label('الوصف')
                    ->rows(2)
                    ->maxLength(1000)
                    ->columnSpan(2),

                Select::make('category_id')
                    ->label('التصنيف')
                    ->native(false)
                    ->searchable()
                    ->preload()
                    ->options(fn () => Category::query()->orderBy('name')->pluck('name', 'id')->toArray())
                    ->createOptionForm([
                        TextInput::make('name')->label('اسم التصنيف')->required()->maxLength(255),
                        Select::make('parent_id')
                            ->label('تصنيف أب (اختياري)')
                            ->native(false)
                            ->searchable()
                            ->options(fn () => Category::query()->orderBy('name')->pluck('name', 'id')->toArray()),
                    ])
                    ->createOptionUsing(function (array $data) {
                        $category = Category::create([
                            'name' => trim((string) ($data['name'] ?? '')),
                            'parent_id' => $data['parent_id'] ?? null,
                            'is_active' => true,
                        ]);

                        return $category->id;
                    }),

                Select::make('brand_id')
                    ->label('العلامة التجارية')
                    ->native(false)
                    ->searchable()
                    ->preload()
                    ->options(fn () => Brand::query()->orderBy('name')->pluck('name', 'id')->toArray())
                    ->createOptionForm([
                        TextInput::make('name')->label('اسم البراند')->required()->maxLength(255),
                        Textarea::make('description')->label('وصف (اختياري)')->rows(2)->maxLength(1000),
                    ])
                    ->createOptionUsing(function (array $data) {
                        $brand = Brand::create([
                            'name' => trim((string) ($data['name'] ?? '')),
                            'description' => $data['description'] ?? null,
                            'is_active' => true,
                        ]);

                        return $brand->id;
                    }),

                Select::make('product_unit')
                    ->label('وحدة القياس')
                    ->options(ProductUnit::options())
                    ->native(false)
                    ->searchable()
                    ->required()
                    ->default(ProductUnit::PIECE->value),

                Toggle::make('is_variable')
                    ->label('منتج متعدد النسخ (Variants)')
                    ->default(false)
                    ->inline(false)
                    ->helperText('فعّلها إذا كان المنتج يعتمد على خواص مثل قياس/لون/مادة.'),
            ]),
        ];
    }

    /* ================================
     * Attributes Section
     * ================================ */

    private static function attributesSection(): Section
    {
        return Section::make('الخواص والألوان')
            ->visible(fn (Get $get) => (bool) $get('product_id'))
            ->schema([
                Grid::make(12)->schema([
                    self::baseAttributesGroup(),
                    self::colorQuantitiesGroup(),
                ]),
            ])
            ->columnSpanFull();
    }

    private static function baseAttributesGroup(): Group
    {
        return Group::make([
            Repeater::make('base_attributes')
                ->label('الخواص الأساسية (غير اللون)')
                ->defaultItems(0)
                ->collapsible()
                ->reorderable()
                ->addActionLabel('إضافة خاصية')
                ->itemLabel(function (array $state): string {
                    $attr = ! empty($state['attribute_id'])
                        ? self::attributeName((int) $state['attribute_id'])
                        : 'خاصية';

                    $val = ! empty($state['attribute_value_id'])
                        ? self::attributeValueName((int) $state['attribute_value_id'])
                        : '—';

                    return trim($attr.': '.$val);
                })
                ->schema([
                    Grid::make(12)->schema([
                        self::attributeIdField(),
                        self::attributeValueIdField(),
                    ]),
                ])
                ->afterStateUpdated(function ($state, Set $set, Get $get) {
                    // داخل item scope
                    self::rebuildColorVariants($set, $get, '');

                    self::syncQtyFromColors($set, $get);
                    self::buildVariantRows($set, $get);
                    self::syncMainVariantIdFromVariantRows($set, $get);

                    self::updateLineTotal($set, $get);
                    self::recalculateTotals($get, $set, '../../');
                }),
        ])->columnSpan(['default' => 12, 'md' => 6]);
    }

    private static function colorQuantitiesGroup(): Group
    {
        return Group::make([
            Repeater::make('color_quantities')
                ->label('الألوان والكميات')
                ->defaultItems(0)
                ->collapsible()
                ->reorderable()
                ->addActionLabel('إضافة لون')
                ->itemLabel(function (array $state): string {
                    $color = ! empty($state['color_value_id'])
                        ? self::attributeValueName((int) $state['color_value_id'])
                        : 'لون';

                    $qty = (int) ($state['qty'] ?? 0);

                    return $qty > 0 ? "{$color} — كمية: {$qty}" : "{$color}";
                })
                ->schema([
                    Grid::make(12)->schema([
                        Hidden::make('product_variant_id')->dehydrated(true)->default(0),
                        self::colorValueIdField(),
                        self::colorQtyField(),
                    ]),
                ])
                ->afterStateUpdated(function ($state, Set $set, Get $get) {
                    self::syncQtyFromColors($set, $get);
                    self::buildVariantRows($set, $get);
                    self::syncMainVariantIdFromVariantRows($set, $get);

                    self::updateLineTotal($set, $get);
                    self::recalculateTotals($get, $set, '../../');
                }),
        ])->columnSpan(['default' => 12, 'md' => 6]);
    }

    /* ================================
     * Attribute Fields
     * ================================ */

    private static function attributeIdField(): Select
    {
        return Select::make('attribute_id')
            ->label('الخاصية')
            ->required()
            ->native(false)
            ->searchable()
            ->options(fn () => self::baseAttributesOptions())
            ->distinct()
            ->disableOptionsWhenSelectedInSiblingRepeaterItems()
            ->live()
            ->afterStateUpdated(fn ($state, Set $set) => $set('attribute_value_id', null))
            ->columnSpan(['default' => 12, 'md' => 5]);
    }

    private static function attributeValueIdField(): Select
    {
        return Select::make('attribute_value_id')
            ->label('القيمة')
            ->required()
            ->native(false)
            ->searchable()
            ->disabled(fn (Get $get) => ! $get('attribute_id'))
            ->options(function (Get $get) {
                $attributeId = (int) ($get('attribute_id') ?? 0);

                return $attributeId ? self::attributeValuesOptions($attributeId) : [];
            })
            ->getSearchResultsUsing(function (Get $get, string $search) {
                $attributeId = (int) ($get('attribute_id') ?? 0);
                if ($attributeId <= 0) {
                    return [];
                }

                $search = trim($search);

                $q = AttributeValue::query()
                    ->where('attribute_id', $attributeId)
                    ->where('is_active', true);

                if ($search !== '') {
                    $q->where('display_value', 'like', "%{$search}%");
                }

                return $q->orderBy('sort_order')
                    ->orderBy('id')
                    ->limit(self::MAX_RESULTS)
                    ->pluck('display_value', 'id')
                    ->toArray();
            })
            ->getOptionLabelUsing(fn ($value) => $value ? self::attributeValueName((int) $value) : null)
            ->createOptionForm(function (Get $get) {
                $attrId = (int) ($get('attribute_id') ?? 0);

                return [
                    Hidden::make('attribute_id')->default($attrId),
                    TextInput::make('display_value')->label('قيمة جديدة')->required()->maxLength(190),
                    TextInput::make('sort_order')->label('الترتيب')->numeric()->default(0),
                ];
            })
            ->createOptionUsing(function (array $data, Get $get) {
                $attrId = (int) ($data['attribute_id'] ?? $get('attribute_id') ?? 0);
                $text = trim((string) ($data['display_value'] ?? ''));

                if ($attrId <= 0 || $text === '') {
                    return null;
                }

                $value = AttributeValue::firstOrCreate(
                    ['attribute_id' => $attrId, 'display_value' => $text],
                    [
                        'value' => Str::slug($text, '_'),
                        'sort_order' => (int) ($data['sort_order'] ?? 0),
                        'is_active' => true,
                    ]
                );

                Cache::forget("attr_values_opts_{$attrId}");

                return $value->id;
            })
            ->columnSpan(['default' => 12, 'md' => 7]);
    }

    /* ================================
     * Color Fields
     * ================================ */

    private static function colorValueIdField(): Select
    {
        return Select::make('color_value_id')
            ->label('اللون')
            ->required()
            ->native(false)
            ->searchable()
            ->distinct()
            ->disableOptionsWhenSelectedInSiblingRepeaterItems()
            ->options(function () {
                $colorAttrId = self::colorAttributeId();

                return $colorAttrId ? self::attributeValuesOptions($colorAttrId) : [];
            })
            ->getSearchResultsUsing(function (string $search) {
                $colorAttrId = self::colorAttributeId();
                if (! $colorAttrId) {
                    return [];
                }

                $search = trim($search);

                $q = AttributeValue::query()
                    ->where('attribute_id', $colorAttrId)
                    ->where('is_active', true);

                if ($search !== '') {
                    $q->where('display_value', 'like', "%{$search}%");
                }

                return $q->orderBy('sort_order')
                    ->orderBy('id')
                    ->limit(self::MAX_RESULTS)
                    ->pluck('display_value', 'id')
                    ->toArray();
            })
            ->getOptionLabelUsing(fn ($value) => $value ? self::attributeValueName((int) $value) : null)
            ->createOptionForm(function () {
                $attrId = self::colorAttributeId();

                return [
                    Hidden::make('attribute_id')->default($attrId),
                    TextInput::make('display_value')->label('لون جديد')->required()->maxLength(190),
                    TextInput::make('color_code')->label('Color HEX (اختياري)')->placeholder('#RRGGBB')->maxLength(7),
                    TextInput::make('sort_order')->label('الترتيب')->numeric()->default(0),
                ];
            })
            ->createOptionUsing(function (array $data) {
                $attrId = (int) ($data['attribute_id'] ?? self::colorAttributeId() ?? 0);
                $text = trim((string) ($data['display_value'] ?? ''));

                if ($attrId <= 0 || $text === '') {
                    return null;
                }

                $value = AttributeValue::firstOrCreate(
                    ['attribute_id' => $attrId, 'display_value' => $text],
                    [
                        'value' => Str::slug($text, '_'),
                        'color_code' => $data['color_code'] ?? null,
                        'sort_order' => (int) ($data['sort_order'] ?? 0),
                        'is_active' => true,
                    ]
                );

                Cache::forget("attr_values_opts_{$attrId}");

                return $value->id;
            })
            ->live()
            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                $colorValueId = (int) ($state ?? 0);

                // داخل color_quantities repeater => basePath = '../../'
                $variantId = self::resolveVariantForColorRow($get, $colorValueId, '../../');
                $set('product_variant_id', $variantId);

                self::syncQtyFromColors($set, $get, '../../');
                self::buildVariantRows($set, $get, '../../');
                self::syncMainVariantIdFromVariantRows($set, $get, '../../');

                self::updateLineTotal($set, $get, '../../');
                self::recalculateTotals($get, $set, '../../../../');
            })
            ->columnSpan(['default' => 12, 'md' => 7]);
    }

    private static function colorQtyField(): TextInput
    {
        return TextInput::make('qty')
            ->label('الكمية')
            ->numeric()
            ->inputMode('numeric')
            ->default(1)
            ->minValue(1)
            ->required()
            ->live(debounce: 400)
            ->dehydrateStateUsing(fn ($state) => max(1, (int) ($state ?? 1)))
            ->columnSpan(['default' => 12, 'md' => 5]);
    }

    /* ================================
     * Item Details Grid
     * ================================ */

    private static function itemDetailsGrid(): Grid
    {
        return Grid::make(12)->schema([
            self::quantityOrderedField(),
            self::unitCostField(),
            self::lineTotalDisplayField(),
            self::itemNotesField(),
        ]);
    }

    private static function quantityOrderedField(): TextInput
    {
        return TextInput::make('quantity_ordered')
            ->label('الكمية المطلوبة')
            ->numeric()
            ->inputMode('numeric')
            ->default(1)
            ->minValue(1)
            ->disabled(function (Get $get) {
                $rows = $get('color_quantities') ?? [];

                return is_array($rows) && count($rows) > 0;
            })
            ->live(debounce: 750)
            ->dehydrateStateUsing(fn ($state) => max(1, (int) ($state ?? 1)))
            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                self::buildVariantRows($set, $get);
                self::syncMainVariantIdFromVariantRows($set, $get);
                self::updateLineTotal($set, $get);
                self::recalculateTotals($get, $set, '../../');
            })
            ->columnSpan(['default' => 12, 'md' => 4]);
    }

    private static function unitCostField(): TextInput
    {
        return self::moneyInput('unit_cost', 'سعر الكلفة')
            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                self::syncQtyFromColors($set, $get);
                self::buildVariantRows($set, $get);

                self::syncMainVariantIdFromVariantRows($set, $get);

                self::updateLineTotal($set, $get);
                self::recalcPricesFromMargins($set, $get);
                self::recalculateTotals($get, $set, '../../');
            })
            ->columnSpan(['default' => 12, 'md' => 4]);
    }

    private static function lineTotalDisplayField(): TextInput
    {
        return TextInput::make('line_total_display')
            ->label('إجمالي السطر')
            ->disabled()
            ->dehydrated(false)
            ->default('0')
            ->live()
            ->afterStateHydrated(function ($state, Set $set, Get $get) {
                $lineTotal = self::toInt($get('line_total') ?? 0);

                if ($lineTotal <= 0) {
                    $qty = max(0, (int) ($get('quantity_ordered') ?? 0));
                    $cost = self::toInt($get('unit_cost') ?? 0);
                    $lineTotal = $qty * $cost;
                }

                $set('line_total_display', self::fmt($lineTotal));
            })
            ->columnSpan(['default' => 12, 'md' => 4]);
    }

    private static function itemNotesField(): Textarea
    {
        return Textarea::make('notes')
            ->label('ملاحظات السطر')
            ->rows(2)
            ->maxLength(500)
            ->columnSpanFull();
    }

    /* ================================
     * Auto Pricing Section
     * ================================ */

    private static function autoPricingSection(): Section
    {
        return Section::make('التسعير')
            ->description('احسب أسعار البيع تلقائياً أو أدخلها يدوياً (المبالغ أرقام صحيحة)')
            ->icon('heroicon-o-calculator')
            ->iconColor('info')
            ->collapsed()
            ->visible(fn (Get $get) => (bool) $get('product_id'))
            ->schema([
                Grid::make(12)->schema([
                    Select::make('pricing_mode')
                        ->label('نوع التسعير')
                        ->options([
                            'margin' => 'بنسبة الربح',
                            'direct' => 'إدخال مباشر',
                        ])
                        ->default('margin')
                        ->native(false)
                        ->live()
                        ->afterStateUpdated(function ($state, Set $set, Get $get) {
                            if ($state === 'margin') {
                                self::recalcPricesFromMargins($set, $get);
                            }
                        })
                        ->columnSpanFull(),

                    Group::make([
                        Grid::make(12)->schema([
                            TextInput::make('retail_margin')
                                ->label('هامش المفرد %')
                                ->default('0')
                                ->suffix('%')
                                ->inputMode('decimal')
                                ->live(debounce: 500)
                                ->afterStateUpdated(fn ($state, Set $set, Get $get) => self::recalcPricesFromMargins($set, $get))
                                ->columnSpan(['default' => 12, 'md' => 4]),

                            TextInput::make('wholesale_margin')
                                ->label('هامش الجملة %')
                                ->default('0')
                                ->suffix('%')
                                ->inputMode('decimal')
                                ->live(debounce: 500)
                                ->afterStateUpdated(fn ($state, Set $set, Get $get) => self::recalcPricesFromMargins($set, $get))
                                ->columnSpan(['default' => 12, 'md' => 4]),

                            TextInput::make('agent_margin')
                                ->label('هامش الوكيل %')
                                ->default('0')
                                ->suffix('%')
                                ->inputMode('decimal')
                                ->live(debounce: 500)
                                ->afterStateUpdated(fn ($state, Set $set, Get $get) => self::recalcPricesFromMargins($set, $get))
                                ->columnSpan(['default' => 12, 'md' => 4]),
                        ]),
                    ])
                        ->visible(fn (Get $get) => ($get('pricing_mode') ?? 'margin') === 'margin')
                        ->columnSpanFull(),

                    Grid::make(12)->schema([
                        self::moneyInput('retail_price', 'سعر المفرد')
                            ->suffix('د.ع')
                            ->disabled(fn (Get $get) => ($get('pricing_mode') ?? 'margin') === 'margin')
                            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                if (($get('pricing_mode') ?? 'margin') !== 'direct') {
                                    return;
                                }

                                $cost = self::toInt($get('unit_cost') ?? 0);
                                $price = self::toInt($state);

                                $set('retail_price', self::fmt($price));

                                if ($cost > 0 && $price > 0) {
                                    $margin = (($price - $cost) / $cost) * 100;
                                    $set('retail_margin', rtrim(rtrim(number_format($margin, 2, '.', ''), '0'), '.'));
                                }
                            })
                            ->columnSpan(['default' => 12, 'md' => 4]),

                        self::moneyInput('wholesale_price', 'سعر الجملة')
                            ->suffix('د.ع')
                            ->disabled(fn (Get $get) => ($get('pricing_mode') ?? 'margin') === 'margin')
                            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                if (($get('pricing_mode') ?? 'margin') !== 'direct') {
                                    return;
                                }

                                $cost = self::toInt($get('unit_cost') ?? 0);
                                $price = self::toInt($state);

                                $set('wholesale_price', self::fmt($price));

                                if ($cost > 0 && $price > 0) {
                                    $margin = (($price - $cost) / $cost) * 100;
                                    $set('wholesale_margin', rtrim(rtrim(number_format($margin, 2, '.', ''), '0'), '.'));
                                }
                            })
                            ->columnSpan(['default' => 12, 'md' => 4]),

                        self::moneyInput('agent_price', 'سعر الوكيل')
                            ->suffix('د.ع')
                            ->disabled(fn (Get $get) => ($get('pricing_mode') ?? 'margin') === 'margin')
                            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                if (($get('pricing_mode') ?? 'margin') !== 'direct') {
                                    return;
                                }

                                $cost = self::toInt($get('unit_cost') ?? 0);
                                $price = self::toInt($state);

                                $set('agent_price', self::fmt($price));

                                if ($cost > 0 && $price > 0) {
                                    $margin = (($price - $cost) / $cost) * 100;
                                    $set('agent_margin', rtrim(rtrim(number_format($margin, 2, '.', ''), '0'), '.'));
                                }
                            })
                            ->columnSpan(['default' => 12, 'md' => 4]),
                    ])->columnSpanFull(),
                ]),
            ])
            ->columnSpanFull();
    }

    private static function recalcPricesFromMargins(Set $set, Get $get): void
    {
        if (($get('pricing_mode') ?? 'margin') !== 'margin') {
            return;
        }

        $cost = self::toInt($get('unit_cost') ?? 0);

        if ($cost <= 0) {
            $set('retail_price', self::fmt(0));
            $set('wholesale_price', self::fmt(0));
            $set('agent_price', self::fmt(0));

            return;
        }

        $retailMargin = self::toFloat($get('retail_margin') ?? 0);
        $wholesaleMargin = self::toFloat($get('wholesale_margin') ?? 0);
        $agentMargin = self::toFloat($get('agent_margin') ?? 0);

        $retailPrice = (int) round($cost + ($cost * $retailMargin / 100));
        $wholesalePrice = (int) round($cost + ($cost * $wholesaleMargin / 100));
        $agentPrice = (int) round($cost + ($cost * $agentMargin / 100));

        $set('retail_price', self::fmt($retailPrice));
        $set('wholesale_price', self::fmt($wholesalePrice));
        $set('agent_price', self::fmt($agentPrice));
    }

    /* ================================
     * Hidden Fields
     * ================================ */

    private static function lineTotalHidden(): Hidden
    {
        return Hidden::make('line_total')
            ->default(0)
            ->dehydrateStateUsing(fn ($state) => self::toInt($state));
    }

    private static function variantRowsHidden(): Hidden
    {
        return Hidden::make('variant_rows')
            ->default([])
            ->dehydrated(true)
            ->dehydrateStateUsing(function ($state, Get $get) {
                return self::composeVariantRows($get);
            });
    }

    /* ================================
     * Financial Fields
     * ================================ */

    private static function subtotalField(): TextInput
    {
        return self::moneyDisplay('subtotal', 'المجموع')
            ->columnSpan(['default' => 12, 'md' => 4]);
    }

    private static function discountField(): TextInput
    {
        return self::moneyInput('discount', 'الخصم')
            ->afterStateUpdated(fn ($state, Set $set, Get $get) => self::recalculateTotals($get, $set))
            ->columnSpan(['default' => 12, 'md' => 4]);
    }

    /**
     * ✅ تحديث: أجور النقل تسجيل فقط => لا تعيد حساب الإجمالي عند تغيّرها
     */
    private static function freightField(): TextInput
    {
        return self::moneyInput('freight', 'أجور النقل')
            ->afterStateUpdated(fn () => null)
            ->columnSpan(['default' => 12, 'md' => 4]);
    }

    private static function totalField(): TextInput
    {
        return self::moneyDisplay('total', 'الإجمالي')
            ->columnSpan(['default' => 12, 'md' => 4]);
    }

    private static function paidField(): TextInput
    {
        return self::moneyInput('paid', 'المدفوع')
            ->afterStateUpdated(fn ($state, Set $set, Get $get) => self::recalculateTotals($get, $set))
            ->columnSpan(['default' => 12, 'md' => 4]);
    }

    private static function dueField(): TextInput
    {
        return self::moneyDisplay('due', 'المتبقي')
            ->columnSpan(['default' => 12, 'md' => 4]);
    }

    /* ================================
     * Calculations
     * ================================ */

    private static function syncQtyFromColors(Set $set, Get $get, string $prefix = ''): void
    {
        $rows = $get($prefix.'color_quantities') ?? [];
        if (! is_array($rows) || count($rows) === 0) {
            return;
        }

        $sum = 0;
        foreach ($rows as $row) {
            $sum += max(0, (int) ($row['qty'] ?? 0));
        }

        $set($prefix.'quantity_ordered', max(1, $sum));
    }

    private static function updateLineTotal(Set $set, Get $get, string $prefix = ''): void
    {
        $qty = max(0, (int) ($get($prefix.'quantity_ordered') ?? 0));
        $cost = self::toInt($get($prefix.'unit_cost'));
        $total = $qty * $cost;

        $set($prefix.'line_total', $total);
        $set($prefix.'line_total_display', self::fmt($total));
    }

    /**
     * ✅ الإجمالي بدون freight
     */
    private static function recalculateTotals(Get $get, Set $set, string $prefix = ''): void
    {
        $items = $get($prefix.'items') ?? [];
        $subtotal = 0;

        if (is_array($items)) {
            foreach ($items as $item) {
                $subtotal += self::toInt($item['line_total'] ?? 0);
            }
        }

        $discount = self::toInt($get($prefix.'discount'));
        $paid = self::toInt($get($prefix.'paid'));

        $total = max(0, $subtotal - $discount);
        $due = max(0, $total - $paid);

        $set($prefix.'subtotal', self::fmt($subtotal));
        $set($prefix.'total', self::fmt($total));
        $set($prefix.'due', self::fmt($due));
    }

    /* ================================
     * Variants
     * ================================ */

    private static function buildVariantRows(Set $set, Get $get, string $prefix = ''): void
    {
        $rows = self::composeVariantRows($get, $prefix);
        $set($prefix.'variant_rows', $rows);
    }

    private static function syncMainVariantIdFromVariantRows(Set $set, Get $get, string $prefix = ''): void
    {
        $rows = $get($prefix.'variant_rows') ?? [];

        if (is_array($rows) && ! empty($rows)) {
            $main = (int) ($rows[0]['product_variant_id'] ?? 0);
            if ($main > 0) {
                $set($prefix.'product_variant_id', $main);

                $productId = (int) ($get($prefix.'product_id') ?? 0);
                if ($productId > 0) {
                    self::applyPricingDefaults($set, $productId, $main);
                    self::recalcPricesFromMargins($set, $get);
                }
            }

            return;
        }

        $productId = (int) ($get($prefix.'product_id') ?? 0);
        if ($productId > 0) {
            $vid = self::getOrCreateDefaultVariant($productId);
            $set($prefix.'product_variant_id', $vid);

            self::applyPricingDefaults($set, $productId, $vid);
            self::recalcPricesFromMargins($set, $get);
        }
    }

    private static function composeVariantRows(Get $get, string $prefix = ''): array
    {
        $colorRows = $get($prefix.'color_quantities') ?? [];

        if (is_array($colorRows) && count($colorRows) > 0) {
            $out = [];

            foreach ($colorRows as $row) {
                $variantId = (int) ($row['product_variant_id'] ?? 0);
                $qty = (int) ($row['qty'] ?? 0);
                $colorVal = (int) ($row['color_value_id'] ?? 0);

                if ($variantId <= 0 || $qty <= 0) {
                    continue;
                }

                $out[] = [
                    'product_variant_id' => $variantId,
                    'quantity' => max(1, $qty),
                    'color_value_id' => $colorVal,
                ];
            }

            return array_values($out);
        }

        $productId = (int) ($get($prefix.'product_id') ?? 0);
        if ($productId <= 0) {
            return [];
        }

        $qty = max(1, (int) ($get($prefix.'quantity_ordered') ?? 1));

        $baseRows = $get($prefix.'base_attributes') ?? [];
        $valueIds = [];

        if (is_array($baseRows)) {
            foreach ($baseRows as $r) {
                $v = (int) ($r['attribute_value_id'] ?? 0);
                if ($v > 0) {
                    $valueIds[] = $v;
                }
            }
        }

        $variantId = ! empty($valueIds)
            ? self::findOrCreateVariantByValues($productId, $valueIds)
            : self::getOrCreateDefaultVariant($productId);

        return [[
            'product_variant_id' => $variantId,
            'quantity' => $qty,
            'color_value_id' => 0,
        ]];
    }

    private static function composeVariantRowsFromArray(array $item): array
    {
        $colorRows = $item['color_quantities'] ?? [];
        if (is_array($colorRows) && ! empty($colorRows)) {
            $out = [];
            foreach ($colorRows as $row) {
                $variantId = (int) ($row['product_variant_id'] ?? 0);
                $qty = (int) ($row['qty'] ?? 0);
                $colorVal = (int) ($row['color_value_id'] ?? 0);

                if ($variantId <= 0 || $qty <= 0) {
                    continue;
                }

                $out[] = [
                    'product_variant_id' => $variantId,
                    'quantity' => max(1, $qty),
                    'color_value_id' => $colorVal,
                ];
            }

            return array_values($out);
        }

        $productId = (int) ($item['product_id'] ?? 0);
        if ($productId <= 0) {
            return [];
        }

        $qty = max(1, (int) ($item['quantity_ordered'] ?? 1));

        $baseRows = $item['base_attributes'] ?? [];
        $valueIds = [];

        if (is_array($baseRows)) {
            foreach ($baseRows as $r) {
                $v = (int) ($r['attribute_value_id'] ?? 0);
                if ($v > 0) {
                    $valueIds[] = $v;
                }
            }
        }

        $variantId = ! empty($valueIds)
            ? self::findOrCreateVariantByValues($productId, $valueIds)
            : self::getOrCreateDefaultVariant($productId);

        return [[
            'product_variant_id' => $variantId,
            'quantity' => $qty,
            'color_value_id' => 0,
        ]];
    }

    private static function resolveVariantForColorRow(Get $get, int $colorValueId, string $basePath = '../../'): int
    {
        $productId = (int) ($get($basePath.'product_id') ?? 0);
        if ($productId <= 0 || $colorValueId <= 0) {
            return 0;
        }

        $baseRows = $get($basePath.'base_attributes') ?? [];
        $valueIds = [];

        if (is_array($baseRows)) {
            foreach ($baseRows as $row) {
                $v = (int) ($row['attribute_value_id'] ?? 0);
                if ($v > 0) {
                    $valueIds[] = $v;
                }
            }
        }

        $valueIds[] = $colorValueId;
        $valueIds = array_values(array_unique(array_filter($valueIds)));

        return self::findOrCreateVariantByValues($productId, $valueIds);
    }

    private static function findOrCreateVariantByValues(int $productId, array $attributeValueIds): int
    {
        $ids = collect($attributeValueIds)
            ->filter(fn ($v) => (int) $v > 0)
            ->map(fn ($v) => (int) $v)
            ->unique()
            ->sort()
            ->values()
            ->all();

        $hash = $ids ? hash('sha256', json_encode($ids)) : null;

        $variant = ProductVariant::query()
            ->where('product_id', $productId)
            ->when($hash, fn ($q) => $q->where('options_hash', $hash))
            ->first();

        if (! $variant) {
            $variant = ProductVariant::create([
                'product_id' => $productId,
                'sku' => null,
                'cost' => 0,
                'is_active' => true,
                'is_default' => false,
                'options_hash' => $hash,
            ]);
        }

        if (! empty($ids)) {
            $values = AttributeValue::query()
                ->whereIn('id', $ids)
                ->get(['id', 'attribute_id']);

            $pivot = [];
            foreach ($values as $value) {
                $pivot[$value->id] = ['attribute_id' => (int) $value->attribute_id];
            }

            $variant->attributeValues()->syncWithoutDetaching($pivot);

            $actualIds = $variant->attributeValues()
                ->pluck('attribute_values.id')
                ->sort()
                ->values()
                ->all();

            $variant->options_hash = $actualIds ? hash('sha256', json_encode($actualIds)) : null;
            $variant->saveQuietly();
        }

        return (int) $variant->id;
    }

    private static function getOrCreateDefaultVariant(int $productId): int
    {
        $variantId = (int) (ProductVariant::query()
            ->where('product_id', $productId)
            ->where('is_default', true)
            ->value('id') ?? 0);

        if ($variantId <= 0) {
            $variantId = (int) ProductVariant::create([
                'product_id' => $productId,
                'sku' => null,
                'cost' => 0,
                'is_active' => true,
                'is_default' => true,
                'options_hash' => null,
            ])->id;
        }

        return $variantId;
    }

    private static function rebuildColorVariants(Set $set, Get $get, string $basePath = ''): void
    {
        $rows = $get($basePath.'color_quantities') ?? [];
        if (! is_array($rows) || empty($rows)) {
            return;
        }

        foreach ($rows as $i => $row) {
            $colorId = (int) ($row['color_value_id'] ?? 0);

            $variantId = $colorId > 0
                ? self::resolveVariantForColorRow($get, $colorId, $basePath)
                : 0;

            $set($basePath."color_quantities.{$i}.product_variant_id", $variantId);
        }
    }

    /* ================================
     * Cached Options + Names
     * ================================ */

    private static function colorAttributeId(): ?int
    {
        return Cache::remember('purchase_color_attribute_id', self::CACHE_TTL, function () {
            $id = Attribute::query()
                ->where('is_active', true)
                ->where('type', 'color')
                ->orderBy('sort_order')
                ->value('id');

            return $id ? (int) $id : null;
        });
    }

    private static function baseAttributesOptions(): array
    {
        return Cache::remember('purchase_base_attributes_options', self::CACHE_TTL, function () {
            return Attribute::query()
                ->where('is_active', true)
                ->where(function ($q) {
                    $q->whereNull('type')->orWhere('type', '!=', 'color');
                })
                ->orderBy('sort_order')
                ->pluck('display_name', 'id')
                ->toArray();
        });
    }

    private static function attributeValuesOptions(int $attributeId): array
    {
        return Cache::remember("attr_values_opts_{$attributeId}", self::CACHE_TTL, function () use ($attributeId) {
            return AttributeValue::query()
                ->where('attribute_id', $attributeId)
                ->where('is_active', true)
                ->orderBy('sort_order')
                ->orderBy('id')
                ->limit(self::OPTIONS_LIMIT)
                ->pluck('display_value', 'id')
                ->toArray();
        });
    }

    private static function productName(int $productId): ?string
    {
        return Cache::remember("product_name_{$productId}", self::CACHE_TTL_LONG, function () use ($productId) {
            return Product::query()->whereKey($productId)->value('name');
        });
    }

    private static function attributeName(int $attributeId): string
    {
        return Cache::remember("attr_name_{$attributeId}", self::CACHE_TTL_LONG, function () use ($attributeId) {
            return (string) (Attribute::query()->whereKey($attributeId)->value('display_name') ?? 'خاصية');
        });
    }

    private static function attributeValueName(int $valueId): string
    {
        return Cache::remember("attr_value_name_{$valueId}", self::CACHE_TTL_LONG, function () use ($valueId) {
            return (string) (AttributeValue::query()->whereKey($valueId)->value('display_value') ?? '—');
        });
    }

    private static function generateInvoiceNumber(): string
    {
        $year = now()->year;
        $prefix = "PI-{$year}-";

        $last = PurchaseInvoice::query()
            ->where('invoice_number', 'like', "{$prefix}%")
            ->orderByDesc('id')
            ->value('invoice_number');

        $next = 1;
        if ($last) {
            $n = (int) str_replace($prefix, '', (string) $last);
            $next = max(1, $n + 1);
        }

        return $prefix.str_pad((string) $next, 6, '0', STR_PAD_LEFT);
    }

    /* ================================
     * Create Supplier Form
     * ================================ */

    private static function supplierCreateForm(): array
    {
        return [
            Grid::make(2)->schema([
                TextInput::make('name')
                    ->label('اسم المورد')
                    ->required()
                    ->maxLength(255),
                TextInput::make('phone')
                    ->label('الهاتف')
                    ->mask('99999999999')
                    ->maxLength(14),
                TextInput::make('company')->label('الشركة')->maxLength(255)->columnSpan(2),
                Textarea::make('address')->label('العنوان')->rows(2)->maxLength(500)->columnSpan(2),
                Hidden::make('branch_id')->default(fn () => user_info('branch_id')),
            ]),
        ];
    }
}
