<?php

namespace App\Filament\Resources\Purchases\PurchaseInvoices\Pages;

use App\Filament\Resources\Purchases\PurchaseInvoices\PurchaseInvoiceResource;
use App\Models\Attribute;
use App\Models\ProductPrice;
use App\Services\Purchases\PurchaseInvoice\DeletePurchaseInvoiceService;
use App\Services\Purchases\PurchaseInvoice\UpdatePurchaseInvoiceService;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\EditRecord;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class EditPurchaseInvoice extends EditRecord
{
    protected static string $resource = PurchaseInvoiceResource::class;

    /**
     * ✅ مهم جداً مع Filament v4:
     * تثبيت مسار حالة الفورم حتى لا تضيع repeaters (items)
     */
    protected function getFormStatePath(): ?string
    {
        return 'data';
    }

    protected function mutateFormDataBeforeFill(array $data): array
    {
        /** @var \App\Models\PurchaseInvoice $invoice */
        $invoice = $this->record->load([
            'items.variant.attributeValues',
            'items.variantRows.optionValues',
        ]);

        $colorAttrId = (int) (Attribute::query()
            ->where('is_active', true)
            ->where('type', 'color')
            ->orderBy('sort_order')
            ->value('id') ?? 0);

        $data['items'] = $invoice->items->map(function ($item) use ($colorAttrId) {
            $variant = $item->variant;
            $productId = (int) ($variant?->product_id ?? 0);
            $mainVariantId = (int) $item->product_variant_id;

            // base_attributes: كل القيم ما عدا اللون
            $baseAttributes = [];
            if ($variant && $variant->relationLoaded('attributeValues')) {
                foreach ($variant->attributeValues as $av) {
                    $attrId = (int) ($av->pivot->attribute_id ?? 0);
                    if ($attrId > 0 && $attrId !== $colorAttrId) {
                        $baseAttributes[] = [
                            'attribute_id' => $attrId,
                            'attribute_value_id' => (int) $av->id,
                        ];
                    }
                }
            }

            // color_quantities
            $colorQuantities = [];

            if ($item->relationLoaded('variantRows') && $item->variantRows->count() > 0) {
                foreach ($item->variantRows as $row) {
                    $colorValueId = 0;

                    if ($row->relationLoaded('optionValues')) {
                        foreach ($row->optionValues as $ov) {
                            if ((int) $ov->attribute_id === $colorAttrId) {
                                $colorValueId = (int) $ov->attribute_value_id;
                                break;
                            }
                        }
                    }

                    $colorQuantities[] = [
                        'product_variant_id' => (int) $row->product_variant_id,
                        'color_value_id' => $colorValueId,
                        'qty' => max(1, (int) $row->quantity),
                    ];
                }

                // اجعل الرئيسي أول variantRow إن لم يكن مضبوط
                if ($mainVariantId <= 0) {
                    $mainVariantId = (int) ($colorQuantities[0]['product_variant_id'] ?? 0);
                }
            } else {
                // fallback: استنتاج اللون من variant نفسه
                $colorValueId = 0;

                if ($variant && $variant->relationLoaded('attributeValues')) {
                    foreach ($variant->attributeValues as $av) {
                        $attrId = (int) ($av->pivot->attribute_id ?? 0);
                        if ($attrId === $colorAttrId) {
                            $colorValueId = (int) $av->id;
                            break;
                        }
                    }
                }

                if ($colorValueId > 0) {
                    $colorQuantities[] = [
                        'product_variant_id' => (int) $item->product_variant_id,
                        'color_value_id' => $colorValueId,
                        'qty' => max(1, (int) $item->quantity_ordered),
                    ];
                }
            }

            // ✅ التسعير من product_prices
            $price = null;
            if ($mainVariantId > 0) {
                $price = ProductPrice::query()
                    ->where('product_variant_id', $mainVariantId)
                    ->where('status', ProductPrice::STATUS_ACTIVE)
                    ->orderByDesc('id')
                    ->first();
            }

            return [
                'id' => (int) $item->id,
                'product_id' => $productId,
                'product_variant_id' => $mainVariantId,

                'quantity_ordered' => (int) $item->quantity_ordered,
                'quantity_received' => (int) $item->quantity_received,

                // ✅ مبالغ Int
                'unit_cost' => (int) $item->unit_cost,
                'line_total' => (int) $item->line_total,

                'notes' => $item->notes,

                'base_attributes' => $baseAttributes,
                'color_quantities' => $colorQuantities,

                // ✅ التسعير: المبالغ Int + الهوامش float
                'pricing_mode' => 'margin',
                'retail_margin' => $price ? (float) ($price->retail_margin ?? 0) : 0.0,
                'wholesale_margin' => $price ? (float) ($price->wholesale_margin ?? 0) : 0.0,
                'agent_margin' => $price ? (float) ($price->agent_margin ?? 0) : 0.0,

                'retail_price' => $price ? (int) ($price->retail_price ?? 0) : 0,
                'wholesale_price' => $price ? (int) ($price->wholesale_price ?? 0) : 0,
                'agent_price' => $price ? (int) ($price->agent_price ?? 0) : 0,
            ];
        })->values()->toArray();

        return $data;
    }

    /* =========================
     * Sanitizers (int + float)
     * ========================= */

    protected function cleanInt(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }
        if (is_int($value)) {
            return $value;
        }

        $value = preg_replace_callback(
            '/[٠-٩]/u',
            fn ($m) => (string) mb_strpos('٠١٢٣٤٥٦٧٨٩', $m[0]),
            (string) $value
        );

        $clean = preg_replace('/[^\d]/u', '', (string) $value);

        return (int) ($clean === '' ? 0 : $clean);
    }

    protected function cleanFloat(mixed $value): float
    {
        if ($value === null || $value === '') {
            return 0.0;
        }

        $value = preg_replace_callback(
            '/[٠-٩]/u',
            fn ($m) => (string) mb_strpos('٠١٢٣٤٥٦٧٨٩', $m[0]),
            (string) $value
        );

        $v = preg_replace('/[^\d.]/u', '', (string) $value);

        if (substr_count($v, '.') > 1) {
            $parts = explode('.', $v);
            $v = $parts[0].'.'.implode('', array_slice($parts, 1));
        }

        return (float) ($v === '' ? 0.0 : $v);
    }

    /* =========================
     * Normalization
     * ========================= */

    protected function normalizeFormData(array $data): array
    {
        // ✅ مبالغ مالية Int فقط
        $data['discount'] = $this->cleanInt($data['discount'] ?? 0);
        $data['freight'] = $this->cleanInt($data['freight'] ?? 0);
        $data['paid'] = $this->cleanInt($data['paid'] ?? 0);

        $items = $data['items'] ?? [];

        if (is_array($items)) {
            foreach ($items as $i => $item) {
                $items[$i]['id'] = ! empty($item['id']) ? (int) $item['id'] : null;

                $items[$i]['product_id'] = (int) ($item['product_id'] ?? 0);
                $items[$i]['product_variant_id'] = (int) ($item['product_variant_id'] ?? 0);

                $items[$i]['quantity_ordered'] = max(0, (int) ($item['quantity_ordered'] ?? 0));
                $items[$i]['quantity_received'] = max(0, (int) ($item['quantity_received'] ?? 0));

                // ✅ مبالغ Int فقط
                $items[$i]['unit_cost'] = $this->cleanInt($item['unit_cost'] ?? 0);
                $items[$i]['line_total'] = $this->cleanInt($item['line_total'] ?? 0);

                $items[$i]['retail_price'] = $this->cleanInt($item['retail_price'] ?? 0);
                $items[$i]['wholesale_price'] = $this->cleanInt($item['wholesale_price'] ?? 0);
                $items[$i]['agent_price'] = $this->cleanInt($item['agent_price'] ?? 0);

                // الهوامش float (ليست مبالغ)
                $items[$i]['retail_margin'] = $this->cleanFloat($item['retail_margin'] ?? 0);
                $items[$i]['wholesale_margin'] = $this->cleanFloat($item['wholesale_margin'] ?? 0);
                $items[$i]['agent_margin'] = $this->cleanFloat($item['agent_margin'] ?? 0);

                $items[$i]['base_attributes'] = is_array($item['base_attributes'] ?? null) ? $item['base_attributes'] : [];
                $items[$i]['color_quantities'] = is_array($item['color_quantities'] ?? null) ? $item['color_quantities'] : [];
            }
        }

        $data['items'] = array_values($items);

        return $data;
    }

    /* =========================
     * Validation (UI-level)
     * ========================= */

    protected function assertValidForService(array $data): void
    {
        $errors = [];

        if (empty($data['supplier_id'])) {
            $errors['supplier_id'] = 'المورد مطلوب.';
        }

        if (empty($data['warehouse_id'])) {
            $errors['warehouse_id'] = 'المستودع مطلوب.';
        }

        $items = $data['items'] ?? [];

        if (! is_array($items) || count($items) === 0) {
            $errors['items'] = 'يجب إضافة منتج واحد على الأقل.';
        } else {
            foreach ($items as $i => $item) {
                $productId = (int) ($item['product_id'] ?? 0);
                $variantId = (int) ($item['product_variant_id'] ?? 0);

                if ($productId <= 0) {
                    $errors["items.$i.product_id"] = 'يجب اختيار المنتج.';
                }

                if ((int) ($item['unit_cost'] ?? 0) <= 0) {
                    $errors["items.$i.unit_cost"] = 'سعر الكلفة مطلوب ويجب أن يكون أكبر من 0.';
                }

                $qty = (int) ($item['quantity_ordered'] ?? 0);

                $colorRows = is_array($item['color_quantities'] ?? null) ? $item['color_quantities'] : [];
                $sumColors = 0;
                foreach ($colorRows as $r) {
                    $sumColors += (int) ($r['qty'] ?? 0);
                }

                if ($qty <= 0 && $sumColors <= 0) {
                    $errors["items.$i.quantity_ordered"] = 'الكمية مطلوبة (أو أدخل كميات الألوان).';
                }

                if ($variantId <= 0) {
                    $errors["items.$i.product_variant_id"] = 'تعذر تحديد متغير المنتج. اختر الخواص/الألوان أو تأكد من إعداد المنتج.';
                }
            }
        }

        if (! empty($errors)) {
            throw ValidationException::withMessages($errors);
        }
    }

    protected function mutateFormDataBeforeSave(array $data): array
    {
        $formState = [];

        if (property_exists($this, 'form') && $this->form) {
            if (method_exists($this->form, 'getRawState')) {
                $formState = $this->form->getRawState();
            } else {
                $formState = $this->form->getState();
            }
        }

        // ✅ لو الحالة مغلفة داخل data بسبب getFormStatePath()
        if (is_array($formState) && isset($formState['data']) && is_array($formState['data'])) {
            $formState = $formState['data'];
        }

        // ✅ لو $data نفسه مغلف داخل data (حسب بعض سيناريوهات Filament)
        if (is_array($data) && isset($data['data']) && is_array($data['data'])) {
            $data = $data['data'];
        }

        /**
         * ✅ اجعل formState هو المرجع إذا كان يحتوي items فعلاً
         * (هذا يمنع ضياع repeaters عند الحفظ)
         */
        if (is_array($formState) && array_key_exists('items', $formState)) {
            $data = array_replace($data, $formState);
        } elseif (is_array($formState) && ! empty($formState)) {
            // fallback: دمج عادي بدون كسر items
            $data = array_replace($data, $formState);
        }

        // ✅ تأكيد items دائماً مصفوفة وبمفاتيح رقمية
        $items = $data['items'] ?? null;
        if (! is_array($items)) {
            $items = [];
        }

        // UUID keys -> array_values
        $items = array_values($items);

        /**
         * ✅ لا تحذف العناصر الموجودة مسبقاً:
         * - إذا عنده id => اعتبره عنصر صحيح حتى لو product_id لم يصل لأي سبب
         * - إذا عنصر جديد بدون id => اشترط product_id > 0
         */
        $items = array_values(array_filter($items, function ($row) {
            if (! is_array($row)) {
                return false;
            }

            $id = (int) ($row['id'] ?? 0);
            if ($id > 0) {
                return true;
            }

            return (int) ($row['product_id'] ?? 0) > 0;
        }));

        $data['items'] = $items;

        // ✅ الآن نفّذ normalize + validate
        $data = $this->normalizeFormData($data);
        $this->assertValidForService($data);

        return $data;
    }

    protected function onValidationError(ValidationException $exception): void
    {
        Notification::make()
            ->danger()
            ->title('خطأ في البيانات')
            ->body('يرجى مراجعة الحقول المطلوبة. تم تمييز الحقول التي تحتوي مشاكل.')
            ->send();

        parent::onValidationError($exception);
    }

    protected function handleRecordUpdate(Model $record, array $data): Model
    {
        try {
            /** @var UpdatePurchaseInvoiceService $service */
            $service = app(UpdatePurchaseInvoiceService::class);

            return $service->handle($record, $data);
        } catch (ValidationException $e) {
            $first = collect($e->errors())->flatten()->first();

            Notification::make()
                ->danger()
                ->title('تعذر حفظ الفاتورة')
                ->body($first ?: 'تأكد من الحقول المطلوبة.')
                ->send();

            throw $e;
        } catch (Throwable $e) {
            Log::error('Purchase invoice update failed', [
                'invoice_id' => $record->getKey(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title('حدث خطأ غير متوقع أثناء التحديث')
                ->body('تم تسجيل الخطأ في السجل. إذا تكرر، راجع تفاصيل اللوج.')
                ->persistent()
                ->send();

            throw $e;
        }
    }

    protected function handleRecordDeletion(Model $record): void
    {
        try {
            /** @var DeletePurchaseInvoiceService $service */
            $service = app(DeletePurchaseInvoiceService::class);

            $service->handle($record);
        } catch (ValidationException $e) {
            $first = collect($e->errors())->flatten()->first();

            Notification::make()
                ->danger()
                ->title('تعذر حذف الفاتورة')
                ->body($first ?: 'تأكد من شروط الحذف.')
                ->send();

            throw $e;
        } catch (Throwable $e) {
            Log::error('Purchase invoice delete failed', [
                'invoice_id' => $record->getKey(),
                'error' => $e->getMessage(),
                'trace' => $e->getTraceAsString(),
            ]);

            Notification::make()
                ->danger()
                ->title('حدث خطأ غير متوقع أثناء الحذف')
                ->body('تم تسجيل الخطأ في السجل. إذا تكرر، راجع تفاصيل اللوج.')
                ->persistent()
                ->send();

            throw $e;
        }
    }
}
