<?php

namespace App\Filament\Resources\Purchases\PurchaseInvoices\Pages;

use App\Filament\Resources\Purchases\PurchaseInvoices\PurchaseInvoiceResource;
use App\Services\Purchases\PurchaseInvoice\PurchaseInvoiceService;
use Filament\Notifications\Notification;
use Filament\Resources\Pages\CreateRecord;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\Facades\Log;
use Illuminate\Validation\ValidationException;
use Throwable;

class CreatePurchaseInvoice extends CreateRecord
{
    protected static string $resource = PurchaseInvoiceResource::class;

    /**
     * ✅ مثل صفحة التعديل
     */
    protected function getFormStatePath(): ?string
    {
        return 'data';
    }

    /**
     * ✅ هذا هو الزر الفعلي (حفظ/إضافة)
     * إذا أنت تضغط ولا يحدث شيء، فهذا سيثبت هل الضغط وصل للـ Livewire أو لا.
     */
    public function create(bool $another = false): void
    {
        // علامة حياة فورية
        Notification::make()
            ->title('تم الضغط على زر الحفظ')
            ->info()
            ->send();

        Log::info('PurchaseInvoice Create clicked', [
            'raw' => method_exists($this->form, 'getRawState') ? $this->form->getRawState() : $this->form->getState(),
        ]);

        parent::create($another);
    }

    /* =========================
     * Sanitizers (int + float)
     * ========================= */

    protected function cleanInt(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }
        if (is_int($value)) {
            return $value;
        }

        $value = preg_replace_callback(
            '/[٠-٩]/u',
            fn ($m) => (string) mb_strpos('٠١٢٣٤٥٦٧٨٩', $m[0]),
            (string) $value
        );

        $clean = preg_replace('/[^\d]/u', '', (string) $value);

        return (int) ($clean === '' ? 0 : $clean);
    }

    protected function cleanFloat(mixed $value): float
    {
        if ($value === null || $value === '') {
            return 0.0;
        }

        $value = preg_replace_callback(
            '/[٠-٩]/u',
            fn ($m) => (string) mb_strpos('٠١٢٣٤٥٦٧٨٩', $m[0]),
            (string) $value
        );

        $v = preg_replace('/[^\d.]/u', '', (string) $value);

        if (substr_count($v, '.') > 1) {
            $parts = explode('.', $v);
            $v = $parts[0].'.'.implode('', array_slice($parts, 1));
        }

        return (float) ($v === '' ? 0.0 : $v);
    }

    protected function normalizeFormData(array $data): array
    {
        // مهم: Filament يغلف state داخل data بسبب getFormStatePath
        if (isset($data['data']) && is_array($data['data'])) {
            $data = $data['data'];
        }

        $data['branch_id'] = (int) ($data['branch_id'] ?? user_info('branch_id'));
        $data['user_id'] = (int) user_info('id');

        $data['discount'] = $this->cleanInt($data['discount'] ?? 0);
        $data['freight'] = $this->cleanInt($data['freight'] ?? 0);
        $data['paid'] = $this->cleanInt($data['paid'] ?? 0);

        $items = $data['items'] ?? [];
        $items = is_array($items) ? array_values($items) : [];

        foreach ($items as $i => $item) {
            $items[$i]['id'] = ! empty($item['id']) ? (int) $item['id'] : null;

            $items[$i]['product_id'] = (int) ($item['product_id'] ?? 0);
            $items[$i]['product_variant_id'] = (int) ($item['product_variant_id'] ?? 0);

            $items[$i]['quantity_ordered'] = max(0, (int) ($item['quantity_ordered'] ?? 0));
            $items[$i]['quantity_received'] = max(0, (int) ($item['quantity_received'] ?? 0));

            $items[$i]['unit_cost'] = $this->cleanInt($item['unit_cost'] ?? 0);
            $items[$i]['line_total'] = $this->cleanInt($item['line_total'] ?? 0);

            $items[$i]['retail_price'] = $this->cleanInt($item['retail_price'] ?? 0);
            $items[$i]['wholesale_price'] = $this->cleanInt($item['wholesale_price'] ?? 0);
            $items[$i]['agent_price'] = $this->cleanInt($item['agent_price'] ?? 0);

            $items[$i]['retail_margin'] = $this->cleanFloat($item['retail_margin'] ?? 0);
            $items[$i]['wholesale_margin'] = $this->cleanFloat($item['wholesale_margin'] ?? 0);
            $items[$i]['agent_margin'] = $this->cleanFloat($item['agent_margin'] ?? 0);

            $items[$i]['base_attributes'] = is_array($item['base_attributes'] ?? null) ? $item['base_attributes'] : [];
            $items[$i]['color_quantities'] = is_array($item['color_quantities'] ?? null) ? $item['color_quantities'] : [];

            // qty داخل color_quantities
            foreach ($items[$i]['color_quantities'] as $k => $row) {
                if (! is_array($row)) {
                    continue;
                }

                $items[$i]['color_quantities'][$k]['product_variant_id'] = (int) ($row['product_variant_id'] ?? 0);
                $items[$i]['color_quantities'][$k]['color_value_id'] = (int) ($row['color_value_id'] ?? 0);
                $items[$i]['color_quantities'][$k]['qty'] = max(0, $this->cleanInt($row['qty'] ?? 0));
            }
        }

        $data['items'] = $items;

        return $data;
    }

    protected function assertValidForService(array $data): void
    {
        $errors = [];

        if (empty($data['supplier_id'])) {
            $errors['supplier_id'] = 'المورد مطلوب.';
        }

        if (empty($data['warehouse_id'])) {
            $errors['warehouse_id'] = 'المستودع مطلوب.';
        }

        $items = $data['items'] ?? [];

        if (! is_array($items) || count($items) === 0) {
            $errors['items'] = 'يجب إضافة منتج واحد على الأقل.';
        } else {
            foreach ($items as $i => $item) {
                $productId = (int) ($item['product_id'] ?? 0);
                $variantId = (int) ($item['product_variant_id'] ?? 0);

                if ($productId <= 0) {
                    $errors["items.$i.product_id"] = 'يجب اختيار المنتج.';
                }

                if ((int) ($item['unit_cost'] ?? 0) <= 0) {
                    $errors["items.$i.unit_cost"] = 'سعر الكلفة مطلوب ويجب أن يكون أكبر من 0.';
                }

                $qty = (int) ($item['quantity_ordered'] ?? 0);

                $colorRows = is_array($item['color_quantities'] ?? null) ? $item['color_quantities'] : [];
                $sumColors = 0;
                foreach ($colorRows as $r) {
                    $sumColors += (int) ($r['qty'] ?? 0);
                }

                if ($qty <= 0 && $sumColors <= 0) {
                    $errors["items.$i.quantity_ordered"] = 'الكمية مطلوبة (أو أدخل كميات الألوان).';
                }

                if ($variantId <= 0) {
                    $errors["items.$i.product_variant_id"] = 'تعذر تحديد متغير المنتج.';
                }
            }
        }

        if (! empty($errors)) {
            throw ValidationException::withMessages($errors);
        }
    }

    protected function mutateFormDataBeforeCreate(array $data): array
    {
        // خذ raw state لأنه الأكثر ثباتاً مع repeaters
        $raw = method_exists($this->form, 'getRawState') ? $this->form->getRawState() : $this->form->getState();

        if (is_array($raw) && isset($raw['data']) && is_array($raw['data'])) {
            $raw = $raw['data'];
        }

        // اجعل raw هو المرجع الأساسي
        if (is_array($raw) && ! empty($raw)) {
            $data = array_replace(is_array($data) ? $data : [], $raw);
        }

        $data = $this->normalizeFormData($data);
        $this->assertValidForService($data);

        return $data;
    }

    protected function handleRecordCreation(array $data): Model
    {
        try {
            return app(PurchaseInvoiceService::class)->createInvoice($data);
        } catch (ValidationException $e) {
            $first = collect($e->errors())->flatten()->first();

            Notification::make()
                ->danger()
                ->title('تعذر إنشاء الفاتورة')
                ->body($first ?: 'تأكد من الحقول المطلوبة.')
                ->send();

            throw $e;
        } catch (Throwable $e) {
            report($e);

            Notification::make()
                ->danger()
                ->title('حدث خطأ غير متوقع أثناء الإنشاء')
                ->body('تم تسجيل الخطأ في السجل.')
                ->persistent()
                ->send();

            throw $e;
        }
    }
}
