<?php

namespace App\Filament\Resources\Promotions\Tables;

use App\Enums\PromotionAppliesTo;
use App\Enums\PromotionType;
use App\Models\Promotion;
use Coolsam\Flatpickr\Forms\Components\Flatpickr;
use Filament\Actions\ActionGroup;
use Filament\Actions\BulkAction;
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\EditAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\Filter;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;
use Illuminate\Support\Carbon;

class PromotionsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->label('اسم العرض')
                    ->searchable()
                    ->sortable()
                    ->weight('bold')
                    ->description(fn (Promotion $record) => $record->description ? mb_substr($record->description, 0, 50).'...' : null),

                TextColumn::make('type')
                    ->label('النوع')
                    ->badge()
                    ->formatStateUsing(fn (PromotionType $state) => $state->label())
                    ->icon(fn (PromotionType $state) => $state->icon())
                    ->color(fn (PromotionType $state) => match ($state) {
                        PromotionType::Percentage => 'success',
                        PromotionType::Fixed => 'info',
                        PromotionType::BuyXGetY => 'warning',
                        PromotionType::FreeShipping => 'primary',
                        PromotionType::MinPurchaseDiscount => 'gray',
                    })
                    ->sortable(),

                TextColumn::make('discount_value')
                    ->label('قيمة الخصم')
                    ->formatStateUsing(function (Promotion $record): string {
                        if ($record->type === PromotionType::BuyXGetY) {
                            return "اشترِ {$record->buy_quantity} واحصل على {$record->get_quantity}";
                        }

                        if ($record->type === PromotionType::FreeShipping) {
                            return 'شحن مجاني';
                        }

                        if ($record->type === PromotionType::Percentage) {
                            return $record->discount_value.'%';
                        }

                        return number_format($record->discount_value, 0, '', ',').' د.ع';
                    })
                    ->badge()
                    ->color('success'),

                TextColumn::make('applies_to')
                    ->label('ينطبق على')
                    ->badge()
                    ->formatStateUsing(fn (PromotionAppliesTo $state) => $state->label())
                    ->icon(fn (PromotionAppliesTo $state) => $state->icon())
                    ->color('gray'),

                TextColumn::make('usage_count')
                    ->label('الاستخدام')
                    ->formatStateUsing(function (Promotion $record): string {
                        if ($record->usage_limit) {
                            return $record->usage_count.'/'.$record->usage_limit;
                        }

                        return $record->usage_count.' (غير محدود)';
                    })
                    ->badge()
                    ->color(fn (Promotion $record) => match (true) {
                        $record->usage_limit && $record->usage_count >= $record->usage_limit => 'danger',
                        $record->usage_limit && $record->usage_count >= ($record->usage_limit * 0.8) => 'warning',
                        default => 'gray',
                    })
                    ->alignCenter(),

                IconColumn::make('is_active')
                    ->label('مفعّل')
                    ->boolean()
                    ->alignCenter()
                    ->sortable(),

                TextColumn::make('is_valid')
                    ->label('الحالة')
                    ->badge()
                    ->formatStateUsing(fn (Promotion $record) => $record->is_valid ? 'ساري' : 'غير ساري')
                    ->color(fn (Promotion $record) => $record->is_valid ? 'success' : 'danger')
                    ->icon(fn (Promotion $record) => $record->is_valid ? 'heroicon-o-check-circle' : 'heroicon-o-x-circle'),

                TextColumn::make('starts_at')
                    ->label('يبدأ')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->placeholder('فوري')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('ends_at')
                    ->label('ينتهي')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->placeholder('بدون نهاية')
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('priority')
                    ->label('الأولوية')
                    ->badge()
                    ->color('gray')
                    ->alignCenter()
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('type')
                    ->label('النوع')
                    ->options(PromotionType::options())
                    ->multiple(),

                SelectFilter::make('applies_to')
                    ->label('ينطبق على')
                    ->options(PromotionAppliesTo::options())
                    ->multiple(),

                TernaryFilter::make('is_active')
                    ->label('الحالة')
                    ->placeholder('الكل')
                    ->trueLabel('مفعّل')
                    ->falseLabel('معطّل'),

                Filter::make('valid')
                    ->label('ساري المفعول')
                    ->query(fn (Builder $query) => $query->valid()),

                Filter::make('expired')
                    ->label('منتهي')
                    ->query(fn (Builder $query) => $query
                        ->where('is_active', true)
                        ->where('ends_at', '<', now())),

                Filter::make('date_range')
                    ->form([
                        Flatpickr::make('from_date')
                            ->label('من تاريخ')
                            ->displayFormat('Y-m-d')
                            ->native(false),
                        Flatpickr::make('to_date')
                            ->label('إلى تاريخ')
                            ->displayFormat('Y-m-d')
                            ->native(false),
                    ])
                    ->query(function (Builder $query, array $data): Builder {
                        return $query
                            ->when(
                                $data['from_date'] ?? null,
                                fn (Builder $q, $date) => $q->whereDate('created_at', '>=', Carbon::parse($date))
                            )
                            ->when(
                                $data['to_date'] ?? null,
                                fn (Builder $q, $date) => $q->whereDate('created_at', '<=', Carbon::parse($date))
                            );
                    })
                    ->indicateUsing(function (array $data): array {
                        $indicators = [];
                        if ($data['from_date'] ?? null) {
                            $indicators['from_date'] = 'من: '.$data['from_date'];
                        }
                        if ($data['to_date'] ?? null) {
                            $indicators['to_date'] = 'إلى: '.$data['to_date'];
                        }

                        return $indicators;
                    }),
            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make(),
                    DeleteAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف العرض')
                        ->modalDescription('هل أنت متأكد من حذف هذا العرض؟'),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف العروض المحددة')
                        ->modalDescription('هل أنت متأكد من حذف العروض المحددة؟'),

                    BulkAction::make('activate')
                        ->label('تفعيل')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->requiresConfirmation()
                        ->modalHeading('تفعيل العروض')
                        ->modalDescription('سيتم تفعيل جميع العروض المحددة')
                        ->action(fn ($records) => $records->each->update(['is_active' => true]))
                        ->deselectRecordsAfterCompletion(),

                    BulkAction::make('deactivate')
                        ->label('إلغاء التفعيل')
                        ->icon('heroicon-o-x-circle')
                        ->color('danger')
                        ->requiresConfirmation()
                        ->modalHeading('إلغاء تفعيل العروض')
                        ->modalDescription('سيتم إلغاء تفعيل جميع العروض المحددة')
                        ->action(fn ($records) => $records->each->update(['is_active' => false]))
                        ->deselectRecordsAfterCompletion(),
                ]),
            ])
            ->defaultSort('priority', 'desc')
            ->emptyStateHeading('لا توجد عروض')
            ->emptyStateDescription('ابدأ بإضافة عرض جديد')
            ->emptyStateIcon('heroicon-o-gift')
            ->striped();
    }
}
