<?php

namespace App\Filament\Resources\Promotions\Schemas;

use App\Enums\PromotionAppliesTo;
use App\Enums\PromotionType;
use App\Models\Brand;
use App\Models\Category;
use App\Models\Product;
use Filament\Forms\Components\DateTimePicker;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Schema;
use Illuminate\Support\HtmlString;

class PromotionForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            self::sectionBasicInfo(),
            self::sectionDiscountSettings(),
            self::sectionApplicability(),
            self::sectionUsageLimits(),
            self::sectionSchedule(),
        ]);
    }

    /* ================== Helpers ================== */

    protected static function formatMoney(mixed $value): string
    {
        return number_format((int) $value, 0, '', ',');
    }

    protected static function toInt(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }

        $value = (string) $value;
        $clean = preg_replace('/[^\d]/u', '', $value);

        return (int) ($clean ?: 0);
    }

    protected static function moneyInput(string $name, string $label, bool $required = false): TextInput
    {
        $input = TextInput::make($name)
            ->label($label)
            ->inputMode('numeric')
            ->step(1)
            ->default('0')
            ->live(onBlur: true)
            ->afterStateHydrated(function ($state, callable $set) use ($name) {
                $set($name, self::formatMoney($state));
            })
            ->afterStateUpdated(function ($state, callable $set) use ($name) {
                $set($name, self::formatMoney($state));
            })
            ->dehydrateStateUsing(fn ($state) => self::toInt($state))
            ->rule('regex:/^[\d,]+$/u');

        return $required ? $input->required() : $input;
    }

    /* ================== المعلومات الأساسية ================== */

    protected static function sectionBasicInfo(): Section
    {
        return Section::make('معلومات العرض')
            ->description('البيانات الأساسية للعرض')
            ->icon('heroicon-o-gift')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(2)->schema([
                    TextInput::make('name')
                        ->label('اسم العرض')
                        ->required()
                        ->maxLength(255)
                        ->placeholder('مثال: خصم الصيف 20%')
                        ->prefixIcon('heroicon-o-tag'),

                    TextInput::make('priority')
                        ->label('الأولوية')
                        ->numeric()
                        ->default(0)
                        ->helperText('العروض ذات الأولوية الأعلى تُطبق أولاً')
                        ->prefixIcon('heroicon-o-arrow-trending-up'),

                    Textarea::make('description')
                        ->label('الوصف')
                        ->rows(2)
                        ->placeholder('وصف العرض للعملاء...')
                        ->columnSpanFull(),

                    Toggle::make('is_active')
                        ->label('مفعّل')
                        ->default(true)
                        ->inline(false),

                    Toggle::make('is_stackable')
                        ->label('قابل للتراكم')
                        ->default(false)
                        ->helperText('هل يمكن تطبيق هذا العرض مع عروض أخرى؟')
                        ->inline(false),
                ]),
            ]);
    }

    /* ================== إعدادات الخصم ================== */

    protected static function sectionDiscountSettings(): Section
    {
        return Section::make('إعدادات الخصم')
            ->description('نوع ومقدار الخصم')
            ->icon('heroicon-o-receipt-percent')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    Select::make('type')
                        ->label('نوع العرض')
                        ->options(PromotionType::options())
                        ->required()
                        ->live()
                        ->prefixIcon('heroicon-o-sparkles')
                        ->afterStateUpdated(function ($state, callable $set) {
                            // مسح الحقول غير المطلوبة
                            if ($state !== PromotionType::BuyXGetY->value) {
                                $set('buy_quantity', null);
                                $set('get_quantity', null);
                            }
                        }),

                    TextInput::make('discount_value')
                        ->label(fn (Get $get) => match ($get('type')) {
                            PromotionType::Percentage->value => 'نسبة الخصم %',
                            PromotionType::Fixed->value, PromotionType::MinPurchaseDiscount->value => 'مبلغ الخصم',
                            default => 'قيمة الخصم',
                        })
                        ->numeric()
                        ->required(fn (Get $get) => in_array($get('type'), [
                            PromotionType::Percentage->value,
                            PromotionType::Fixed->value,
                            PromotionType::MinPurchaseDiscount->value,
                        ]))
                        ->visible(fn (Get $get) => in_array($get('type'), [
                            PromotionType::Percentage->value,
                            PromotionType::Fixed->value,
                            PromotionType::MinPurchaseDiscount->value,
                        ]))
                        ->suffix(fn (Get $get) => $get('type') === PromotionType::Percentage->value ? '%' : 'د.ع')
                        ->prefixIcon('heroicon-o-calculator'),

                    TextInput::make('buy_quantity')
                        ->label('اشترِ (كمية)')
                        ->numeric()
                        ->minValue(1)
                        ->visible(fn (Get $get) => $get('type') === PromotionType::BuyXGetY->value)
                        ->required(fn (Get $get) => $get('type') === PromotionType::BuyXGetY->value)
                        ->placeholder('مثال: 2')
                        ->prefixIcon('heroicon-o-shopping-cart'),

                    TextInput::make('get_quantity')
                        ->label('واحصل على (كمية مجانية)')
                        ->numeric()
                        ->minValue(1)
                        ->visible(fn (Get $get) => $get('type') === PromotionType::BuyXGetY->value)
                        ->required(fn (Get $get) => $get('type') === PromotionType::BuyXGetY->value)
                        ->placeholder('مثال: 1')
                        ->prefixIcon('heroicon-o-gift'),

                    self::moneyInput('min_purchase_amount', 'الحد الأدنى للشراء')
                        ->helperText('اترك فارغاً لعدم وجود حد أدنى'),

                    self::moneyInput('max_discount_amount', 'الحد الأقصى للخصم')
                        ->helperText('اترك فارغاً لعدم وجود حد أقصى')
                        ->visible(fn (Get $get) => $get('type') === PromotionType::Percentage->value),
                ]),

                Placeholder::make('type_description')
                    ->hiddenLabel()
                    ->content(function (Get $get): HtmlString {
                        $type = $get('type');
                        if (! $type) {
                            return new HtmlString('');
                        }

                        $enumType = PromotionType::tryFrom($type);
                        if (! $enumType) {
                            return new HtmlString('');
                        }

                        return new HtmlString(
                            '<div class="rounded-lg border-r-4 p-3 bg-blue-50 border-blue-500 text-blue-800 dark:bg-blue-900/40 dark:border-blue-400 dark:text-blue-100">
                                <strong>'.$enumType->label().':</strong> '.$enumType->description().'
                            </div>'
                        );
                    }),
            ]);
    }

    /* ================== نطاق التطبيق ================== */

    protected static function sectionApplicability(): Section
    {
        return Section::make('نطاق التطبيق')
            ->description('تحديد المنتجات التي ينطبق عليها العرض')
            ->icon('heroicon-o-funnel')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(2)->schema([
                    Select::make('applies_to')
                        ->label('ينطبق على')
                        ->options(PromotionAppliesTo::options())
                        ->default(PromotionAppliesTo::All->value)
                        ->required()
                        ->live()
                        ->prefixIcon('heroicon-o-globe-alt'),

                    Select::make('applicable_ids')
                        ->label(fn (Get $get) => match ($get('applies_to')) {
                            PromotionAppliesTo::Product->value => 'المنتجات',
                            PromotionAppliesTo::Category->value => 'التصنيفات',
                            PromotionAppliesTo::Brand->value => 'الماركات',
                            default => 'العناصر',
                        })
                        ->multiple()
                        ->searchable()
                        ->preload()
                        ->visible(fn (Get $get) => $get('applies_to') !== PromotionAppliesTo::All->value)
                        ->options(function (Get $get) {
                            return match ($get('applies_to')) {
                                PromotionAppliesTo::Product->value => Product::query()
                                    ->where('is_active', true)
                                    ->pluck('name', 'id')
                                    ->all(),
                                PromotionAppliesTo::Category->value => Category::query()
                                    ->pluck('name', 'id')
                                    ->all(),
                                PromotionAppliesTo::Brand->value => Brand::query()
                                    ->pluck('name', 'id')
                                    ->all(),
                                default => [],
                            };
                        })
                        ->prefixIcon(fn (Get $get) => match ($get('applies_to')) {
                            PromotionAppliesTo::Product->value => 'heroicon-o-cube',
                            PromotionAppliesTo::Category->value => 'heroicon-o-tag',
                            PromotionAppliesTo::Brand->value => 'heroicon-o-building-storefront',
                            default => 'heroicon-o-squares-2x2',
                        }),
                ]),
            ]);
    }

    /* ================== حدود الاستخدام ================== */

    protected static function sectionUsageLimits(): Section
    {
        return Section::make('حدود الاستخدام')
            ->description('التحكم في عدد مرات استخدام العرض')
            ->icon('heroicon-o-chart-bar')
            ->collapsible()
            ->collapsed()
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    TextInput::make('usage_limit')
                        ->label('الحد الأقصى للاستخدام')
                        ->numeric()
                        ->minValue(1)
                        ->placeholder('غير محدود')
                        ->helperText('اترك فارغاً لعدم وجود حد')
                        ->prefixIcon('heroicon-o-user-group'),

                    TextInput::make('usage_count')
                        ->label('عدد مرات الاستخدام')
                        ->numeric()
                        ->default(0)
                        ->disabled()
                        ->dehydrated(false)
                        ->prefixIcon('heroicon-o-calculator')
                        ->visibleOn('edit'),

                    Placeholder::make('remaining_usage')
                        ->label('الاستخدامات المتبقية')
                        ->content(function (Get $get): string {
                            $limit = $get('usage_limit');
                            $count = $get('usage_count') ?? 0;

                            if (! $limit) {
                                return 'غير محدود';
                            }

                            $remaining = max(0, (int) $limit - (int) $count);

                            return $remaining.' من '.$limit;
                        })
                        ->visibleOn('edit'),
                ]),
            ]);
    }

    /* ================== الجدولة ================== */

    protected static function sectionSchedule(): Section
    {
        return Section::make('جدولة العرض')
            ->description('تحديد فترة صلاحية العرض')
            ->icon('heroicon-o-calendar')
            ->collapsible()
            ->collapsed()
            ->columnSpanFull()
            ->schema([
                Grid::make(2)->schema([
                    DateTimePicker::make('starts_at')
                        ->label('يبدأ في')
                        ->placeholder('فوري')
                        ->helperText('اترك فارغاً للبدء فوراً')
                        ->prefixIcon('heroicon-o-play'),

                    DateTimePicker::make('ends_at')
                        ->label('ينتهي في')
                        ->placeholder('بدون نهاية')
                        ->helperText('اترك فارغاً لعدم وجود نهاية')
                        ->afterOrEqual('starts_at')
                        ->prefixIcon('heroicon-o-stop'),
                ]),

                Placeholder::make('validity_status')
                    ->hiddenLabel()
                    ->content(function (Get $get): HtmlString {
                        $startsAt = $get('starts_at');
                        $endsAt = $get('ends_at');
                        $isActive = $get('is_active');

                        if (! $isActive) {
                            return new HtmlString(
                                '<div class="rounded-lg border-r-4 p-3 bg-gray-100 border-gray-400 text-gray-600 dark:bg-gray-800 dark:text-gray-300">
                                    <span class="font-semibold">⏸️ العرض معطّل</span>
                                </div>'
                            );
                        }

                        $now = now();

                        if ($startsAt && now()->lt($startsAt)) {
                            return new HtmlString(
                                '<div class="rounded-lg border-r-4 p-3 bg-amber-50 border-amber-500 text-amber-800 dark:bg-amber-900/40 dark:text-amber-100">
                                    <span class="font-semibold">⏳ العرض لم يبدأ بعد</span>
                                </div>'
                            );
                        }

                        if ($endsAt && now()->gt($endsAt)) {
                            return new HtmlString(
                                '<div class="rounded-lg border-r-4 p-3 bg-red-50 border-red-500 text-red-800 dark:bg-red-900/40 dark:text-red-100">
                                    <span class="font-semibold">❌ العرض منتهي</span>
                                </div>'
                            );
                        }

                        return new HtmlString(
                            '<div class="rounded-lg border-r-4 p-3 bg-green-50 border-green-500 text-green-800 dark:bg-green-900/40 dark:text-green-100">
                                <span class="font-semibold">✅ العرض ساري المفعول</span>
                            </div>'
                        );
                    })
                    ->visibleOn('edit'),
            ]);
    }
}
