<?php

namespace App\Filament\Resources\Products\Tables;

use App\Models\Product;
use App\Enums\ProductUnit;
use Filament\Tables\Table;
use Filament\Actions\BulkAction;
use Filament\Actions\ViewAction;
use Filament\Actions\EditAction;
use Filament\Actions\ActionGroup;
use Filament\Actions\DeleteAction;
use Filament\Actions\RestoreAction;        // ✅ جديد
use Filament\Actions\ForceDeleteAction;    // ✅ جديد
use Filament\Actions\BulkActionGroup;
use Filament\Actions\DeleteBulkAction;
use Filament\Actions\RestoreBulkAction;
use Filament\Actions\ForceDeleteBulkAction;
use Filament\Tables\Columns\IconColumn;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Filters\TernaryFilter;
use Filament\Tables\Filters\TrashedFilter;
use Illuminate\Contracts\Database\Eloquent\Builder;

class ProductsTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('name')
                    ->label('اسم المنتج')
                    ->searchable()
                    ->sortable()
                    ->weight('bold')
                    ->description(fn(Product $record) => $record->sku ?? null),

                TextColumn::make('category.name')
                    ->label('التصنيف')
                    ->badge()
                    ->searchable()
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('brand.name')
                    ->label('الماركة')
                    ->badge()
                    ->color('info')
                    ->searchable()
                    ->sortable()
                    ->toggleable(),

                IconColumn::make('is_variable')
                    ->label('النوع')
                    ->boolean()
                    ->trueIcon('heroicon-o-squares-2x2')
                    ->falseIcon('heroicon-o-cube')
                    ->trueColor('primary')
                    ->falseColor('gray')
                    ->tooltip(fn($state) => $state ? 'متعدد الخواص' : 'بسيط')
                    ->alignCenter(),

                TextColumn::make('variants_count')
                    ->label('النُسخ')
                    ->counts('variants')
                    ->badge()
                    ->color(fn($state) => $state > 0 ? 'success' : 'gray')
                    ->alignCenter()
                    ->sortable()
                    ->toggleable(),

                TextColumn::make('total_stock')
                    ->label('المخزون الكلي')
                    ->state(fn(Product $record) => $record->total_stock)
                    ->badge()
                    ->color(fn($state) => match (true) {
                        $state === 0 => 'danger',
                        $state <= 10 => 'warning',
                        default => 'success',
                    })
                    ->alignCenter()
                    ->sortable()
                    ->toggleable(),

                IconColumn::make('is_active')
                    ->label('الحالة')
                    ->boolean()
                    ->alignCenter()
                    ->sortable(),

                TextColumn::make('created_at')
                    ->label('تاريخ الإنشاء')
                    ->dateTime('Y-m-d h:i:s A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),

                TextColumn::make('updated_at')
                    ->label('تاريخ التحديث')
                    ->dateTime('Y-m-d h:i:s A')
                    ->sortable()
                    ->toggleable(isToggledHiddenByDefault: true),
            ])
            ->filters([
                SelectFilter::make('category_id')
                    ->label('التصنيف')
                    ->relationship('category', 'name')
                    ->multiple()
                    ->preload(),

                SelectFilter::make('brand_id')
                    ->label('الماركة')
                    ->relationship('brand', 'name')
                    ->multiple()
                    ->preload(),

                TernaryFilter::make('is_variable')
                    ->label('نوع المنتج')
                    ->placeholder('الكل')
                    ->trueLabel('متعدد الخواص')
                    ->falseLabel('بسيط'),

                TernaryFilter::make('is_active')
                    ->label('الحالة')
                    ->placeholder('الكل')
                    ->trueLabel('مفعّل')
                    ->falseLabel('معطّل'),

                TrashedFilter::make(),
            ])
            ->recordActions([
                ActionGroup::make([
                    ViewAction::make(),
                    EditAction::make(),

                    // حذف منطقي عند عدم الحذف
                    DeleteAction::make()
                        ->visible(fn(Product $record) => ! $record->trashed())
                        ->requiresConfirmation()
                        ->modalHeading('حذف المنتج')
                        ->modalDescription('هل أنت متأكد من حذف هذا المنتج؟ سيتم حذف جميع نسخه أيضاً.')
                        ->modalSubmitActionLabel('نعم، احذف'),

                    // استعادة عند كونه محذوفًا
                    RestoreAction::make()
                        ->visible(fn(Product $record) => $record->trashed())
                        ->requiresConfirmation()
                        ->modalHeading('استعادة المنتج')
                        ->modalDescription('سيتم استعادة هذا المنتج وجميع نسخه.')
                        ->modalSubmitActionLabel('نعم، استعد'),

                    // حذف نهائي عند كونه محذوفًا
                    ForceDeleteAction::make()
                        ->visible(fn(Product $record) => $record->trashed())
                        ->color('danger')
                        ->requiresConfirmation()
                        ->modalHeading('حذف نهائي')
                        ->modalDescription('تحذير: سيتم حذف المنتج وكل نسخه نهائيًا ولا يمكن التراجع!')
                        ->modalSubmitActionLabel('حذف نهائي'),
                ]),
            ])
            ->toolbarActions([
                BulkActionGroup::make([
                    DeleteBulkAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف المنتجات المحددة')
                        ->modalDescription('هل أنت متأكد من حذف المنتجات المحددة؟')
                        ->modalSubmitActionLabel('نعم، احذف'),

                    // ✅ الحذف النهائي الجماعي
                    ForceDeleteBulkAction::make()
                        ->requiresConfirmation()
                        ->modalHeading('حذف نهائي للمنتجات المحددة')
                        ->modalDescription('تحذير: سيتم حذف كل المنتجات المحددة نهائيًا ولا يمكن التراجع!')
                        ->modalSubmitActionLabel('حذف نهائي'),

                    RestoreBulkAction::make(),

                    BulkAction::make('activate')
                        ->label('تفعيل')
                        ->icon('heroicon-o-check-circle')
                        ->color('success')
                        ->requiresConfirmation()
                        ->modalHeading('تفعيل المنتجات')
                        ->modalDescription('سيتم تفعيل جميع المنتجات المحددة')
                        ->modalSubmitActionLabel('نعم، فعّل')
                        ->action(fn($records) => $records->each->update(['is_active' => true]))
                        ->deselectRecordsAfterCompletion(),

                    BulkAction::make('deactivate')
                        ->label('إلغاء التفعيل')
                        ->icon('heroicon-o-x-circle')
                        ->color('danger')
                        ->requiresConfirmation()
                        ->modalHeading('إلغاء تفعيل المنتجات')
                        ->modalDescription('سيتم إلغاء تفعيل جميع المنتجات المحددة')
                        ->modalSubmitActionLabel('نعم، ألغِ التفعيل')
                        ->action(fn($records) => $records->each->update(['is_active' => false]))
                        ->deselectRecordsAfterCompletion(),
                ]),
            ])
            ->defaultSort('created_at', 'desc')
            ->emptyStateHeading('لا توجد منتجات')
            ->emptyStateDescription('ابدأ بإضافة منتج جديد')
            ->emptyStateIcon('heroicon-o-cube')
            ->striped();
    }
}
