<?php

namespace App\Filament\Resources\Products\Schemas;

use App\Enums\ProductUnit;
use App\Models\Attribute;
use App\Models\AttributeValue;
use App\Models\BranchWarehouse;
use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\ProductVariant;
use App\Models\StockBalance;
use App\Models\Warehouse;
use Filament\Forms\Components\Placeholder;
use Filament\Forms\Components\Repeater;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\SpatieMediaLibraryFileUpload;
use Filament\Forms\Components\Textarea;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Infolists\Components\TextEntry;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Schema;
use Illuminate\Database\Eloquent\Model;
use Illuminate\Support\HtmlString;

class ProductForm
{
    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            self::sectionBasicInfo(),
            self::sectionMedia(),
            self::sectionSimpleProduct(),
            self::sectionVariantsUnified(),
        ]);
    }

    /* =========================================================
     * Helpers مشتركة
     * ========================================================= */

    protected static function toInt(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }

        $value = (string) $value;

        $value = strtr($value, [
            '٠' => '0',
            '١' => '1',
            '٢' => '2',
            '٣' => '3',
            '٤' => '4',
            '٥' => '5',
            '٦' => '6',
            '٧' => '7',
            '٨' => '8',
            '٩' => '9',
            '۰' => '0',
            '۱' => '1',
            '۲' => '2',
            '۳' => '3',
            '۴' => '4',
            '۵' => '5',
            '۶' => '6',
            '۷' => '7',
            '۸' => '8',
            '۹' => '9',
        ]);

        $clean = preg_replace('/[^\d]/u', '', $value);

        return (int) ($clean ?: 0);
    }

    protected static function formatMoney(mixed $value): string
    {
        return number_format(self::toInt($value), 0, '', ',');
    }

    protected static function moneyInput(string $name, string $label, bool $required = true): TextInput
    {
        $input = TextInput::make($name)
            ->label($label)
            ->inputMode('numeric')
            ->step(1)
            ->default('0')
            ->live(onBlur: true)
            ->afterStateHydrated(function ($state, callable $set) use ($name) {
                $set($name, self::formatMoney($state));
            })
            ->afterStateUpdated(function ($state, callable $set) use ($name) {
                $set($name, self::formatMoney($state));
            })
            ->dehydrateStateUsing(fn ($state) => self::toInt($state))
            ->rule('regex:/^[\d,]+$/u');

        return $required ? $input->required() : $input;
    }

    protected static function cleanDesc(?string $desc, int $max = 60): string
    {
        $desc = trim((string) $desc);
        if ($desc === '') {
            return '';
        }

        $desc = preg_replace("/\s+/u", ' ', $desc);
        if (mb_strlen($desc) > $max) {
            $desc = mb_substr($desc, 0, $max).'…';
        }

        return $desc;
    }

    protected static function productLabel(string $name, ?string $desc): string
    {
        $name = trim($name);
        $d = self::cleanDesc($desc);

        return $d !== '' ? ($name.' — '.$d) : ($name !== '' ? $name : '—');
    }

    /**
     * resolve variant:
     * - simple_ => base variant (defaultVariant) من المنتج
     * - داخل Repeater => نقرأ id من state
     */
    protected static function resolveVariant(Get $get, ?Model $record, string $prefix): ?ProductVariant
    {
        if ($prefix === 'simple_') {
            if ($record instanceof Product) {
                return $record->getBaseVariant();
            }

            return null;
        }

        $variantId = $get('id');
        if ($variantId) {
            return ProductVariant::find($variantId);
        }

        if ($record instanceof ProductVariant) {
            return $record;
        }

        return null;
    }

    protected static function getLatestStockForVariant(ProductVariant $variant): ?StockBalance
    {
        return $variant->stockBalances()->orderByDesc('id')->first();
    }

    protected static function getStockForVariantWarehouse(ProductVariant $variant, ?int $warehouseId): ?StockBalance
    {
        if ($warehouseId && $warehouseId > 0) {
            return $variant->stockBalances()
                ->where('warehouse_id', $warehouseId)
                ->orderByDesc('id')
                ->first();
        }

        return self::getLatestStockForVariant($variant);
    }

    /**
     * ✅ يجلب سعر النسخة أولاً، وإن لم يوجد يرجع سعر المنتج العام (product_variant_id = null)
     */
    protected static function getPriceModelForVariant(ProductVariant $variant): ?ProductPrice
    {
        return $variant->prices()->active()->first()
            ?: ProductPrice::query()
                ->where('product_id', $variant->product_id)
                ->whereNull('product_variant_id')
                ->active()
                ->first();
    }

    /**
     * ✅ تعبئة الأسعار/النسب من product_prices داخل الفورم (بدون كسر إدخالك اليدوي)
     * ✅ مع Flag يمنع التكرار داخل Repeater
     */
    protected static function hydratePricingFromDb(Get $get, callable $set, ?Model $record, string $prefix): void
    {
        $loadedField = "{$prefix}pricing_loaded";

        // لا تعيد التحميل إذا سبق وتم (مهم داخل Repeater)
        if ((string) $get($loadedField) === '1') {
            return;
        }

        // إذا المستخدم كتب شيئاً -> لا نكتب فوقه
        $fields = [
            "{$prefix}retail_margin",
            "{$prefix}wholesale_margin",
            "{$prefix}agent_margin",
            "{$prefix}retail_price",
            "{$prefix}wholesale_price",
            "{$prefix}agent_price",
        ];

        foreach ($fields as $f) {
            $v = $get($f);
            if ($v !== null && $v !== '' && self::toInt($v) > 0) {
                $set($loadedField, '1');

                return;
            }
        }

        $variant = self::resolveVariant($get, $record, $prefix);
        if (! $variant) {
            return;
        }

        $price = self::getPriceModelForVariant($variant);
        if (! $price) {
            $set($loadedField, '1');

            return;
        }

        $set("{$prefix}retail_margin", (float) ($price->retail_margin ?? 0));
        $set("{$prefix}wholesale_margin", (float) ($price->wholesale_margin ?? 0));
        $set("{$prefix}agent_margin", (float) ($price->agent_margin ?? 0));

        $set("{$prefix}retail_price", self::formatMoney((int) ($price->retail_price ?? 0)));
        $set("{$prefix}wholesale_price", self::formatMoney((int) ($price->wholesale_price ?? 0)));
        $set("{$prefix}agent_price", self::formatMoney((int) ($price->agent_price ?? 0)));

        $set($loadedField, '1');
    }

    /* =========================================================
     * 1) المعلومات الأساسية
     * ========================================================= */
    protected static function sectionBasicInfo(): Section
    {
        return Section::make('المعلومات الأساسية')
            ->description('معلومات المنتج الرئيسية')
            ->icon('heroicon-o-information-circle')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(2)->schema([
                    TextInput::make('name')
                        ->label('اسم المنتج')
                        ->required()
                        ->maxLength(255)
                        ->placeholder('مثال: كرسي مكتبي فخم')
                        ->prefixIcon('heroicon-o-cube')
                        ->columnSpan(1),

                    Select::make('unit')
                        ->label('وحدة القياس')
                        ->options(ProductUnit::options())
                        ->required()
                        ->searchable()
                        ->preload()
                        ->prefixIcon('heroicon-o-scale')
                        ->columnSpan(1),

                    Select::make('category_id')
                        ->label('التصنيف')
                        ->relationship('category', 'name')
                        ->searchable()
                        ->preload()
                        ->placeholder('اختر التصنيف')
                        ->prefixIcon('heroicon-o-folder')
                        ->columnSpan(1),

                    Select::make('brand_id')
                        ->label('الماركة')
                        ->relationship('brand', 'name')
                        ->searchable()
                        ->preload()
                        ->placeholder('اختر الماركة')
                        ->prefixIcon('heroicon-o-building-storefront')
                        ->columnSpan(1),

                    Textarea::make('description')
                        ->label('الوصف')
                        ->rows(3)
                        ->placeholder('وصف تفصيلي للمنتج...')
                        ->columnSpanFull(),

                    Grid::make(2)->schema([
                        Toggle::make('is_active')
                            ->label('مفعّل')
                            ->default(true)
                            ->visibleOn('edit')
                            ->inline(false),

                        Toggle::make('is_variable')
                            ->label('منتج متغيّر ؟')
                            ->default(false)
                            ->live()
                            ->helperText('قم بالتفعيل إذا كان للمنتج ألوان أو أحجام مختلفة.')
                            ->inline(true),
                    ])->columnSpan(1),
                ]),
            ]);
    }

    /* =========================================================
     * 2) الصور (Thumbnail + Gallery)
     * ========================================================= */
    protected static function sectionMedia(): Section
    {
        return Section::make('صور المنتج')
            ->description('أضف الصورة الرئيسية ومعرض الصور للمنتج')
            ->icon('heroicon-o-photo')
            ->collapsible()
            ->columnSpanFull()
            ->schema([
                Grid::make(2)->schema([
                    // الصورة الرئيسية (Thumbnail)
                    Section::make('الصورة الرئيسية')
                        ->description('الصورة التي تظهر في قوائم المنتجات')
                        ->icon('heroicon-o-camera')
                        ->compact()
                        ->schema([
                            SpatieMediaLibraryFileUpload::make('thumbnail')
                                ->collection(Product::MEDIA_COLLECTION_THUMBNAIL)
                                ->label('الصورة الرئيسية')
                                ->hiddenLabel()
                                ->image()
                                ->imageResizeMode('cover')
                                ->imageCropAspectRatio('1:1')
                                ->imageResizeTargetWidth('600')
                                ->imageResizeTargetHeight('600')
                                ->maxSize(5120) // 5MB
                                ->acceptedFileTypes(['image/jpeg', 'image/png', 'image/webp'])
                                ->disk('upload')
                                ->visibility('public')
                                ->downloadable()
                                ->openable()
                                ->imagePreviewHeight('200')
                                ->helperText('الحد الأقصى: 5MB | الأنواع المسموحة: JPG, PNG, WebP'),
                        ])
                        ->columnSpan(1),

                    // معرض الصور (Gallery)
                    Section::make('معرض الصور')
                        ->description('صور إضافية للمنتج')
                        ->icon('heroicon-o-squares-2x2')
                        ->compact()
                        ->schema([
                            SpatieMediaLibraryFileUpload::make('gallery')
                                ->collection(Product::MEDIA_COLLECTION_GALLERY)
                                ->label('صور المعرض')
                                ->hiddenLabel()
                                ->image()
                                ->multiple()
                                ->reorderable()
                                ->maxFiles(10)
                                ->maxSize(5120) // 5MB per file
                                ->acceptedFileTypes(['image/jpeg', 'image/png', 'image/webp'])
                                ->disk('upload')
                                ->visibility('public')
                                ->downloadable()
                                ->openable()
                                ->imagePreviewHeight('150')
                                ->panelLayout('grid')
                                ->helperText('الحد الأقصى: 10 صور | 5MB لكل صورة'),
                        ])
                        ->columnSpan(1),
                ]),
            ]);
    }

    /* =========================================================
     * 3) منتج بسيط
     * ========================================================= */
    protected static function sectionSimpleProduct(): Section
    {
        return Section::make('بيانات المنتج (بسيط)')
            ->description('يُستخدم عندما يكون المنتج بدون متغيرات')
            ->icon('heroicon-o-cube')
            ->collapsible()
            ->collapsed(false)
            ->columnSpanFull()
            ->visible(fn (Get $get) => ! (bool) $get('is_variable'))
            ->schema([
                Placeholder::make('simple_hint')
                    ->hiddenLabel()
                    ->content(new HtmlString(<<<'HTML'
                        <div class="rounded-lg border-r-4 p-3 bg-emerald-50 border-emerald-500 text-emerald-800 dark:bg-emerald-900/40 dark:border-emerald-400 dark:text-emerald-100">
                            <strong class="font-semibold">💡 منتج بسيط:</strong>
                            <ul class="mt-2 list-disc pr-5 space-y-1 text-sm">
                                <li>سيتم إنشاء <b>نسخة افتراضية واحدة</b> لهذا المنتج.</li>
                                <li>يمكنك تحديد <b>التكلفة، الأسعار، والكمية الأولية</b>.</li>
                                <li>سيتم إنشاء رصيد في المستودع الافتراضي تلقائيًا.</li>
                            </ul>
                        </div>
                        HTML))
                    ->columnSpanFull(),

                Grid::make(4)->schema([
                    Toggle::make('simple_is_active')
                        ->label('مفعّل')
                        ->default(true)
                        ->inline(false)
                        ->visibleOn('edit')
                        ->columnSpan(2),
                ])->columnSpanFull(),

                self::stockSection('simple_', true),
                self::pricingSection('simple_'),
            ]);
    }

    /* =========================================================
     * 3) المتغيرات (منتج متغيّر)
     * ========================================================= */
    protected static function sectionVariantsUnified(): Section
    {
        return Section::make('متغيرات المنتج')
            ->description('أضف أو عدّل نسخ المنتج وخصائصها')
            ->icon('heroicon-o-squares-2x2')
            ->collapsible()
            ->collapsed(false)
            ->columnSpanFull()
            ->visible(fn (Get $get) => (bool) $get('is_variable'))
            ->schema([
                Placeholder::make('hint')
                    ->hiddenLabel()
                    ->content(new HtmlString(<<<'HTML'
                        <div class="rounded-lg border-r-4 p-3 bg-sky-50 border-sky-500 text-sky-800 dark:bg-sky-900/40 dark:border-sky-400 dark:text-sky-100">
                            <strong class="font-semibold">💡 منتج متغيّر:</strong>
                            <ul class="mt-2 list-disc pr-5 space-y-1 text-sm">
                                <li>أضف نسخة لكل تركيبة (مثال: لون + مقاس).</li>
                                <li>اختر الخصائص والقيم لكل نسخة.</li>
                                <li>أدخل التكلفة، الأسعار، والمخزون لكل نسخة.</li>
                            </ul>
                        </div>
                        HTML))
                    ->visible(fn ($get, $record) => $record === null)
                    ->columnSpanFull(),

                Repeater::make('variants')
                    ->relationship('variableVariants') // ✅ فلترة النسخة الافتراضية من الموديل
                    ->saveRelationshipsUsing(fn () => null)
                    ->dehydrated(true)
                    ->label('النسخ')
                    ->hiddenLabel()
                    ->collapsible()
                    ->collapsed(true)
                    ->defaultItems(fn (?Model $record) => $record ? 0 : 1)
                    ->addActionLabel('+ إضافة نسخة')
                    ->itemLabel(function (array $state): ?string {
                        $variantName = trim((string) ($state['variant_label'] ?? $state['full_name'] ?? ''));

                        if ($variantName === '') {
                            $pairs = collect();

                            if (! empty($state['variant_attributes_rel']) && is_array($state['variant_attributes_rel'])) {
                                foreach ($state['variant_attributes_rel'] as $row) {
                                    $attrId = $row['attribute_id'] ?? null;
                                    $valId = $row['id'] ?? null;

                                    $attrName = $attrId
                                        ? Attribute::query()->whereKey($attrId)->value('display_name')
                                        : null;

                                    $valText = $valId
                                        ? AttributeValue::query()->whereKey($valId)->value('display_value')
                                        : null;

                                    if ($attrName && $valText) {
                                        $pairs->push("{$attrName} {$valText}");
                                    }
                                }
                            }

                            if ($pairs->isEmpty() && ! empty($state['variant_attributes']) && is_array($state['variant_attributes'])) {
                                foreach ($state['variant_attributes'] as $row) {
                                    $a = $row['attribute_name'] ?? null;
                                    $v = $row['attribute_value_display'] ?? null;
                                    if ($a && $v) {
                                        $pairs->push("{$a} {$v}");
                                    }
                                }
                            }

                            $variantName = trim((string) $pairs->join(' - '));
                        }

                        if ($variantName === '') {
                            $id = (int) ($state['id'] ?? 0);
                            if ($id > 0) {
                                $v = ProductVariant::query()->find($id);
                                $variantName = trim((string) ($v?->variant_label ?? $v?->full_name ?? $v?->sku ?? ''));
                            }
                        }

                        return $variantName !== '' ? ('نسخة - '.$variantName) : 'نسخة';
                    })
                    ->schema([
                        Grid::make(4)->schema([
                            Toggle::make('is_active')
                                ->label('مفعّل')
                                ->default(true)
                                ->visibleOn('edit')
                                ->inline(false),

                            Toggle::make('is_default')
                                ->label('النسخة الافتراضية')
                                ->inline(true)
                                ->visible(false),
                        ])->columnSpanFull(),

                        self::attributesSection(),
                        self::stockSection('', false),
                        self::pricingSection(),
                    ]),
            ]);
    }

    /* =========================================================
     * 4) قسم الخصائص
     * ========================================================= */
    protected static function attributesSection(): Section
    {
        return Section::make('الخصائص')
            ->description('اختر الخاصية ثم القيمة لكل نسخة')
            ->icon('heroicon-o-swatch')
            ->collapsible()
            ->collapsed(false)
            ->columnSpanFull()
            ->schema([
                Repeater::make('variant_attributes')
                    ->label('المتغيرات')
                    ->hiddenLabel()
                    ->dehydrated(true)
                    ->reorderableWithButtons()
                    ->collapsible()
                    ->visible(fn ($get, $record) => $record === null)
                    ->itemLabel(fn (array $state): string => ($state['attribute_name'] ?? null) ? $state['attribute_name'] : 'خاصية')
                    ->schema([
                        Grid::make(2)->schema([
                            Select::make('attribute_id')
                                ->label('الخاصية')
                                ->options(fn () => cache()->remember(
                                    'attr_options',
                                    300,
                                    fn () => Attribute::orderBy('sort_order')->pluck('display_name', 'id')->all()
                                ))
                                ->searchable()
                                ->required()
                                ->live()
                                ->distinct()
                                ->disableOptionsWhenSelectedInSiblingRepeaterItems()
                                ->prefixIcon('heroicon-o-tag')
                                ->afterStateUpdated(function ($state, callable $set) {
                                    $attr = Attribute::find($state);
                                    $set('attribute_name', $attr?->display_name);
                                    $set('attribute_value_id', null);
                                    $set('attribute_value_display', null);
                                }),

                            Select::make('attribute_value_id')
                                ->label('القيمة')
                                ->options(function (Get $get) {
                                    $attrId = $get('attribute_id');
                                    if (! $attrId) {
                                        return [];
                                    }

                                    return cache()->remember(
                                        "attr_values_{$attrId}",
                                        300,
                                        fn () => AttributeValue::where('attribute_id', $attrId)
                                            ->where('is_active', true)
                                            ->orderBy('sort_order')
                                            ->pluck('display_value', 'id')
                                            ->all()
                                    );
                                })
                                ->searchable()
                                ->required()
                                ->live()
                                ->disabled(fn (Get $get) => ! $get('attribute_id'))
                                ->prefixIcon('heroicon-o-check-circle')
                                ->afterStateUpdated(function ($state, callable $set) {
                                    $val = AttributeValue::find($state);
                                    $set('attribute_value_display', $val?->display_value);
                                }),
                        ]),

                        TextInput::make('attribute_name')->hidden()->dehydrated(false),
                        TextInput::make('attribute_value_display')->hidden()->dehydrated(false),
                    ]),

                Repeater::make('variant_attributes_rel')
                    ->label('الخصائص')
                    ->hiddenLabel()
                    ->relationship('attributeValues')
                    ->visible(fn ($get, $record) => $record !== null)
                    ->schema([
                        Grid::make(2)->schema([
                            Select::make('attribute_id')
                                ->label('الخاصية')
                                ->options(fn () => Attribute::orderBy('sort_order')->pluck('display_name', 'id'))
                                ->required()
                                ->searchable(),

                            Select::make('id')
                                ->label('القيمة')
                                ->required()
                                ->searchable()
                                ->options(function (Get $get) {
                                    $attrId = $get('attribute_id');
                                    if (! $attrId) {
                                        return [];
                                    }

                                    return AttributeValue::where('attribute_id', $attrId)
                                        ->where('is_active', true)
                                        ->pluck('display_value', 'id');
                                }),
                        ]),
                    ]),
            ]);
    }

    /* =========================================================
     * 5) إدارة المخزون + التكلفة
     * ========================================================= */
    protected static function stockSection(string $prefix = '', bool $isSimple = false): Section
    {
        $skuField = $isSimple ? 'simple_sku' : 'sku';
        $barcodeField = $isSimple ? 'simple_manual_barcode' : 'manual_barcode';
        $costField = "{$prefix}initial_unit_cost";

        return Section::make('إدارة المخزون')
            ->description('إعدادات المخزون والتكلفة والتنبيهات')
            ->icon('heroicon-o-archive-box')
            ->collapsible()
            ->collapsed(false)
            ->columnSpanFull()
            ->schema([
                Grid::make(3)->schema([
                    Select::make("{$prefix}warehouse_id")
                        ->label('المستودع')
                        ->options(function () {
                            $branchId = user_info('branch_id');

                            $warehouseIds = BranchWarehouse::query()
                                ->where('branch_id', $branchId)
                                ->pluck('warehouse_id');

                            return Warehouse::query()
                                ->whereIn('id', $warehouseIds)
                                ->orderBy('name')
                                ->pluck('name', 'id');
                        })
                        ->preload()
                        ->required()
                        ->searchable()
                        ->live()
                        ->prefixIcon('heroicon-o-building-office-2')
                        ->afterStateHydrated(function ($state, callable $set, Get $get, ?Model $record) use ($prefix) {
                            if ($state !== null && $state !== '') {
                                return;
                            }

                            $variant = self::resolveVariant($get, $record, $prefix);
                            if (! $variant) {
                                return;
                            }

                            $stock = self::getLatestStockForVariant($variant);
                            if ($stock) {
                                $set("{$prefix}warehouse_id", $stock->warehouse_id);
                            }

                            $wid = (int) ($stock->warehouse_id ?? 0);
                            $stockRow = self::getStockForVariantWarehouse($variant, $wid);
                            $on = (int) ($stockRow->on_hand ?? 0);
                            $res = (int) ($stockRow->reserved_qty ?? 0);
                            $set("{$prefix}stock_on_hand", $on);
                            $set("{$prefix}stock_reserved", $res);
                            $set("{$prefix}stock_available", max(0, $on - $res));
                        })
                        ->afterStateUpdated(function ($state, callable $set, Get $get, ?Model $record) use ($prefix) {
                            $variant = self::resolveVariant($get, $record, $prefix);
                            if (! $variant) {
                                return;
                            }

                            $wid = (int) ($state ?? 0);
                            $s = self::getStockForVariantWarehouse($variant, $wid);
                            $on = (int) ($s->on_hand ?? 0);
                            $res = (int) ($s->reserved_qty ?? 0);
                            $set("{$prefix}stock_on_hand", $on);
                            $set("{$prefix}stock_reserved", $res);
                            $set("{$prefix}stock_available", max(0, $on - $res));
                        }),

                    TextInput::make("{$prefix}stock_on_hand")
                        ->label('الموجود بالمخزن حاليا')
                        ->disabled()
                        ->dehydrated(false)
                        ->default(0)
                        ->suffix('قطعة')
                        ->prefixIcon('heroicon-o-cube'),

                    TextInput::make("{$prefix}stock_available")
                        ->label('الكمية المتوفرة')
                        ->disabled()
                        ->dehydrated(false)
                        ->default(0)
                        ->suffix('قطعة')
                        ->prefixIcon('heroicon-o-check-badge')
                        ->helperText('الكمية المتوفرة = الموجود - المحجوز'),

                    TextInput::make("{$prefix}stock_reserved")
                        ->label('المحجوز للطلبات')
                        ->disabled()
                        ->dehydrated(false)
                        ->default(0)
                        ->suffix('قطعة')
                        ->prefixIcon('heroicon-o-lock-closed'),

                    self::moneyInput($costField, 'تكلفة الوحدة (د.ع)')
                        ->prefixIcon('heroicon-o-calculator')
                        ->afterStateHydrated(function ($state, callable $set, Get $get, ?Model $record) use ($costField, $prefix) {
                            if ($state !== null && $state !== '') {
                                $set($costField, self::formatMoney($state));

                                return;
                            }

                            $variant = self::resolveVariant($get, $record, $prefix);
                            if (! $variant) {
                                return;
                            }

                            $stock = self::getLatestStockForVariant($variant);
                            $cost = $stock->last_unit_cost ?? $variant->cost ?? 0;

                            if ($cost > 0) {
                                $set($costField, self::formatMoney($cost));
                            }
                        })
                        ->dehydrateStateUsing(fn ($state) => self::toInt($state)),

                    TextInput::make($skuField)
                        ->label('SKU')
                        ->maxLength(100)
                        ->placeholder('مثال: PRD-0001')
                        ->prefixIcon('heroicon-o-tag')
                        ->visibleOn('edit'),

                    TextInput::make($barcodeField)
                        ->label('الباركود (اختياري)')
                        ->placeholder('12 أو 13 رقم')
                        ->rule('nullable')
                        ->rule('digits_between:12,13')
                        ->prefixIcon('heroicon-o-qr-code')
                        ->live(onBlur: true),

                    TextInput::make("{$prefix}initial_stock")
                        ->label('الكمية الأولية')
                        ->default(0)
                        ->minValue(0)
                        ->suffix('قطعة')
                        ->prefixIcon('heroicon-o-cube')
                        ->placeholder('كمية المنتج التي ستدخل إلى المخزن')
                        ->extraAttributes(['inputmode' => 'numeric', 'dir' => 'rtl']),

                    TextInput::make("{$prefix}min_stock_level")
                        ->label('حد التنبيه (الحد الأدنى)')
                        ->default(0)
                        ->minValue(0)
                        ->suffix('قطعة')
                        ->extraAttributes(['inputmode' => 'numeric', 'dir' => 'rtl']),
                ])->columnSpanFull(),
            ]);
    }

    /* =========================================================
     * 6) التسعير
     * ========================================================= */

    protected static function calcPriceFromMargin(Get $get, string $prefix, float $margin): ?int
    {
        $rawCost = $get("{$prefix}initial_unit_cost");
        $cost = self::toInt($rawCost);

        if ($cost <= 0) {
            return null;
        }

        if ($margin === 0.0) {
            return $cost;
        }

        return (int) round($cost * (1 + $margin / 100));
    }

    protected static function calcMarginFromPrice(Get $get, string $prefix, int $price): float
    {
        $rawCost = $get("{$prefix}initial_unit_cost");
        $cost = self::toInt($rawCost);

        if ($cost <= 0 || $price <= 0) {
            return 0.0;
        }

        if ($price === $cost) {
            return 0.0;
        }

        return (($price - $cost) / $cost) * 100;
    }

    protected static function pricingSection(string $prefix = ''): Section
    {
        $retailPriceField = "{$prefix}retail_price";
        $wholesalePriceField = "{$prefix}wholesale_price";
        $agentPriceField = "{$prefix}agent_price";

        $loadedField = "{$prefix}pricing_loaded";

        return Section::make('التسعير')
            ->description('حدد أسعار البيع لأنواع العملاء الثلاثة')
            ->icon('heroicon-o-currency-dollar')
            ->collapsible()
            ->collapsed(false)
            ->columnSpanFull()
            ->schema([
                // ✅ Flag (مخفي) يمنع تكرار التحميل داخل Repeater
                TextInput::make($loadedField)
                    ->hidden()
                    ->dehydrated(false)
                    ->default('0'),

                TextEntry::make("{$prefix}pricing_hint")
                    ->hiddenLabel()
                    ->state(new HtmlString(<<<'HTML'
                        <div
                            x-data="{ show: localStorage.getItem('hide_pricing_hint') !== '1' }"
                            x-show="show"
                            x-cloak
                            class="rounded-lg border-r-4 p-3 bg-amber-50 border-amber-500 text-amber-900 dark:bg-amber-900/40 dark:border-amber-400 dark:text-amber-100 relative"
                        >
                            <button
                                type="button"
                                class="absolute left-3 top-3 text-xs underline hover:opacity-75"
                                @click="
                                    show = false;
                                    localStorage.setItem('hide_pricing_hint', '1');
                                "
                            >
                                لا تُظهر هذه النصائح مرة أخرى
                            </button>

                            <strong class="font-semibold">💰 نظام التسعير:</strong>
                            <ul class="mt-2 list-disc pr-5 space-y-1 text-sm">
                                <li>المعادلة: <b>سعر البيع = تكلفة الوحدة × (1 + نسبة الربح ÷ 100)</b>.</li>
                                <li>عند إدخال <b>0%</b> ربح يتم جعل <b>سعر البيع = تكلفة الوحدة</b> تلقائياً.</li>
                                <li>كل المبالغ أعداد صحيحة (مناسبة للـدينار العراقي).</li>
                            </ul>
                        </div>
                    HTML))
                    ->columnSpanFull(),

                Grid::make(3)->schema([
                    Section::make('سعر المفرد')
                        ->icon('heroicon-o-user')
                        ->columns(1)
                        ->schema([
                            TextInput::make("{$prefix}retail_margin")
                                ->label('نسبة الربح %')
                                ->numeric()
                                ->default(0)
                                ->suffix('%')
                                ->placeholder('مثال: 20')
                                ->dehydrated(true)
                                ->dehydrateStateUsing(fn ($state) => (float) ($state ?? 0))

                                // ✅ هنا التحميل الحقيقي (بدل Placeholder المخفي)
                                ->afterStateHydrated(function ($state, callable $set, Get $get, ?Model $record) use ($prefix) {
                                    self::hydratePricingFromDb($get, $set, $record, $prefix);
                                })

                                ->live(onBlur: true)
                                ->afterStateUpdated(function ($state, callable $set, Get $get) use ($prefix, $retailPriceField) {
                                    if ($state === null || $state === '') {
                                        return;
                                    }

                                    $margin = (float) $state;
                                    $price = self::calcPriceFromMargin($get, $prefix, $margin);

                                    if ($price !== null) {
                                        $set($retailPriceField, self::formatMoney($price));
                                    }
                                }),

                            self::moneyInput($retailPriceField, 'سعر البيع المباشر')
                                ->dehydrated(true)
                                ->prefixIcon('heroicon-o-banknotes')
                                ->afterStateUpdated(function ($state, callable $set, Get $get) use ($prefix, $retailPriceField) {
                                    $price = self::toInt($state);
                                    $margin = self::calcMarginFromPrice($get, $prefix, $price);

                                    $set("{$prefix}retail_margin", round($margin, 2));
                                    $set($retailPriceField, self::formatMoney($price));
                                }),
                        ]),

                    Section::make('سعر الجملة')
                        ->icon('heroicon-o-building-storefront')
                        ->columns(1)
                        ->schema([
                            TextInput::make("{$prefix}wholesale_margin")
                                ->label('نسبة الربح %')
                                ->numeric()
                                ->default(0)
                                ->suffix('%')
                                ->placeholder('مثال: 10')
                                ->dehydrated(true)
                                ->dehydrateStateUsing(fn ($state) => (float) ($state ?? 0))
                                ->live(onBlur: true)
                                ->afterStateUpdated(function ($state, callable $set, Get $get) use ($prefix, $wholesalePriceField) {
                                    if ($state === null || $state === '') {
                                        return;
                                    }

                                    $margin = (float) $state;
                                    $price = self::calcPriceFromMargin($get, $prefix, $margin);

                                    if ($price !== null) {
                                        $set($wholesalePriceField, self::formatMoney($price));
                                    }
                                }),

                            self::moneyInput($wholesalePriceField, 'سعر البيع المباشر')
                                ->dehydrated(true)
                                ->prefixIcon('heroicon-o-banknotes')
                                ->afterStateUpdated(function ($state, callable $set, Get $get) use ($prefix, $wholesalePriceField) {
                                    $price = self::toInt($state);
                                    $margin = self::calcMarginFromPrice($get, $prefix, $price);

                                    $set("{$prefix}wholesale_margin", round($margin, 2));
                                    $set($wholesalePriceField, self::formatMoney($price));
                                }),
                        ]),

                    Section::make('سعر الوكيل')
                        ->icon('heroicon-o-briefcase')
                        ->columns(1)
                        ->schema([
                            TextInput::make("{$prefix}agent_margin")
                                ->label('نسبة الربح %')
                                ->numeric()
                                ->default(0)
                                ->suffix('%')
                                ->placeholder('مثال: 0 أو 5')
                                ->dehydrated(true)
                                ->dehydrateStateUsing(fn ($state) => (float) ($state ?? 0))
                                ->live(onBlur: true)
                                ->afterStateUpdated(function ($state, callable $set, Get $get) use ($prefix, $agentPriceField) {
                                    if ($state === null || $state === '') {
                                        return;
                                    }

                                    $margin = (float) $state;
                                    $price = self::calcPriceFromMargin($get, $prefix, $margin);

                                    if ($price !== null) {
                                        $set($agentPriceField, self::formatMoney($price));
                                    }
                                }),

                            self::moneyInput($agentPriceField, 'سعر البيع المباشر')
                                ->dehydrated(true)
                                ->prefixIcon('heroicon-o-banknotes')
                                ->afterStateUpdated(function ($state, callable $set, Get $get) use ($prefix, $agentPriceField) {
                                    $price = self::toInt($state);
                                    $margin = self::calcMarginFromPrice($get, $prefix, $price);

                                    $set("{$prefix}agent_margin", round($margin, 2));
                                    $set($agentPriceField, self::formatMoney($price));
                                }),
                        ]),
                ]),
            ]);
    }
}
