<?php

namespace App\Filament\Resources\Products\Pages;

use App\Filament\Resources\Products\ProductResource;
use App\Models\Product;
use App\Models\ProductPrice;
use App\Models\StockBalance;
use App\Services\Products\ProductService;
use Filament\Actions\DeleteAction;
use Filament\Resources\Pages\EditRecord;
use Illuminate\Database\Eloquent\Model;

class EditProduct extends EditRecord
{
    protected static string $resource = ProductResource::class;

    protected function getHeaderActions(): array
    {
        return [
            DeleteAction::make(),
        ];
    }

    /**
     * تعبئة بيانات الفورم عند فتح صفحة التعديل
     * - للمنتج البسيط: تملأ حقول simple_*
     * - للمنتج المتغيّر: تملأ مصفوفة variants لكل نسخة (variableVariants فقط)
     */
    protected function mutateFormDataBeforeFill(array $data): array
    {
        /** @var Product $product */
        $product = $this->getRecord();

        if (! $product) {
            return $data;
        }

        /* ===================== منتج بسيط ===================== */
        if (! $product->is_variable) {
            $variant = $product->getBaseVariant();

            if (! $variant) {
                return $data;
            }

            $data['simple_is_active'] = $variant->is_active;
            $data['simple_sku'] = $variant->sku;
            $data['simple_manual_barcode'] = $variant->primary_barcode;

            // المخزون (أحدث رصيد)
            $stock = StockBalance::query()
                ->where('product_variant_id', $variant->id)
                ->orderByDesc('id')
                ->first();

            if ($stock) {
                $data['simple_warehouse_id'] = $stock->warehouse_id;
                $data['simple_initial_unit_cost'] = $stock->last_unit_cost ?? 0;
                $data['simple_initial_stock'] = (int) $stock->on_hand;
                $data['simple_min_stock_level'] = (int) $stock->min_stock_level;
                $data['simple_max_stock_level'] = (int) $stock->max_stock_level;
                $data['simple_reorder_point'] = (int) $stock->reorder_point;
                $data['simple_reorder_quantity'] = (int) $stock->reorder_quantity;
            }

            // الأسعار (active للنسخة، ثم fallback لسعر المنتج العام)
            $price = $variant->prices()->active()->first()
                ?: ProductPrice::query()
                    ->where('product_id', $product->id)
                    ->whereNull('product_variant_id')
                    ->active()
                    ->first();

            if ($price) {
                $data['simple_retail_margin'] = (float) $price->retail_margin;
                $data['simple_retail_price'] = (int) $price->retail_price;
                $data['simple_wholesale_margin'] = (float) $price->wholesale_margin;
                $data['simple_wholesale_price'] = (int) $price->wholesale_price;
                $data['simple_agent_margin'] = (float) $price->agent_margin;
                $data['simple_agent_price'] = (int) $price->agent_price;
            }

            return $data;
        }

        /* ===================== منتج متغيّر ===================== */
        $data['variants'] = $product->variableVariants // ✅ مهم جداً
            ->map(function ($variant) use ($product) {
                $row = [
                    'id' => $variant->id,
                    'sku' => $variant->sku,
                    'is_active' => (bool) $variant->is_active,
                    'is_default' => (bool) $variant->is_default,
                    'manual_barcode' => $variant->primary_barcode,
                ];

                // المخزون (أحدث رصيد)
                $stock = $variant->stockBalances()
                    ->orderByDesc('id')
                    ->first();

                if ($stock) {
                    $row['warehouse_id'] = (int) $stock->warehouse_id;
                    $row['initial_unit_cost'] = (int) ($stock->last_unit_cost ?? 0);
                    $row['initial_stock'] = (int) $stock->on_hand;
                    $row['min_stock_level'] = (int) $stock->min_stock_level;
                    $row['max_stock_level'] = (int) $stock->max_stock_level;
                    $row['reorder_point'] = (int) $stock->reorder_point;
                    $row['reorder_quantity'] = (int) $stock->reorder_quantity;
                }

                // الأسعار (active للنسخة، ثم fallback لسعر المنتج العام)
                $price = $variant->prices()->active()->first()
                    ?: ProductPrice::query()
                        ->where('product_id', $product->id)
                        ->whereNull('product_variant_id')
                        ->active()
                        ->first();

                if ($price) {
                    $row['retail_margin'] = (float) $price->retail_margin;
                    $row['retail_price'] = (int) $price->retail_price;
                    $row['wholesale_margin'] = (float) $price->wholesale_margin;
                    $row['wholesale_price'] = (int) $price->wholesale_price;
                    $row['agent_margin'] = (float) $price->agent_margin;
                    $row['agent_price'] = (int) $price->agent_price;
                }

                // خصائص النسخة: الفورم عندك يتكفّل بها عبر relationship
                return $row;
            })
            ->values()
            ->all();

        return $data;
    }

    /**
     * حفظ التعديلات باستخدام ProductService
     */
    protected function handleRecordUpdate(Model $record, array $data): Model
    {
        return app(ProductService::class)->createOrUpdate($record, $data);
    }
}
