<?php

namespace App\Filament\Resources\ProductPrices\Tables;

use App\Models\ProductVariant;
use Filament\Actions\ActionGroup;
use Filament\Actions\EditAction;
use Filament\Tables\Columns\TextColumn;
use Filament\Tables\Columns\ToggleColumn;
use Filament\Tables\Filters\SelectFilter;
use Filament\Tables\Table;
use Illuminate\Database\Eloquent\Builder;

class ProductPricesTable
{
    public static function configure(Table $table): Table
    {
        return $table
            ->columns([
                TextColumn::make('variant.full_name')
                    ->label('المنتج / المتغير')
                    ->placeholder('سعر عام للمنتج')
                    ->sortable()
                    ->searchable(),

                TextColumn::make('retail_price')
                    ->label('سعر المفرد')
                    ->formatStateUsing(fn ($state) => number_format((int) $state))
                    ->color('primary')
                    ->suffix(' د.ع'),

                TextColumn::make('wholesale_price')
                    ->label('سعر الجملة')
                    ->formatStateUsing(fn ($state) => number_format((int) $state))
                    ->color('success')
                    ->suffix(' د.ع'),

                TextColumn::make('agent_price')
                    ->label('سعر الوكيل')
                    ->formatStateUsing(fn ($state) => number_format((int) $state))
                    ->color('warning')
                    ->suffix(' د.ع'),

                ToggleColumn::make('status')
                    ->label('فعال'),
            ])
            ->filters([
                SelectFilter::make('product_id')
                    ->label('المنتج')
                    ->relationship('product', 'name')
                    ->searchable()
                    ->preload(),

                SelectFilter::make('product_variant_id')
                    ->label('المتغير')
                    ->relationship('variant', 'sku')
                    ->getOptionLabelFromRecordUsing(
                        fn (ProductVariant $v) => $v->full_name
                    )
                    ->searchable()
                    ->preload(),

                SelectFilter::make('price_scope')
                    ->label('نوع السعر')
                    ->options([
                        'base' => 'سعر عام',
                        'variant' => 'سعر متغير',
                    ])
                    ->query(fn (Builder $q, $v) => match ($v) {
                        'base' => $q->whereNull('product_variant_id'),
                        'variant' => $q->whereNotNull('product_variant_id'),
                        default => $q,
                    }),

                SelectFilter::make('status')
                    ->label('الحالة')
                    ->options([
                        1 => 'فعال',
                        0 => 'متوقف',
                    ]),
            ])
            ->recordActions([
                ActionGroup::make([
                    EditAction::make()
                        ->modalWidth('3xl'),
                ]),
            ])
            ->striped()
            ->persistFiltersInSession()
            ->persistSearchInSession();
    }
}
