<?php

namespace App\Filament\Resources\ProductPrices\Schemas;

use App\Models\ProductVariant;
use App\Models\StockMovement;
use Filament\Forms\Components\Select;
use Filament\Forms\Components\TextInput;
use Filament\Forms\Components\Toggle;
use Filament\Schemas\Components\Grid;
use Filament\Schemas\Components\Group;
use Filament\Schemas\Components\Section;
use Filament\Schemas\Components\Utilities\Get;
use Filament\Schemas\Components\Utilities\Set;
use Filament\Schemas\Schema;

class ProductPricesForm
{
    /* ================================
     * Helpers: أرقام صحيحة + تنسيق
     * ================================ */

    private static function toInt(mixed $value): int
    {
        if ($value === null || $value === '') {
            return 0;
        }
        if (is_int($value)) {
            return $value;
        }

        $value = preg_replace_callback(
            '/[٠-٩]/u',
            fn ($m) => (string) mb_strpos('٠١٢٣٤٥٦٧٨٩', $m[0]),
            (string) $value
        );

        $clean = preg_replace('/[^\d]/u', '', (string) $value);

        return (int) ($clean ?: 0);
    }

    private static function fmt(mixed $value): string
    {
        return number_format(self::toInt($value), 0, '', ',');
    }

    private static function toFloat(mixed $value): float
    {
        if ($value === null || $value === '') {
            return 0.0;
        }

        $v = preg_replace_callback(
            '/[٠-٩]/u',
            fn ($m) => (string) mb_strpos('٠١٢٣٤٥٦٧٨٩', $m[0]),
            (string) $value
        );

        $v = preg_replace('/[^\d.]/u', '', (string) $v);

        if (substr_count($v, '.') > 1) {
            $parts = explode('.', $v);
            $v = $parts[0].'.'.implode('', array_slice($parts, 1));
        }

        return (float) ($v ?: 0);
    }

    private static function moneyInput(string $name, string $label): TextInput
    {
        return TextInput::make($name)
            ->label($label)
            ->inputMode('numeric')
            ->step(1)
            ->default('0')
            ->live(debounce: 400)
            ->afterStateHydrated(function ($state, Set $set) use ($name) {
                $set($name, self::fmt($state));
            })
            ->afterStateUpdated(function ($state, Set $set) use ($name) {
                $formatted = self::fmt($state);
                if ((string) $state !== $formatted) {
                    $set($name, $formatted);
                }
            })
            // ✅ مهم: نخليها تنحفظ حتى لو disabled
            ->dehydrated(true)
            ->dehydrateStateUsing(fn ($state) => self::toInt($state))
            ->required()
            ->rule('regex:/^[\d,]+$/u')
            ->suffix(' د.ع');
    }

    private static function moneyDisplay(string $name, string $label): TextInput
    {
        return TextInput::make($name)
            ->label($label)
            ->disabled()
            // ✅ عرض فقط: لا ينحفظ
            ->dehydrated(false)
            ->default('0')
            ->live()
            ->afterStateHydrated(function ($state, Set $set) use ($name) {
                $set($name, self::fmt($state));
            })
            ->suffix(' د.ع');
    }

    private static function marginInput(string $name, string $label): TextInput
    {
        return TextInput::make($name)
            ->label($label)
            ->default('0')
            ->suffix('%')
            ->inputMode('decimal')
            ->live(debounce: 400)
            // ✅ نخزّنها float بالـ DB
            ->dehydrateStateUsing(fn ($state) => self::toFloat($state))
            ->rule('numeric')
            ->minValue(0);
    }

    /* ================================
     * Schema
     * ================================ */

    public static function configure(Schema $schema): Schema
    {
        return $schema->components([
            Section::make('تسعير المنتج')
                ->description('حدد المنتج والمتغير، سيتم جلب سعر الكلفة، ثم اختر نوع التسعير.')
                ->columnSpanFull()
                ->schema([
                    Grid::make(4)->schema([
                        Select::make('product_id')
                            ->label('المنتج')
                            ->relationship('product', 'name')
                            ->searchable()
                            ->preload()
                            ->required()
                            ->live()
                            ->afterStateUpdated(function (Set $set) {
                                $set('product_variant_id', null);
                                $set('unit_cost_preview', 0);

                                // UI فقط
                                $set('pricing_mode', 'margin');
                            })
                            ->columnSpan(1),

                        Select::make('product_variant_id')
                            ->label('المتغير')
                            ->options(fn (Get $get) => $get('product_id')
                                ? ProductVariant::query()
                                    ->where('product_id', $get('product_id'))
                                    ->orderBy('id', 'desc')
                                    ->get()
                                    ->pluck('full_name', 'id')
                                    ->toArray()
                                : [])
                            ->searchable()
                            ->preload()
                            ->required()
                            ->live()
                            ->disabled(fn (Get $get) => ! $get('product_id'))
                            ->afterStateUpdated(function ($state, Set $set, Get $get) {
                                $cost = self::getLatestCost((int) $state);
                                $set('unit_cost_preview', $cost);

                                // إذا على نسبة: احسب الأسعار فوراً
                                self::recalcPricesFromMargins($set, $get);
                            })
                            ->columnSpan(1),

                        self::moneyDisplay('unit_cost_preview', 'سعر الكلفة')
                            ->columnSpan(1),

                        Toggle::make('status')
                            ->label('السعر فعال')
                            ->default(true)
                            ->inline(false)
                            ->columnSpan(1),
                    ]),

                    // ✅ تهيئة عند فتح صفحة التعديل
                    Grid::make(12)->schema([
                        TextInput::make('pricing_mode')
                            ->default('margin')
                            ->dehydrated(false) // ✅ UI فقط
                            ->hidden()
                            ->afterStateHydrated(function (Set $set, Get $get) {
                                self::syncPricingMode($set, $get);

                                $variantId = (int) ($get('product_variant_id') ?? 0);
                                if ($variantId > 0) {
                                    $set('unit_cost_preview', self::getLatestCost($variantId));
                                }

                                self::recalcPricesFromMargins($set, $get);
                            })
                            ->columnSpanFull(),
                    ])->columnSpanFull(),

                    self::autoPricingSection(),
                ]),
        ]);
    }

    private static function autoPricingSection(): Section
    {
        return Section::make('التسعير')
            ->description('اختر: (بنسبة الربح) لحساب الأسعار تلقائياً أو (إدخال مباشر) لكتابة السعر. كل المبالغ أرقام صحيحة.')
            ->icon('heroicon-o-calculator')
            ->iconColor('info')
            ->collapsed(false)
            ->visible(fn (Get $get) => (bool) $get('product_variant_id'))
            ->schema([
                Grid::make(12)->schema([
                    Select::make('pricing_mode')
                        ->label('نوع التسعير')
                        ->options([
                            'margin' => 'بنسبة الربح',
                            'direct' => 'إدخال مباشر',
                        ])
                        ->default('margin')
                        ->native(false)
                        ->dehydrated(false) // ✅ UI فقط
                        ->live()
                        ->afterStateUpdated(function ($state, Set $set, Get $get) {
                            if ($state === 'margin') {
                                self::recalcPricesFromMargins($set, $get);
                            }
                        })
                        ->columnSpanFull(),

                    Group::make([
                        Grid::make(12)->schema([
                            self::marginInput('retail_margin', 'هامش المفرد %')
                                ->afterStateUpdated(fn ($state, Set $set, Get $get) => self::recalcPricesFromMargins($set, $get))
                                ->columnSpan(['default' => 12, 'md' => 4]),

                            self::marginInput('wholesale_margin', 'هامش الجملة %')
                                ->afterStateUpdated(fn ($state, Set $set, Get $get) => self::recalcPricesFromMargins($set, $get))
                                ->columnSpan(['default' => 12, 'md' => 4]),

                            self::marginInput('agent_margin', 'هامش الوكيل %')
                                ->afterStateUpdated(fn ($state, Set $set, Get $get) => self::recalcPricesFromMargins($set, $get))
                                ->columnSpan(['default' => 12, 'md' => 4]),
                        ]),
                    ])
                        ->visible(fn (Get $get) => ($get('pricing_mode') ?? 'margin') === 'margin')
                        ->columnSpanFull(),

                    Grid::make(12)->schema([
                        self::moneyInput('retail_price', 'سعر المفرد')
                            ->disabled(fn (Get $get) => ($get('pricing_mode') ?? 'margin') === 'margin')
                            ->afterStateUpdated(fn ($state, Set $set, Get $get) => self::syncMarginsFromPrice('retail', $state, $set, $get))
                            ->columnSpan(['default' => 12, 'md' => 4]),

                        self::moneyInput('wholesale_price', 'سعر الجملة')
                            ->disabled(fn (Get $get) => ($get('pricing_mode') ?? 'margin') === 'margin')
                            ->afterStateUpdated(fn ($state, Set $set, Get $get) => self::syncMarginsFromPrice('wholesale', $state, $set, $get))
                            ->columnSpan(['default' => 12, 'md' => 4]),

                        self::moneyInput('agent_price', 'سعر الوكيل')
                            ->disabled(fn (Get $get) => ($get('pricing_mode') ?? 'margin') === 'margin')
                            ->afterStateUpdated(fn ($state, Set $set, Get $get) => self::syncMarginsFromPrice('agent', $state, $set, $get))
                            ->columnSpan(['default' => 12, 'md' => 4]),
                    ])->columnSpanFull(),
                ]),
            ])
            ->columnSpanFull();
    }

    private static function syncMarginsFromPrice(string $type, mixed $state, Set $set, Get $get): void
    {
        if (($get('pricing_mode') ?? 'margin') !== 'direct') {
            return;
        }

        $cost = self::toInt($get('unit_cost_preview') ?? 0);
        $price = self::toInt($state);

        // تثبيت تنسيق السعر
        $set("{$type}_price", self::fmt($price));

        if ($cost <= 0 || $price <= 0) {
            $set("{$type}_margin", 0);

            return;
        }

        $margin = (($price - $cost) / $cost) * 100;
        $set("{$type}_margin", (float) number_format($margin, 2, '.', ''));
    }

    private static function getLatestCost(int $variantId): int
    {
        if ($variantId <= 0) {
            return 0;
        }

        $cost = (int) (StockMovement::query()
            ->where('product_variant_id', $variantId)
            ->orderByDesc('id')
            ->value('unit_cost') ?? 0);

        if ($cost > 0) {
            return $cost;
        }

        return (int) (ProductVariant::query()->whereKey($variantId)->value('cost') ?? 0);
    }

    private static function recalcPricesFromMargins(Set $set, Get $get): void
    {
        if (($get('pricing_mode') ?? 'margin') !== 'margin') {
            return;
        }

        $cost = self::toInt($get('unit_cost_preview') ?? 0);

        if ($cost <= 0) {
            $set('retail_price', self::fmt(0));
            $set('wholesale_price', self::fmt(0));
            $set('agent_price', self::fmt(0));

            return;
        }

        $retailMargin = self::toFloat($get('retail_margin') ?? 0);
        $wholesaleMargin = self::toFloat($get('wholesale_margin') ?? 0);
        $agentMargin = self::toFloat($get('agent_margin') ?? 0);

        $retailPrice = (int) round($cost + ($cost * $retailMargin / 100));
        $wholesalePrice = (int) round($cost + ($cost * $wholesaleMargin / 100));
        $agentPrice = (int) round($cost + ($cost * $agentMargin / 100));

        $set('retail_price', self::fmt($retailPrice));
        $set('wholesale_price', self::fmt($wholesalePrice));
        $set('agent_price', self::fmt($agentPrice));
    }

    private static function syncPricingMode(Set $set, Get $get): void
    {
        $retailM = self::toFloat($get('retail_margin') ?? 0);
        $wholesaleM = self::toFloat($get('wholesale_margin') ?? 0);
        $agentM = self::toFloat($get('agent_margin') ?? 0);

        $hasAnyMargin = ($retailM > 0) || ($wholesaleM > 0) || ($agentM > 0);

        $set('pricing_mode', $hasAnyMargin ? 'margin' : 'direct');
    }
}
